/* Copyright (c) 1997-1999 Miller Puckette.
 * For information on usage and redistribution, and for a DISCLAIMER OF ALL
 * WARRANTIES, see the file, "LICENSE.txt," in this distribution. */

/* g_7_guis.c written by Thomas Musil (c) IEM KUG Graz Austria 2000-2001 */
/* thanks to Miller Puckette, Guenther Geiger and Krzystof Czaja */


#ifdef ROCKBOX
#include "plugin.h"
#include "../../pdbox.h"
#include "m_pd.h"
#include "g_canvas.h"
#include "g_all_guis.h"
#else /* ROCKBOX */
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include "m_pd.h"
#include "g_canvas.h"
#include "t_tk.h"
#include "g_all_guis.h"
#include <math.h>

#ifdef MSW
#include <io.h>
#else
#include <unistd.h>
#endif
#endif /* ROCKBOX */


/* ------------ vsl gui-vertical  slider ----------------------- */

t_widgetbehavior vslider_widgetbehavior;
static t_class *vslider_class;

/* widget helper functions */

static void vslider_draw_update(t_vslider *x, t_glist *glist)
{
#ifdef ROCKBOX
    (void) x;
    (void) glist;
#else /* ROCKBOX */
    if (glist_isvisible(glist))
    {
	int r = text_ypix(&x->x_gui.x_obj, glist) + x->x_gui.x_h - (x->x_val + 50)/100;
        int xpos=text_xpix(&x->x_gui.x_obj, glist);

	sys_vgui(".x%x.c coords %xKNOB %d %d %d %d\n",
		 glist_getcanvas(glist), x, xpos+1, r,
		 xpos + x->x_gui.x_w, r);
    }
#endif /* ROCKBOX */
}

static void vslider_draw_new(t_vslider *x, t_glist *glist)
{
#ifdef ROCKBOX
    (void) x;
    (void) glist;
#else /* ROCKBOX */
    int xpos=text_xpix(&x->x_gui.x_obj, glist);
    int ypos=text_ypix(&x->x_gui.x_obj, glist);
    int r = ypos + x->x_gui.x_h - (x->x_val + 50)/100;
    t_canvas *canvas=glist_getcanvas(glist);

    sys_vgui(".x%x.c create rectangle %d %d %d %d -fill #%6.6x -tags %xBASE\n",
	     canvas, xpos, ypos-2,
	     xpos + x->x_gui.x_w, ypos + x->x_gui.x_h+3,
	     x->x_gui.x_bcol, x);
    sys_vgui(".x%x.c create line %d %d %d %d -width 3 -fill #%6.6x -tags %xKNOB\n",
	     canvas, xpos+1, r,
	     xpos + x->x_gui.x_w, r, x->x_gui.x_fcol, x);
    sys_vgui(".x%x.c create text %d %d -text {%s} -anchor w \
	     -font {%s %d bold} -fill #%6.6x -tags %xLABEL\n",
	     canvas, xpos+x->x_gui.x_ldx, ypos+x->x_gui.x_ldy,
	     strcmp(x->x_gui.x_lab->s_name, "empty")?x->x_gui.x_lab->s_name:"",
	     x->x_gui.x_font, x->x_gui.x_fontsize, x->x_gui.x_lcol, x);
    if(!x->x_gui.x_fsf.x_snd_able)
        sys_vgui(".x%x.c create rectangle %d %d %d %d -tags %xOUT%d\n",
	     canvas,
	     xpos, ypos + x->x_gui.x_h+2,
	     xpos+7, ypos + x->x_gui.x_h+3,
	     x, 0);
    if(!x->x_gui.x_fsf.x_rcv_able)
	sys_vgui(".x%x.c create rectangle %d %d %d %d -tags %xIN%d\n",
	     canvas,
	     xpos, ypos-2,
	     xpos+7, ypos-1,
	     x, 0);
#endif /* ROCKBOX */
}

static void vslider_draw_move(t_vslider *x, t_glist *glist)
{
#ifdef ROCKBOX
    (void) x;
    (void) glist;
#else /* ROCKBOX */
    int xpos=text_xpix(&x->x_gui.x_obj, glist);
    int ypos=text_ypix(&x->x_gui.x_obj, glist);
    int r = ypos + x->x_gui.x_h - (x->x_val + 50)/100;
    t_canvas *canvas=glist_getcanvas(glist);

    sys_vgui(".x%x.c coords %xBASE %d %d %d %d\n",
	     canvas, x,
	     xpos, ypos-2,
	     xpos + x->x_gui.x_w, ypos + x->x_gui.x_h+3);
    sys_vgui(".x%x.c coords %xKNOB %d %d %d %d\n",
	     canvas, x, xpos+1, r,
	     xpos + x->x_gui.x_w, r);
    sys_vgui(".x%x.c coords %xLABEL %d %d\n",
	     canvas, x, xpos+x->x_gui.x_ldx, ypos+x->x_gui.x_ldy);
    if(!x->x_gui.x_fsf.x_snd_able)
        sys_vgui(".x%x.c coords %xOUT%d %d %d %d %d\n",
	     canvas, x, 0,
	     xpos, ypos + x->x_gui.x_h+2,
	     xpos+7, ypos + x->x_gui.x_h+3);
    if(!x->x_gui.x_fsf.x_rcv_able)
	sys_vgui(".x%x.c coords %xIN%d %d %d %d %d\n",
	     canvas, x, 0,
	     xpos, ypos-2,
	     xpos+7, ypos-1);
#endif /* ROCKBOX */
}

static void vslider_draw_erase(t_vslider* x,t_glist* glist)
{
#ifdef ROCKBOX
    (void) x;
    (void) glist;
#else /* ROCKBOX */
    t_canvas *canvas=glist_getcanvas(glist);

    sys_vgui(".x%x.c delete %xBASE\n", canvas, x);
    sys_vgui(".x%x.c delete %xKNOB\n", canvas, x);
    sys_vgui(".x%x.c delete %xLABEL\n", canvas, x);
    if(!x->x_gui.x_fsf.x_snd_able)
        sys_vgui(".x%x.c delete %xOUT%d\n", canvas, x, 0);
    if(!x->x_gui.x_fsf.x_rcv_able)
	sys_vgui(".x%x.c delete %xIN%d\n", canvas, x, 0);
#endif /* ROCKBOX */
}

static void vslider_draw_config(t_vslider* x,t_glist* glist)
{
#ifdef ROCKBOX
    (void) x;
    (void) glist;
#else /* ROCKBOX */
    t_canvas *canvas=glist_getcanvas(glist);

    sys_vgui(".x%x.c itemconfigure %xLABEL -font {%s %d bold} -fill #%6.6x -text {%s} \n",
	     canvas, x, x->x_gui.x_font, x->x_gui.x_fontsize,
	     x->x_gui.x_fsf.x_selected?IEM_GUI_COLOR_SELECTED:x->x_gui.x_lcol,
	     strcmp(x->x_gui.x_lab->s_name, "empty")?x->x_gui.x_lab->s_name:"");
    sys_vgui(".x%x.c itemconfigure %xKNOB -fill #%6.6x\n", canvas,
	     x, x->x_gui.x_fcol);
    sys_vgui(".x%x.c itemconfigure %xBASE -fill #%6.6x\n", canvas,
	     x, x->x_gui.x_bcol);
#endif /* ROCKBOX */
}

static void vslider_draw_io(t_vslider* x,t_glist* glist, int old_snd_rcv_flags)
{
#ifdef ROCKBOX
    (void) x;
    (void) glist;
    (void) old_snd_rcv_flags;
#else /* ROCKBOX */
    int xpos=text_xpix(&x->x_gui.x_obj, glist);
    int ypos=text_ypix(&x->x_gui.x_obj, glist);
    t_canvas *canvas=glist_getcanvas(glist);

    if((old_snd_rcv_flags & IEM_GUI_OLD_SND_FLAG) && !x->x_gui.x_fsf.x_snd_able)
        sys_vgui(".x%x.c create rectangle %d %d %d %d -tags %xOUT%d\n",
	     canvas,
	     xpos, ypos + x->x_gui.x_h+2,
	     xpos+7, ypos + x->x_gui.x_h+3,
	     x, 0);
    if(!(old_snd_rcv_flags & IEM_GUI_OLD_SND_FLAG) && x->x_gui.x_fsf.x_snd_able)
        sys_vgui(".x%x.c delete %xOUT%d\n", canvas, x, 0);
    if((old_snd_rcv_flags & IEM_GUI_OLD_RCV_FLAG) && !x->x_gui.x_fsf.x_rcv_able)
        sys_vgui(".x%x.c create rectangle %d %d %d %d -tags %xIN%d\n",
	     canvas,
	     xpos, ypos-2,
	     xpos+7, ypos-1,
	     x, 0);
    if(!(old_snd_rcv_flags & IEM_GUI_OLD_RCV_FLAG) && x->x_gui.x_fsf.x_rcv_able)
        sys_vgui(".x%x.c delete %xIN%d\n", canvas, x, 0);
#endif /* ROCKBOX */
}

static void vslider_draw_select(t_vslider *x, t_glist *glist)
{
#ifdef ROCKBOX
    (void) x;
    (void) glist;
#else /* ROCKBOX */
    t_canvas *canvas=glist_getcanvas(glist);

    if(x->x_gui.x_fsf.x_selected)
    {
	sys_vgui(".x%x.c itemconfigure %xBASE -outline #%6.6x\n", canvas, x, IEM_GUI_COLOR_SELECTED);
	sys_vgui(".x%x.c itemconfigure %xLABEL -fill #%6.6x\n", canvas, x, IEM_GUI_COLOR_SELECTED);
    }
    else
    {
	sys_vgui(".x%x.c itemconfigure %xBASE -outline #%6.6x\n", canvas, x, IEM_GUI_COLOR_NORMAL);
	sys_vgui(".x%x.c itemconfigure %xLABEL -fill #%6.6x\n", canvas, x, x->x_gui.x_lcol);
    }
#endif /* ROCKBOX */
}

void vslider_draw(t_vslider *x, t_glist *glist, int mode)
{
    if(mode == IEM_GUI_DRAW_MODE_UPDATE)
	vslider_draw_update(x, glist);
    else if(mode == IEM_GUI_DRAW_MODE_MOVE)
	vslider_draw_move(x, glist);
    else if(mode == IEM_GUI_DRAW_MODE_NEW)
	vslider_draw_new(x, glist);
    else if(mode == IEM_GUI_DRAW_MODE_SELECT)
	vslider_draw_select(x, glist);
    else if(mode == IEM_GUI_DRAW_MODE_ERASE)
	vslider_draw_erase(x, glist);
    else if(mode == IEM_GUI_DRAW_MODE_CONFIG)
	vslider_draw_config(x, glist);
    else if(mode >= IEM_GUI_DRAW_MODE_IO)
	vslider_draw_io(x, glist, mode - IEM_GUI_DRAW_MODE_IO);
}

/* ------------------------ vsl widgetbehaviour----------------------------- */


static void vslider_getrect(t_gobj *z, t_glist *glist,
			    int *xp1, int *yp1, int *xp2, int *yp2)
{
    t_vslider* x = (t_vslider*)z;

    *xp1 = text_xpix(&x->x_gui.x_obj, glist);
    *yp1 = text_ypix(&x->x_gui.x_obj, glist) - 2;
    *xp2 = *xp1 + x->x_gui.x_w;
    *yp2 = *yp1 + x->x_gui.x_h + 5;
}

static void vslider_save(t_gobj *z, t_binbuf *b)
{
    t_vslider *x = (t_vslider *)z;
    int bflcol[3];
    t_symbol *srl[3];

    iemgui_save(&x->x_gui, srl, bflcol);
    binbuf_addv(b, "ssiisiiffiisssiiiiiiiii", gensym("#X"),gensym("obj"),
		(t_int)x->x_gui.x_obj.te_xpix, (t_int)x->x_gui.x_obj.te_ypix,
		gensym("vsl"), x->x_gui.x_w, x->x_gui.x_h,
		(float)x->x_min, (float)x->x_max,
		x->x_lin0_log1, iem_symargstoint(&x->x_gui.x_isa),
		srl[0], srl[1], srl[2],
		x->x_gui.x_ldx, x->x_gui.x_ldy,
		iem_fstyletoint(&x->x_gui.x_fsf), x->x_gui.x_fontsize,
		bflcol[0], bflcol[1], bflcol[2],
		x->x_val, x->x_steady);
    binbuf_addv(b, ";");
}

void vslider_check_height(t_vslider *x, int h)
{
    if(h < IEM_SL_MINSIZE)
	h = IEM_SL_MINSIZE;
    x->x_gui.x_h = h;
    if(x->x_val > (x->x_gui.x_h*100 - 100))
    {
	x->x_pos = x->x_gui.x_h*100 - 100;
	x->x_val = x->x_pos;
    }
    if(x->x_lin0_log1)
	x->x_k = log(x->x_max/x->x_min)/(double)(x->x_gui.x_h - 1);
    else
	x->x_k = (x->x_max - x->x_min)/(double)(x->x_gui.x_h - 1);
}

void vslider_check_minmax(t_vslider *x, double min, double max)
{
    if(x->x_lin0_log1)
    {
	if((min == 0.0)&&(max == 0.0))
	    max = 1.0;
	if(max > 0.0)
	{
	    if(min <= 0.0)
		min = 0.01*max;
	}
	else
	{
	    if(min > 0.0)
		max = 0.01*min;
	}
    }
    x->x_min = min;
    x->x_max = max;
    if(x->x_min > x->x_max)                /* bugfix */
	x->x_gui.x_isa.x_reverse = 1;
    else
        x->x_gui.x_isa.x_reverse = 0;
    if(x->x_lin0_log1)
	x->x_k = log(x->x_max/x->x_min)/(double)(x->x_gui.x_h - 1);
    else
	x->x_k = (x->x_max - x->x_min)/(double)(x->x_gui.x_h - 1);
}

static void vslider_properties(t_gobj *z, t_glist *owner)
{
#ifdef ROCKBOX
    (void) z;
    (void) owner;
#else /* ROCKBOX */
    t_vslider *x = (t_vslider *)z;
    char buf[800];
    t_symbol *srl[3];

    iemgui_properties(&x->x_gui, srl);

    sprintf(buf, "pdtk_iemgui_dialog %%s VSLIDER \
	    --------dimensions(pix)(pix):-------- %d %d width: %d %d height: \
	    -----------output-range:----------- %g bottom: %g top: %d \
	    %d lin log %d %d empty %d \
	    %s %s \
	    %s %d %d \
	    %d %d \
	    %d %d %d\n",
	    x->x_gui.x_w, IEM_GUI_MINSIZE, x->x_gui.x_h, IEM_SL_MINSIZE,
	    x->x_min, x->x_max, 0,/*no_schedule*/
	    x->x_lin0_log1, x->x_gui.x_isa.x_loadinit, x->x_steady, -1,/*no multi, but iem-characteristic*/
	    srl[0]->s_name, srl[1]->s_name,
	    srl[2]->s_name, x->x_gui.x_ldx, x->x_gui.x_ldy,
	    x->x_gui.x_fsf.x_font_style, x->x_gui.x_fontsize,
	    0xffffff & x->x_gui.x_bcol, 0xffffff & x->x_gui.x_fcol, 0xffffff & x->x_gui.x_lcol);
    gfxstub_new(&x->x_gui.x_obj.ob_pd, x, buf);
#endif
}

static void vslider_bang(t_vslider *x)
{
    double out;

    if(x->x_lin0_log1)
	out = x->x_min*exp(x->x_k*(double)(x->x_val)*0.01);
    else
	out = (double)(x->x_val)*0.01*x->x_k + x->x_min;
    if((out < 1.0e-10)&&(out > -1.0e-10))
	out = 0.0;

    outlet_float(x->x_gui.x_obj.ob_outlet, out);
    if(x->x_gui.x_fsf.x_snd_able && x->x_gui.x_snd->s_thing)
	pd_float(x->x_gui.x_snd->s_thing, out);
}

static void vslider_dialog(t_vslider *x, t_symbol *s, int argc, t_atom *argv)
{
    t_symbol *srl[3];
    int w = (int)atom_getintarg(0, argc, argv);
    int h = (int)atom_getintarg(1, argc, argv);
    double min = (double)atom_getfloatarg(2, argc, argv);
    double max = (double)atom_getfloatarg(3, argc, argv);
    int lilo = (int)atom_getintarg(4, argc, argv);
    int steady = (int)atom_getintarg(17, argc, argv);
    int sr_flags;

#ifdef ROCKBOX
    (void) s;
#endif

    if(lilo != 0) lilo = 1;
    x->x_lin0_log1 = lilo;
    if(steady)
	x->x_steady = 1;
    else
	x->x_steady = 0;
    sr_flags = iemgui_dialog(&x->x_gui, srl, argc, argv);
    x->x_gui.x_w = iemgui_clip_size(w);
    vslider_check_height(x, h);
    vslider_check_minmax(x, min, max);
    (*x->x_gui.x_draw)(x, x->x_gui.x_glist, IEM_GUI_DRAW_MODE_CONFIG);
    (*x->x_gui.x_draw)(x, x->x_gui.x_glist, IEM_GUI_DRAW_MODE_IO + sr_flags);
    (*x->x_gui.x_draw)(x, x->x_gui.x_glist, IEM_GUI_DRAW_MODE_MOVE);
    canvas_fixlinesfor(glist_getcanvas(x->x_gui.x_glist), (t_text*)x);
}

static void vslider_motion(t_vslider *x, t_floatarg dx, t_floatarg dy)
{
    int old = x->x_val;

#ifdef ROCKBOX
    (void) dx;
#endif

    if(x->x_gui.x_fsf.x_finemoved)
	x->x_pos -= (int)dy;
    else
	x->x_pos -= 100*(int)dy;
    x->x_val = x->x_pos;
    if(x->x_val > (100*x->x_gui.x_h - 100))
    {
	x->x_val = 100*x->x_gui.x_h - 100;
	x->x_pos += 50;
	x->x_pos -= x->x_pos%100;
    }
    if(x->x_val < 0)
    {
	x->x_val = 0;
	x->x_pos -= 50;
	x->x_pos -= x->x_pos%100;
    }
    if(old != x->x_val)
    {
	(*x->x_gui.x_draw)(x, x->x_gui.x_glist, IEM_GUI_DRAW_MODE_UPDATE);
	vslider_bang(x);
    }
}

static void vslider_click(t_vslider *x, t_floatarg xpos, t_floatarg ypos,
			  t_floatarg shift, t_floatarg ctrl, t_floatarg alt)
{
#ifdef ROCKBOX
    (void) shift;
    (void) ctrl;
    (void) alt;
#endif
    if(!x->x_steady)
	x->x_val = (int)(100.0 * (x->x_gui.x_h + text_ypix(&x->x_gui.x_obj, x->x_gui.x_glist) - ypos));
    if(x->x_val > (100*x->x_gui.x_h - 100))
	x->x_val = 100*x->x_gui.x_h - 100;
    if(x->x_val < 0)
	x->x_val = 0;
    x->x_pos = x->x_val;
    (*x->x_gui.x_draw)(x, x->x_gui.x_glist, IEM_GUI_DRAW_MODE_UPDATE);
    vslider_bang(x);
    glist_grab(x->x_gui.x_glist, &x->x_gui.x_obj.te_g,
    	(t_glistmotionfn)vslider_motion, 0, xpos, ypos);
}

static int vslider_newclick(t_gobj *z, struct _glist *glist,
			    int xpix, int ypix, int shift, int alt, int dbl, int doit)
{
    t_vslider* x = (t_vslider *)z;

#ifdef ROCKBOX
    (void) glist;
    (void) dbl;
#endif

    if(doit)
    {
	vslider_click( x, (t_floatarg)xpix, (t_floatarg)ypix, (t_floatarg)shift,
		       0, (t_floatarg)alt);
	if(shift)
	    x->x_gui.x_fsf.x_finemoved = 1;
	else
	    x->x_gui.x_fsf.x_finemoved = 0;
    }
    return (1);
}

static void vslider_set(t_vslider *x, t_floatarg f)
{
    double g;

    if(x->x_gui.x_isa.x_reverse)    /* bugfix */
    {
	if(f > x->x_min)
	    f = x->x_min;
	if(f < x->x_max)
	    f = x->x_max;
    }
    else
    {
	if(f > x->x_max)
	    f = x->x_max;
	if(f < x->x_min)
	    f = x->x_min;
    }
    if(x->x_lin0_log1)
	g = log(f/x->x_min)/x->x_k;
    else
	g = (f - x->x_min) / x->x_k;
    x->x_val = (int)(100.0*g + 0.49999);
    x->x_pos = x->x_val;
    (*x->x_gui.x_draw)(x, x->x_gui.x_glist, IEM_GUI_DRAW_MODE_UPDATE);
}

static void vslider_float(t_vslider *x, t_floatarg f)
{
    vslider_set(x, f);
    if(x->x_gui.x_fsf.x_put_in2out)
	vslider_bang(x);
}

static void vslider_size(t_vslider *x, t_symbol *s, int ac, t_atom *av)
{
#ifdef ROCKBOX
    (void) s;
#endif
    x->x_gui.x_w = iemgui_clip_size((int)atom_getintarg(0, ac, av));
    if(ac > 1)
	vslider_check_height(x, (int)atom_getintarg(1, ac, av));
    iemgui_size((void *)x, &x->x_gui);
}

static void vslider_delta(t_vslider *x, t_symbol *s, int ac, t_atom *av)
{iemgui_delta((void *)x, &x->x_gui, s, ac, av);}

static void vslider_pos(t_vslider *x, t_symbol *s, int ac, t_atom *av)
{iemgui_pos((void *)x, &x->x_gui, s, ac, av);}

static void vslider_range(t_vslider *x, t_symbol *s, int ac, t_atom *av)
{
#ifdef ROCKBOX
    (void) s;
#endif
    vslider_check_minmax(x, (double)atom_getfloatarg(0, ac, av),
			 (double)atom_getfloatarg(1, ac, av));
}

static void vslider_color(t_vslider *x, t_symbol *s, int ac, t_atom *av)
{iemgui_color((void *)x, &x->x_gui, s, ac, av);}

static void vslider_send(t_vslider *x, t_symbol *s)
{iemgui_send(x, &x->x_gui, s);}

static void vslider_receive(t_vslider *x, t_symbol *s)
{iemgui_receive(x, &x->x_gui, s);}

static void vslider_label(t_vslider *x, t_symbol *s)
{iemgui_label((void *)x, &x->x_gui, s);}

static void vslider_label_pos(t_vslider *x, t_symbol *s, int ac, t_atom *av)
{iemgui_label_pos((void *)x, &x->x_gui, s, ac, av);}

static void vslider_label_font(t_vslider *x, t_symbol *s, int ac, t_atom *av)
{iemgui_label_font((void *)x, &x->x_gui, s, ac, av);}

static void vslider_log(t_vslider *x)
{
    x->x_lin0_log1 = 1;
    vslider_check_minmax(x, x->x_min, x->x_max);
}

static void vslider_lin(t_vslider *x)
{
    x->x_lin0_log1 = 0;
    x->x_k = (x->x_max - x->x_min)/(double)(x->x_gui.x_h - 1);
}

static void vslider_init(t_vslider *x, t_floatarg f)
{
    x->x_gui.x_isa.x_loadinit = (f==0.0)?0:1;
}

static void vslider_steady(t_vslider *x, t_floatarg f)
{
    x->x_steady = (f==0.0)?0:1;
}

static void vslider_loadbang(t_vslider *x)
{
    if(!sys_noloadbang && x->x_gui.x_isa.x_loadinit)
    {
	(*x->x_gui.x_draw)(x, x->x_gui.x_glist, IEM_GUI_DRAW_MODE_UPDATE);
	vslider_bang(x);
    }
}

static void *vslider_new(t_symbol *s, int argc, t_atom *argv)
{
    t_vslider *x = (t_vslider *)pd_new(vslider_class);
    int bflcol[]={-262144, -1, -1};
    int w=IEM_GUI_DEFAULTSIZE, h=IEM_SL_DEFAULTSIZE;
#ifdef ROCKBOX
    int lilo=0, ldx=0, ldy=-8;
#else
    int lilo=0, f=0, ldx=0, ldy=-8;
#endif
    int fs=8, v=0, steady=1;
    double min=0.0, max=(double)(IEM_SL_DEFAULTSIZE-1);
#ifndef ROCKBOX
    char str[144];
#endif

#ifdef ROCKBOX
    (void) s;
#endif

    iem_inttosymargs(&x->x_gui.x_isa, 0);
    iem_inttofstyle(&x->x_gui.x_fsf, 0);

    if(((argc == 17)||(argc == 18))&&IS_A_FLOAT(argv,0)&&IS_A_FLOAT(argv,1)
       &&IS_A_FLOAT(argv,2)&&IS_A_FLOAT(argv,3)
       &&IS_A_FLOAT(argv,4)&&IS_A_FLOAT(argv,5)
       &&(IS_A_SYMBOL(argv,6)||IS_A_FLOAT(argv,6))
       &&(IS_A_SYMBOL(argv,7)||IS_A_FLOAT(argv,7))
       &&(IS_A_SYMBOL(argv,8)||IS_A_FLOAT(argv,8))
       &&IS_A_FLOAT(argv,9)&&IS_A_FLOAT(argv,10)
       &&IS_A_FLOAT(argv,11)&&IS_A_FLOAT(argv,12)&&IS_A_FLOAT(argv,13)
       &&IS_A_FLOAT(argv,14)&&IS_A_FLOAT(argv,15)&&IS_A_FLOAT(argv,16))
    {
	w = (int)atom_getintarg(0, argc, argv);
	h = (int)atom_getintarg(1, argc, argv);
	min = (double)atom_getfloatarg(2, argc, argv);
	max = (double)atom_getfloatarg(3, argc, argv);
	lilo = (int)atom_getintarg(4, argc, argv);
	iem_inttosymargs(&x->x_gui.x_isa, atom_getintarg(5, argc, argv));
	iemgui_new_getnames(&x->x_gui, 6, argv);
	ldx = (int)atom_getintarg(9, argc, argv);
	ldy = (int)atom_getintarg(10, argc, argv);
	iem_inttofstyle(&x->x_gui.x_fsf, atom_getintarg(11, argc, argv));
	fs = (int)atom_getintarg(12, argc, argv);
	bflcol[0] = (int)atom_getintarg(13, argc, argv);
	bflcol[1] = (int)atom_getintarg(14, argc, argv);
	bflcol[2] = (int)atom_getintarg(15, argc, argv);
	v = (int)atom_getintarg(16, argc, argv);
    }
    else iemgui_new_getnames(&x->x_gui, 6, 0);
    if((argc == 18)&&IS_A_FLOAT(argv,17))
	steady = (int)atom_getintarg(17, argc, argv);
    x->x_gui.x_draw = (t_iemfunptr)vslider_draw;
    x->x_gui.x_fsf.x_snd_able = 1;
    x->x_gui.x_fsf.x_rcv_able = 1;
    x->x_gui.x_glist = (t_glist *)canvas_getcurrent();
    if(x->x_gui.x_isa.x_loadinit)
	x->x_val = v;
    else
	x->x_val = 0;
    x->x_pos = x->x_val;
    if(lilo != 0) lilo = 1;
    x->x_lin0_log1 = lilo;
    if(steady != 0) steady = 1;
    x->x_steady = steady;
    if(!strcmp(x->x_gui.x_snd->s_name, "empty")) x->x_gui.x_fsf.x_snd_able = 0;
    if(!strcmp(x->x_gui.x_rcv->s_name, "empty")) x->x_gui.x_fsf.x_rcv_able = 0;
    if(x->x_gui.x_fsf.x_font_style == 1) strcpy(x->x_gui.x_font, "helvetica");
    else if(x->x_gui.x_fsf.x_font_style == 2) strcpy(x->x_gui.x_font, "times");
    else { x->x_gui.x_fsf.x_font_style = 0;
	strcpy(x->x_gui.x_font, "courier"); }
    if(x->x_gui.x_fsf.x_rcv_able) pd_bind(&x->x_gui.x_obj.ob_pd, x->x_gui.x_rcv);
    x->x_gui.x_ldx = ldx;
    x->x_gui.x_ldy = ldy;
    if(fs < 4)
	fs = 4;
    x->x_gui.x_fontsize = fs;
    x->x_gui.x_w = iemgui_clip_size(w);
    vslider_check_height(x, h);
    vslider_check_minmax(x, min, max);
    iemgui_all_colfromload(&x->x_gui, bflcol);
    iemgui_verify_snd_ne_rcv(&x->x_gui);
    outlet_new(&x->x_gui.x_obj, &s_float);
    return (x);
}

static void vslider_free(t_vslider *x)
{
    if(x->x_gui.x_fsf.x_rcv_able)
	pd_unbind(&x->x_gui.x_obj.ob_pd, x->x_gui.x_rcv);
#ifndef ROCKBOX
    gfxstub_deleteforkey(x);
#endif
}

void g_vslider_setup(void)
{
    vslider_class = class_new(gensym("vsl"), (t_newmethod)vslider_new,
			      (t_method)vslider_free, sizeof(t_vslider), 0, A_GIMME, 0);
    class_addcreator((t_newmethod)vslider_new, gensym("vslider"), A_GIMME, 0);
    class_addbang(vslider_class,vslider_bang);
    class_addfloat(vslider_class,vslider_float);
    class_addmethod(vslider_class, (t_method)vslider_click, gensym("click"),
		    A_FLOAT, A_FLOAT, A_FLOAT, A_FLOAT, A_FLOAT, 0);
    class_addmethod(vslider_class, (t_method)vslider_motion, gensym("motion"),
		    A_FLOAT, A_FLOAT, 0);
    class_addmethod(vslider_class, (t_method)vslider_dialog, gensym("dialog"),
		    A_GIMME, 0);
    class_addmethod(vslider_class, (t_method)vslider_loadbang, gensym("loadbang"), 0);
    class_addmethod(vslider_class, (t_method)vslider_set, gensym("set"), A_FLOAT, 0);
    class_addmethod(vslider_class, (t_method)vslider_size, gensym("size"), A_GIMME, 0);
    class_addmethod(vslider_class, (t_method)vslider_delta, gensym("delta"), A_GIMME, 0);
    class_addmethod(vslider_class, (t_method)vslider_pos, gensym("pos"), A_GIMME, 0);
    class_addmethod(vslider_class, (t_method)vslider_range, gensym("range"), A_GIMME, 0);
    class_addmethod(vslider_class, (t_method)vslider_color, gensym("color"), A_GIMME, 0);
    class_addmethod(vslider_class, (t_method)vslider_send, gensym("send"), A_DEFSYM, 0);
    class_addmethod(vslider_class, (t_method)vslider_receive, gensym("receive"), A_DEFSYM, 0);
    class_addmethod(vslider_class, (t_method)vslider_label, gensym("label"), A_DEFSYM, 0);
    class_addmethod(vslider_class, (t_method)vslider_label_pos, gensym("label_pos"), A_GIMME, 0);
    class_addmethod(vslider_class, (t_method)vslider_label_font, gensym("label_font"), A_GIMME, 0);
    class_addmethod(vslider_class, (t_method)vslider_log, gensym("log"), 0);
    class_addmethod(vslider_class, (t_method)vslider_lin, gensym("lin"), 0);
    class_addmethod(vslider_class, (t_method)vslider_init, gensym("init"), A_FLOAT, 0);
    class_addmethod(vslider_class, (t_method)vslider_steady, gensym("steady"), A_FLOAT, 0);
    vslider_widgetbehavior.w_getrectfn =    vslider_getrect;
    vslider_widgetbehavior.w_displacefn =   iemgui_displace;
    vslider_widgetbehavior.w_selectfn =     iemgui_select;
    vslider_widgetbehavior.w_activatefn =   NULL;
    vslider_widgetbehavior.w_deletefn =     iemgui_delete;
    vslider_widgetbehavior.w_visfn =        iemgui_vis;
    vslider_widgetbehavior.w_clickfn =      vslider_newclick;
    class_setwidget(vslider_class, &vslider_widgetbehavior);
    class_sethelpsymbol(vslider_class, gensym("vslider"));
    class_setsavefn(vslider_class, vslider_save);
    class_setpropertiesfn(vslider_class, vslider_properties);
}

