"""The main class for the application"""
import gtk
import hildon
import os.path
import location
from tempfile import mkdtemp
import time
import osso
import gobject

import import_data
import db

class  roadrunnerApp(hildon.Program):

    def __init__(self):
        """Contains main methods for the app class"""
        
        hildon.Program.__init__(self)
        
        self.previousfixtime = float("nan")
        self.previousradarid = -1
        self.previousradardistance = -1
        self.alert = False
        
        self.mydb = db.radar_db()
        gtk.set_application_name("Roadrunner")
        
        self.control = location.GPSDControl.get_default()
        self.device = location.GPSDevice()
        self.device.reset_last_known()
        self.control.set_properties(
                                preferred_method=location.METHOD_USER_SELECTED,
                                preferred_interval=location.INTERVAL_DEFAULT)

        self.control.connect("error-verbose", self.gps_on_error, None)
        self.device.connect("changed", self.gps_on_changed, self.control)
        self.control.start()
        
        self.osso_c = osso.Context("roadrunner", "0.1.0", False)
        self.osso_r = osso.Rpc(self.osso_c)

        self.odevice = osso.DeviceState(self.osso_c)
        
        # Only in Scratchbox
        #gobject.timeout_add(3000, self.test_alert)        

        self.window = hildon.Window()
        self.add_window(self.window)

        menu_label = gtk.Label("Parameters")
        self.menu = self.create_menu(menu_label)
        self.window.set_app_menu(self.menu)
        
        self.build_interface()

        self.window.show_all()
        self.window.connect("delete_event", self.quit, self.control)

        self.build_window_alert()
        gtk.main()        
        

    def imp_asc(self,  window,  widget):
        """Download and import speed cameras database"""
                
        tmpdir = mkdtemp(suffix='roadrunner')
        print("Temp folder : %s" % tmpdir)
        tmpzipfile = tmpdir + "/" + "radars.zip"
        hildon.hildon_gtk_window_set_progress_indicator(self.window, 1)
        (result,  message) = import_data.download_file(tmpzipfile)
        print result,  message
        if result:            
            self.warning_message(message)
            return
        (result,  message) = import_data.extract_zip(tmpzipfile)
        print result,  message
        if result:            
            self.warning_message(message)
            return
        
        hildon.hildon_gtk_window_set_progress_indicator(self.window, 0)
        os.remove(tmpzipfile)
        
        self.menu.set_sensitive(False)
         
        #tmpdir = "/var/tmp/tmpZ1odpWroadrunner/"
        ascfiles = os.listdir(tmpdir)
        
        for ascfile in ascfiles:
            
            result = self.confirm_file_import(ascfile)
            print result
            if result:
                hildon.hildon_gtk_window_set_progress_indicator(self.window, 1)
                import_data.import_ascfile(tmpdir + "/" + ascfile,  self.pb)
                hildon.hildon_gtk_window_set_progress_indicator(self.window, 0)
            os.remove(tmpdir + "/" + ascfile)            
            
        os.rmdir(tmpdir)
        self.menu.set_sensitive(True)
        self.pb.set_fraction(0)
        self.pb.set_text("")
        print "Import done !!!"""
            
    
    def confirm_file_import(self,  file):
        """Confirmation for each type of speed camera file"""
        
        boite = gtk.Dialog("Confirmation", self.window,
            gtk.DIALOG_MODAL,
            (gtk.STOCK_OK, gtk.RESPONSE_YES,
            gtk.STOCK_CANCEL, gtk.RESPONSE_NO))
            
        if file=="RF.asc":
            message = "Import fixed speed cameras ?"
        elif file=="RFR.asc":
            message = "Import red light cameras ?"
        elif file=="RM.asc":
            message = "Import mobile speed cameras ?"
        elif file=="ZD.asc":
            message = "Import dangerous zones ?"
        else:
            message = "Import %s ?" % file
            
        boite.vbox.pack_start(gtk.Label(message), True, False, 0)
        boite.vbox.show_all()

        reponse = boite.run()
        if reponse == gtk.RESPONSE_YES:
            result = True
        else:
            result = False
        boite.destroy()
        return result


    def imp_file(self,  widget,  data):
        """import speed cameras database from local file"""
        dlg =gobject.new(hildon.FileChooserDialog, action=gtk.FILE_CHOOSER_ACTION_OPEN)
        dlg.set_default_response(gtk.RESPONSE_OK)
        response = dlg.run()
        if response == gtk.RESPONSE_OK:
            hildon.hildon_gtk_window_set_progress_indicator(self.window, 1)
            self.menu.set_sensitive(False)
            localfile = dlg.get_filename()
            print localfile
            dlg.destroy()
            try:
                if localfile.endswith("gpx"):
                   import_data.import_gpxfile(localfile, self.pb)
                elif localfile.endswith("csv"):
                   import_data.import_csvfile(localfile, self.pb)
                else:
                    self.warning_message("Unknown format\nCheck file extension")
            except Exception, err:
                self.warning_message("Error when importing file")
                print type(err)
                print err.args
            hildon.hildon_gtk_window_set_progress_indicator(self.window, 0)
        
        dlg.destroy()
        self.menu.set_sensitive(True)
        self.menu.set_sensitive(True)
        self.pb.set_fraction(0)
        self.pb.set_text("")
        print "Import done !!!"""
        
        
    def create_menu(self,  label):
        """Build the menu"""
        
        menu = hildon.AppMenu()
        button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
        button.set_label("Download speed cameras...")
        button.connect("clicked", self.imp_asc, None)
        menu.append(button)
        button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
        button.set_label("Import local file...")
        button.connect("clicked", self.imp_file, None)
        menu.append(button)
        button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
        button.set_label("Clear database...")
        button.connect("clicked", self.clear_database, None)
        menu.append(button)
        menu.show_all()
        return menu
        
        
    def clear_database(self,  widget,  data):
        """Clear all records in database"""
        self.mydb.clear_radars()
        

    def quit(self,  control,  window,  widget):
        """Quit application"""
        self.control.stop()
        gtk.main_quit()
        
        
    def gps_on_error(self,  user_data):
        """Display message when location error occurs"""
        print "GPS error..."
        

    def gps_on_changed(self,  device,  user_data):
        """callback each time the location changes"""
        
        print"on_change"
        if device.fix:
            print "sat in view: %d, in use: %d" %(device.satellites_in_view, device.satellites_in_use) 
            print "time : %s" % time.strftime(
                                            "%d/%m/%Y %H:%M:%S",
                                            time.localtime(device.fix[2]))
            if (device.fix[2]==self.previousfixtime):
                print "No update"
                print "time : %s" % time.strftime(
                                            "%d/%m/%Y %H:%M:%S",
                                            time.localtime(device.fix[2]))
                print "latitude, longitude : %f, %f" % device.fix[4:6]
                print "speed : %f" % device.fix[11]
                
            elif (str(device.fix[4]) != "nan") & (str(device.fix[5]) != "nan"):
                print "Gps fix time changed"
                print "time : %s" % time.strftime(
                                            "%d/%m/%Y %H:%M:%S",
                                            time.localtime(device.fix[2]))
                print "latitude, longitude : %f, %f" % device.fix[4:6]
                print "speed : %f" % device.fix[11]
                
                self.lastupdate_lbl.set_text("%s" % time.ctime(device.fix[2]))
                self.latitude_lbl.set_text("%f" % device.fix[4])
                self.longitude_lbl.set_text("%f" % device.fix[5])
                self.speed_lbl.set_text("%f" % device.fix[11])
                self.previousfixtime = device.fix[2]
                
                print "Checking radar..."
                (rid, rname, rdistance) = self.mydb.get_nearest_radar(
                                                device.fix[4],  device.fix[5])
                
                if rid == -1: 
                    print "main : No radar detected"
                    self.name_lbl.set_text("None")
                    self.distance_lbl.set_text("None")
                    self.stop_alert()
                else:
                    print "main : Radar detected"
                    self.name_lbl.set_text(rname)
                    if rdistance > 1:
                        print "%.03f" % rdistance
                        self.distance_lbl.set_text("%.03f" % rdistance)
                        pass
                    else:
                        self.distance_lbl.set_text("%.01f" % (rdistance * 1000))
                        pass
                    print "Previous radar : %s, %s" % (
                            self.previousradarid,  self.previousradardistance)
                    print "Current radar : %s, %s" % (rid,  rdistance)
                    if (rdistance < self.previousradardistance) & (
                                                    rid==self.previousradarid) & (
                                                    device.fix[11] > 10):
                        print "main : start_alert"
                        self.alert_label.set_markup("<span size='250000'><b>%.0f</b></span>" % (rdistance*1000))
                        self.start_alert()
                        pass
                    else:
                        self.stop_alert()
                        print "main : stop_alert"
                        pass
                                
                self.previousradarid = rid
                self.previousradardistance = rdistance
                self.previousfixtime = device.fix[2]

    
    def build_interface(self):
        """Build main interface"""
        
        hbox = gtk.HBox(False, 0)
        vbox = gtk.VBox(False, 0)

        vbox.pack_start(gtk.Label("Last update : "), True, True, 0)
        vbox.pack_start(gtk.Label("Latitude : "), True, True, 0)
        vbox.pack_start(gtk.Label("Longitude : "), True, True, 0)
        vbox.pack_start(gtk.Label("Speed : "), True, True, 0)
        vbox.pack_start(gtk.Label("Speedcam name : "), True, True, 0)
        vbox.pack_start(gtk.Label("Speedcam distance : "), True, True, 0)
        hbox.pack_start(vbox, True, True, 0)
        
        vbox = gtk.VBox(False, 0)
        self.lastupdate_lbl = gtk.Label("")
        vbox.pack_start(self.lastupdate_lbl,  True, True, 0)
        self.latitude_lbl = gtk.Label("")
        vbox.pack_start(self.latitude_lbl,  True, True, 0)
        self.longitude_lbl = gtk.Label("")
        vbox.pack_start(self.longitude_lbl,  True, True, 0)
        self.speed_lbl = gtk.Label("")
        vbox.pack_start(self.speed_lbl,  True, True, 0)
        self.name_lbl = gtk.Label("")
        vbox.pack_start(self.name_lbl,  True, True, 0)
        self.distance_lbl = gtk.Label("")
        vbox.pack_start(self.distance_lbl,  True, True, 0)
        hbox.pack_start(vbox, True, True, 0)      
        
        vbox = gtk.VBox(False, 0)
        vbox.pack_start(hbox, True, True, 0)      
        self.pb = gtk.ProgressBar()
        vbox.pack_start(self.pb, False, True, 0)      
        self.window.add(vbox)

        
    def build_window_alert(self):
        """Build alert window"""
        self.alert_window = hildon.Window()
        self.add_window(self.alert_window)
        self.alert_window.grab_focus()
        self.alert_window.fullscreen()
        
        hbox = gtk.HBox(False, 0)

        color = gtk.gdk.color_parse('#FF0000')
        self.alert_window.modify_bg(gtk.STATE_NORMAL, color)

        self.alert_label = gtk.Label("Speed camera detected")        

        hbox.pack_start(self.alert_label, True, True, 0)
        self.alert_window.add(hbox)
        
        self.alert_window.set_transient_for(self.window)
        
        
    def start_alert(self):        
        """activate the alert"""
        
        if self.alert == False:
            print "start alert"
            self.osso_r.rpc_run('com.nokia.mce', '/com/nokia/mce/request',
                'com.nokia.mce.request', 'req_tklock_mode_change',
                ('unlocked',),
                False, True)
            gobject.timeout_add(10000, self.keep_display_on)
            gobject.timeout_add(1500, self.activate_vibrator)
            self.alert = True
            self.alert_window.show_all()
        
        
    def stop_alert(self):
        """desactivate alert"""
        
        if self.alert == True:
            print "stop alert"
            self.alert = False
            self.alert_window.hide_all()
            
    
    def keep_display_on(self):
        """Keep display on, when alert is on"""
        
        self.odevice.display_state_on()
        if self.alert == True:
            return True
        else:
            return False
            
            
    def activate_vibrator(self):
        self.osso_r.rpc_run('com.nokia.mce',
                                    '/com/nokia/mce/request', 
                                    'com.nokia.mce.request', 
                                    'req_vibrator_pattern_activate',
                                    ('PatternChatAndEmail',), 
                                    False, True)
        if self.alert == True:
            return True
        else:
            return False
    
    
    def test_alert(self):
        print "test_alert"
        if os.path.isfile("zz"):
            self.start_alert()
        else:
            self.stop_alert()
        return True
        
        
    def warning_message(self,  message):
        """"Display error dialog window"""
        
        boite = gtk.Dialog("Error", self.window,
            gtk.DIALOG_MODAL,
            (gtk.STOCK_OK, gtk.RESPONSE_YES))
        msg_label = gtk.Label(message)
        msg_label.set_line_wrap(True)
        msg_label.set_size_request(-1, 300)
        boite.vbox.pack_start(msg_label, True, True, 0)
        boite.vbox.show_all()

        reponse = boite.run()
        boite.destroy()

