/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "CSVSymbianEventLogWriter.h"

#include "EventTypes/iCSVSymbianEvent.h"
#include "EventTypes/iEvent.h"
#include "Settings.h"

#include <QDebug>

#include <QDir>
#include <QFile>
#include <QTextStream>

#include <typeinfo>
#include <utime.h>

using namespace EventWriters;

CSVSymbianEventLogWriter::CSVSymbianEventLogWriter(const Settings &settings, const NumberToNameLookup &numberToNameLookup)
	: m_Settings(settings), m_NumberToNameLookup(numberToNameLookup), m_Stream(NULL), m_Delimiter('|'), m_File(NULL)
{
	m_Headers << "EType" << "ETime" << "DType" << "Flag1" << "Flag2" << "Flag3"
			<< "Flag4" << "Id" << "Remote" << "Direction" << "Duration"
			<< "Status" << "Subject" << "Number" << "Contact" << "Link"
			<< "Data" << "Recent" << "Duplicate";
}

CSVSymbianEventLogWriter::~CSVSymbianEventLogWriter()
{
	if(Stream())
	{
		Stream()->flush();
		delete Stream();
	}

	if(File())
	{
		File()->close();
		delete File();
	}

	// Build the strings filepath
	QString stringsFilename(CurrentSettings().Directory());
	stringsFilename += "/Call Logs/Strings.csv";

	QFile stringsFile(stringsFilename);
	if (stringsFile.open(QFile::WriteOnly | QFile::Truncate))
	{
		QTextStream stream(&stringsFile);
		stream.setAutoDetectUnicode(false);
		stream.setCodec("UTF-16LE");

		WriteStrings(stream);
	}
}

void CSVSymbianEventLogWriter::Write(EventTypes::iEvent &event)
{
	EventTypes::iCSVSymbianEvent *csvEvent(dynamic_cast<EventTypes::iCSVSymbianEvent *>(&event));
	if(csvEvent)
	{
		if(!Stream())
			OpenStream();

		csvEvent->WriteCSVSymbian(*Stream(), Headers(), Delimiter(), NameLookup(), Strings());
	}
}

void CSVSymbianEventLogWriter::OpenStream()
{
	// Build the path and ensure it exists...
	QString eventFilename(CurrentSettings().Directory());
	eventFilename += "/Call Logs";
	QDir().mkpath(eventFilename);

	// ...then build the filename and open it.
	eventFilename += "/Symbian Event Log.csv";
	File(new QFile(eventFilename));
	if (File()->open(QFile::WriteOnly | QFile::Truncate))
	{
		Stream(new QTextStream(File()));
		Stream()->setAutoDetectUnicode(false);
		Stream()->setCodec("UTF-16LE");

		WriteHeaders();
	}
}

void CSVSymbianEventLogWriter::WriteHeaders()
{
	foreach(QString header, Headers())
	{
		*Stream() << header;
		if(Headers().back() != header)
			*Stream() << "|";
	}
	*Stream() << endl;
}

void CSVSymbianEventLogWriter::WriteStrings(QTextStream &stringsStream)
{
	foreach(QString string, Strings().keys())
		stringsStream << "\"" << string << "\"|" << Strings().value(string) << endl;
}
