/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "VMGParser.h"

#include "EventParsers/VMGEntities/Factory.h"
#include "EventParsers/VMGEntities/iReader.h"
#include "EventTypes/SMS.h"

#include <QDebug>

#include <QFile>
#include <QHash>
#include <QList>
#include <QString>
#include <QStringList>

#include <stdexcept>

using namespace EventParsers;

iEventParser *VMGParser::IsValid(const Settings &settings, QFile &eventFile)
{
	qDebug() << "Checking if a VMG file...";

	char fileIDBuf[0x16];
	qint64 bytesRead(eventFile.read(fileIDBuf, sizeof(fileIDBuf) - 1));
	eventFile.seek(0);
	if(bytesRead == sizeof(fileIDBuf) - 1)
	{
		// This is the string BEGIN:VMSG<LF> in little-endian UTF16.
		char signatureVMG[] = { 0x42, 0x00, 0x45, 0x00, 0x47, 0x00,
								0x49, 0x00, 0x4E, 0x00, 0x3A, 0x00,
								0x56, 0x00, 0x4D, 0x00, 0x53, 0x00,
								0x47, 0x00, 0x0A, 0x00 };

		if(memcmp(fileIDBuf, signatureVMG, sizeof(fileIDBuf) - 1) == 0)
		{
			qDebug() << eventFile.fileName() << " is a VMG file.";
			return new VMGParser(settings, eventFile.fileName());
		}
		else
		{
			QString hexs;
			for(int i(0); i<0x16; ++i)
				hexs.append(QString::number(fileIDBuf[i], 16).rightJustified(2, '0'));
			qDebug() << eventFile.fileName() << " has bad signature: " << hexs;
		}
	}
	else
		qDebug() << eventFile.fileName() << " has size mismatch.";

	return false;
}

VMGParser::VMGParser(const Settings &settings, const QString &/*filename*/) : m_Settings(settings)
{
}

EventTypes::EventFromFileList VMGParser::ParseFile(QFile &eventFile, const QList<uint> &recordsToReturn)
{
	EventTypes::EventFromFileList retList;

	// VMGs only support single events per file
	if(recordsToReturn.count() == 0 || recordsToReturn.contains(0))
	{
		// VMG files are stored in Little-Endian UTF16, with no BOM.
		QTextStream eventStream(&eventFile);
		eventStream.setCodec("UTF-16LE");

		// Parse the event
		EventTypes::SMS *event(new EventTypes::SMS(CurrentSettings()));
		QString lineData = eventStream.readLine();
		EventParsers::VMGEntities::iReader* reader = EventParsers::VMGEntities::Factory::Instantiate(CurrentSettings(), lineData, NULL);
		bool valid(NULL != reader && reader->Read(QString(""), eventStream, *event));
		delete reader;
		if (!valid)
			throw std::runtime_error(QString("Unsupported format. Unable to open: %1").arg(eventFile.fileName()).toStdString());

		qDebug() << "\nParsed event:";
		qDebug() << event;

		retList.append(EventTypes::EventFromFile(QSharedPointer<EventTypes::iEvent>(event), 0));
	}

	return retList;
}
