/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#ifndef SETTINGS_H
#define SETTINGS_H

#include <QString>

class Settings
{
public:
	enum eAppMode
	{
		APPMODE_CONSOLE,
		APPMODE_GUI
	};

	// Options for main mode - importing or exporting content
	enum eMode
	{
		MODE_IMPORT,
		MODE_EXPORT,
	};
	static const int kNumModes = MODE_EXPORT + 1;

	// Options for targets - sent or recieved
	enum eTargets
	{
		TYPE_SENT,
		TYPE_RECIEVED,
	};
	static const int kNumTargets = TYPE_RECIEVED + 1;

	// Options for supported event types
	enum eEventTypes
	{
		EVENTTYPE_SMS,
		EVENTTYPE_MMS,
		EVENTTYPE_CHAT,
	};
	static const int kNumEventTypes = EVENTTYPE_CHAT + 1;

private:
	// The current UI mode of the application (Console/GUI)
	eAppMode m_AppMode;

	// Have the settings been confirmed by the user?
	bool m_Confirmed;

	// The current operation mode of the application (Import/Export)
	eMode m_Mode;

	// The current targets & event types being processed
	bool m_ShouldProcess[kNumTargets][kNumEventTypes];

	// The root directory to import or export from
	QString m_Directory;

	// The ITU code for this country
	uint m_CountryCode;

	// Is it OK to disable the cellular radio to prevent calls and messages?
	bool m_DisableCellular;

public:
	Settings();

	eAppMode const AppMode() const { return m_AppMode; }
	eAppMode const setAppMode(const eAppMode appMode) { return m_AppMode = appMode; }

	bool const IsConfirmed() const { return m_Confirmed; }
	bool const setConfirmed(const bool confirmed) { return m_Confirmed = confirmed; }

	eMode const Mode() const { return m_Mode; }
	eMode const setMode(const eMode mode) { return m_Mode = mode; }

	bool const ShouldProcess(const eTargets target, const eEventTypes eventType) const { return m_ShouldProcess[target][eventType]; }
	bool const setShouldProcess(const eTargets target, const eEventTypes eventType, const bool shouldProcess = true){ return m_ShouldProcess[target][eventType] = shouldProcess; }
	bool const anyToProcess()
	{
		for(int targetIndex = 0; targetIndex < kNumTargets; ++targetIndex)
			for(int eventTypeIndex = 0; eventTypeIndex < kNumEventTypes; ++eventTypeIndex)
				if (m_ShouldProcess[targetIndex][eventTypeIndex])
					return true;
		return false;
	}

	QString const Directory() const{ return m_Directory; }
	QString const setDirectory(const QString& directory){ return m_Directory = directory; }

	uint const CountryCode() const { return m_CountryCode; }
	uint const setCountryCode(const uint countryCode) { return m_CountryCode = countryCode; }

	bool const DisableCellular() const { return m_DisableCellular; }
	bool const setDisableCellular(const bool disableCellular) { return m_DisableCellular = disableCellular; }
};

#endif // SETTINGS_H
