/**************************************************************************
**
** This file is part of Qt Creator
**
** Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
**
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** Commercial Usage
**
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
**
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at http://qt.nokia.com/contact.
**
**************************************************************************/

#include <signal.h>
#include <sys/types.h>

#include <QtCore/QTime>
#include <QtDeclarative/qdeclarative.h>
#include <QtCore/QDebug>
#include <QtCore/QTextCodec>

#include "ProcessObject.h"

ProcessObject::ProcessObject(QObject *parent):
        QObject(parent),
        m_suspend(false), m_started(false)
{
    m_process = new QProcess(this);
    connect(m_process, SIGNAL(finished(int, QProcess::ExitStatus)),
            this,
            SLOT(processFinished(int, QProcess::ExitStatus)));
    connect(m_process, SIGNAL(error(QProcess::ProcessError)),this,
            SLOT(processErrored(QProcess::ProcessError)));
}

ProcessObject::~ProcessObject()
{
    if (m_started)
    {
        terminate();
    }
    delete m_process;
}

void ProcessObject::processErrored(QProcess::ProcessError error) {
    QString err(m_process->readAllStandardError());
    qDebug() << "failed with exitCode" << m_process->exitCode();
    emit failed(m_process->exitCode(), err, error);
    m_started = false;
}

void ProcessObject::processFinished(int exitCode,
                                    QProcess::ExitStatus exitStatus)
{
    QTextCodec* codec = QTextCodec::codecForName("utf-8");
    if (exitStatus == QProcess::NormalExit)
    {
        if (exitCode == 0)
        {	 
	  QString out = codec->toUnicode(m_process->readAllStandardOutput());
	  qDebug() << "normal";
	  emit completed(out.trimmed());
        }
        else
        {
	  QString err = codec->toUnicode(m_process->readAllStandardError());
	  qDebug() << "failed";
	  emit failed(exitCode, err.trimmed(), m_process->error());
        }
    }
    else
    {
	QString err = codec->toUnicode(m_process->readAllStandardError());
        qDebug() << "failed";
        emit failed(exitCode, err, m_process->error());
    }
    m_started = false;
}

QString ProcessObject::command() const
{
    return m_command;
}

void ProcessObject::setCommand(const QString &command)
{
    if (m_command != command) {
        m_command = command;
    }
}

void ProcessObject::run()
{
    if (m_started)
    {
        return;
        qDebug() << "already started";
    }
    qDebug() << m_command;
    m_process->start(m_command);
    qDebug() << "started";
    m_started = true;
}

bool ProcessObject::isSuspended() const
{
    return m_suspend;
}

void ProcessObject::setSuspend(bool suspend)
{
    if (!m_started || m_process->pid() == 0)
    {
        qDebug() << "No process running";
        return;
    }
    if (m_suspend != suspend)
    {
        if (m_suspend)
        {
            ::kill(m_process->pid(), SIGCONT);
            qDebug() << "Process resumed";

        } else {
            ::kill(m_process->pid(), SIGSTOP);
            qDebug() << "Process suspended";

        }
        emit suspendChanged(suspend);
        m_suspend = suspend;
    }
}

void ProcessObject::terminate()
{
    if (m_started && m_process->pid() != 0)
    {
        m_process->kill();
    }
}


QML_DECLARE_TYPE(ProcessObject);
