/*
 * Copyright (c) 2010 David Galindo <nowheremanmail@gmail.com>

 */

#include <openssl/aes.h>
#include <openssl/evp.h>
#include <openssl/sha.h>

#include "utils.h"
#include "Constants.h"
#include "DatabaseManager.h"
#include "eWallet.h"
#include <QStringList>
#include <QDir>
#include <QVariant>
#include <QBuffer>
#include <QFile>
#include <QDesktopServices>
#include <QDebug>



#define nBASE64

// ---------------------------------------------------------------------------
// Renter
// ---------------------------------------------------------------------------
Target::Target()
{
	name = "";
	groupName = "";
	type = "";
	id = 0;
	pictureSmall = NULL;
	picture = NULL;
	pictureBack = NULL;
	listLoaded = false;
}
Target::~Target()
{
	freeFieldList ();
	freeResources();
	if(pictureSmall) delete pictureSmall;
}

void Target::freeResources()
{
	//qDebug () << " free resources for " << this->id;
	if(picture) delete picture;
	if(pictureBack) delete pictureBack;
	picture = NULL;
	pictureBack = NULL;
}

//bool operator<(const Entry &entry, const QString &family)
//{
//    return entry.familyName < family;
//}
//
//bool operator<(const QString &family, const Entry &entry)


void Target::freeFieldList()
{
	//qDebug () << " free list for " << this->id;
	TargetField * field;
	foreach(field,this->fields) {
		delete field;
	}
	fields.clear ();
	listLoaded = false;
}

bool Target::existField (QString name) {
	TargetField * field;
	foreach(field,this->fields) {
		if (field->name == name) return true;
	}
	return false;
}

int Target::getFieldByName (QString name) {
	for (int i = 0; i < this->fields.size(); i++) {
		TargetField * field = this->fields.at(i);
		if (field->name == name) return i;
	}
	return -1;
}

QString Target::getName () {
	return name;
//	QString tmp = "<qt>%1<br>%2 %3</qt>";
//	tmp = tmp.arg(this->name).arg(this->creationdate).arg(this->modificationdate);
////	//return this->type;
//	return tmp;
}

QString Target::getSummary () {
//	QString tmp;
//	tmp << this->type << " " << this->creationdate << " " << this->modificationdate;
	return this->type;
//	return tmp;
}
QImage * Target::getPictureSmall () {
	////qDebug () << "getPictureSmall " << this->id << " " << pictureSmall;
	if(!pictureSmall) {
		QImage * orig = getPicture();

		if (orig) {
			pictureSmall = new QImage (orig->scaled (SMALL_WIDTH, SMALL_HEIGH));
		}
		else
			pictureSmall = NULL;

	}
	return pictureSmall;
}
QImage * Target::getPicture () {
	return getPicture (::dbManager);
}
QImage * Target::getPictureBack () {
	return getPictureBack (::dbManager);
}
QImage * Target::getPicture (DatabaseManager * mgr) {
	if(!picture) {
		picture = mgr->getPicture (this);
	}
	return picture;
}
QImage * Target::getPictureBack (DatabaseManager * mgr) {
	if(!pictureBack) {
		pictureBack = mgr->getPictureBack (this);
	}
	return pictureBack;
}
/*void Target::setPictureSmall (QImage * a) {
	if(pictureSmall) delete pictureSmall;
	pictureSmall = a;
}*/
void Target::setPicture (QImage * a) {
	if(picture) delete picture;
	picture = a;
	if(pictureSmall) delete pictureSmall;
	pictureSmall = NULL;
}
void Target::setPictureBack (QImage * a) {
	if(pictureBack) delete pictureBack;
	pictureBack = a;
}

TargetField::~TargetField()
{
}
// ---------------------------------------------------------------------------
// DatabaseManager
// ---------------------------------------------------------------------------
DatabaseManager::DatabaseManager(QObject *parent) :
		QObject(parent)
{
	opendb = false;
	initdb = false;
}

DatabaseManager::~DatabaseManager()
{
	if (connectionName.length() > 0)
		db.removeDatabase(connectionName);
//	else
//		db.removeDatabase();

	//qDebug () << "CLOSE DB " << connectionName;
	if (db.isOpen()) {
		//qDebug () << "CLOSing DB";
		db.close();
	}
}

bool DatabaseManager::isInit()
{
	//qDebug () << "is init?";


	return db.tables().count() >= 3;
}

bool DatabaseManager::initDB()
{
	//qDebug () << "INIT DB " << connectionName;
	bool ret = true;

	// Create tables
	if (createIdTable()) {
		createTargetTable();
		createTargetFieldTable();
		createTableVersion ();
		ret = addVersion (SOFT_VERSION);
	}
	currentVersion = getVersion ();

	// Check that tables exists
	if (db.tables().count() != 4)
		ret = false;

	return ret;
}

bool DatabaseManager::migrateVersion (int fromV, int toV, const QString & pas) {
	bool ret = true;

	qDebug () << "MIGRATING DB VERSION FROM " << fromV << " TO " << toV;

	if (fromV == 0 && toV == 1) {
		if (backupDB ("BACK")) {
			QSqlQuery query(db);
			ret = query.exec("alter table targets ADD COLUMN groupName varchar(100)");
		}
		else
			ret = false;
	}

	if (fromV == 1 && toV == 2) {
		if (backupDB ("BACK")) {
			if (createPassword(pas, 2)) {
				if (!addVersion(2)) {
					ret = false;
				}
			}
			else
				ret = false;
		}
		else
			ret = false;
	}

	if (fromV == 2 && toV == 3) {
		if (backupDB ("BACK")) {
			QSqlQuery query(db);
			qDebug () << "query 0 " << ret;
			ret &= query.exec("alter table targets ADD COLUMN creationdate varchar(24)");
			qDebug () << "query 1 " << ret << " " << query.lastError();
			ret &= query.exec("alter table targets ADD COLUMN modificationdate varchar(24)");
			qDebug () << "query 2 " << ret << " " << query.lastError();

			ret &= query.exec("update targets set creationdate = CURRENT_TIMESTAMP, modificationdate = CURRENT_TIMESTAMP");
			qDebug () << "query 3 " << ret << " " << query.lastError();

			if (ret && !addVersion(3)) {
				ret = false;
			}
		}
		else
			ret = false;
	}

	//qDebug () << "MIGRATED DB VERSION FROM " << fromV << " TO " << toV << " WITH RESULT " << ret;

	return ret;
}

bool DatabaseManager::deleteDB(const QString & prefix)
{
	//qDebug () << "REMOVE DB " << connectionName;
	db.close();

	QString path(QDir::home().path());
	path.append(QDir::separator()).append(prefix).append("qtwallet.db.sqlite");
	path = QDir::toNativeSeparators(path);

	return QFile::remove(path);
}
bool DatabaseManager::backupDB(const QString & prefix) {
	QString path(QDir::home().path());
	path.append(QDir::separator()).append("qtwallet.db.sqlite");
	path = QDir::toNativeSeparators(path);

	QString pathBack(QDir::home().path());
	pathBack.append(QDir::separator()).append(prefix).append("qtwallet.db.sqlite");
	pathBack = QDir::toNativeSeparators(pathBack);

	bool dup = true;

	QFile fileBack (pathBack);
	if (fileBack.exists())
		dup = fileBack.remove();

	if (dup) {
		QFile file (path);
		dup = file.copy(pathBack);
	}

	qDebug () << "duplicate " << prefix << " DB " << dup;

	return dup;
}
bool DatabaseManager::restoreDB(const QString & prefix) {
	QString path(QDir::home().path());
	path.append(QDir::separator()).append("qtwallet.db.sqlite");
	path = QDir::toNativeSeparators(path);

	QString pathBack(QDir::home().path());
	pathBack.append(QDir::separator()).append(prefix).append("qtwallet.db.sqlite");
	pathBack = QDir::toNativeSeparators(pathBack);

	bool dup = true;

	QFile file (path);
	if (file.exists())
		dup = file.remove();

	if (dup) {
		QFile fileBack (pathBack);
		dup = fileBack.copy(path);
	}

	//qDebug () << "restore " << prefix << " DB " << dup;

	return dup;
}
//bool DatabaseManager::openDB()
//{
//	return _openDB ("qtwallet.db.sqlite");
//}

bool DatabaseManager::migrationFrom (DatabaseManager * mgr) {
	bool res;

    QList<Target *> list = mgr->getTargets(NULL);

	for (int i = 0; i < list.size (); i++) {
		Target * target = list[i];

		//qDebug () << "MIGRATING " << target->name << " " << target->id;

		mgr->getFields(target);
		target->getPicture(mgr);
		target->getPictureBack(mgr);
		res = copyTarget(target);

		//qDebug () << "MIGRATING 1 " << target->name << res << " " << target->id;
		if (!res) return false;

//		if (target->getPicture(mgr) != NULL) {
//			res = updateTargetPicture (target);
//			//qDebug () << "MIGRATING 2 " << target->name << res;
//			if (!res) return false;
//		}
//
//		if (target->getPictureBack(mgr) != NULL) {
//			res = updateTargetPictureBack (target);
//			//qDebug () << "MIGRATING 3 " << target->name << res;
//			if (!res) return false;
//		}

		res = updateFields (target);
		//qDebug () << "MIGRATING 4 " << target->name << res;
		if (!res) return false;

		target->freeFieldList();
		target->freeResources();
	}

	return res;
}

bool DatabaseManager::_openDB(const QString & name, const QString & cName) {
	//qDebug () << "OPEN DB " << name << "(" << cName << ")";

	connectionName = cName;

	// Find QSLite driver
	if (connectionName.length() > 0)
		db = QSqlDatabase::addDatabase("QSQLITE", connectionName);
	else
		db = QSqlDatabase::addDatabase("QSQLITE");
	// http://doc.trolltech.com/4.5/sql-driver.html#qsqlite
	QString path(QDir::home().path());
	path.append(QDir::separator()).append(name);
	path = QDir::toNativeSeparators(path);

	db.setDatabaseName(path);

	// Open databasee
	bool res = false;

	if (db.open()) {
		return true;
	}
	return false;
}

QSqlError DatabaseManager::lastError()
{
	return db.lastError();
}

bool DatabaseManager::createIdTable()
{
	// Create table
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);
		ret = query.exec("create table id (id integer primary key, phrase varchar(20))");
	}
	return ret;
}

bool DatabaseManager::createTableVersion()
{
	// Create table
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);
		ret = query.exec("create table softwareVersion (versionNumber integer)");
	}
	return ret;
}
int DatabaseManager::getVersion() {
	if (db.isOpen()) {
		QSqlQuery query (db);
		bool ret = query.exec("select max(versionNumber) from softwareVersion");
		if (ret) {
			if (query.next()) {
				return query.value(0).toInt();
			}
		}
	}
	return -1;
}
bool DatabaseManager::addVersion(int v) {
	qDebug () << "add new version " << v;

	QSqlQuery query (db);
	bool ret = query.prepare("insert into softwareVersion values (:ver)");
	if (ret) {
		query.bindValue(":ver", v);
		ret = query.exec();
	}
	return ret;
}

int DatabaseManager::nextId()
{
	int ret = -1;
	if (db.isOpen()) {
		QSqlQuery query("select id from id", db);
		if (query.next()) {
			// Get last used id
			ret = query.value(0).toInt();
			// Increase that
			ret++;
			// Store new value
			query.exec(QString("update id set id=%1 where id=%2").arg(ret).arg(ret - 1));
		} else {
			// Set first id to zero
			if (!query.exec("insert into id values(1, NULL)"))
				return -1;
			else
				ret = 1;
		}
	}

	return ret;
}

void DatabaseManager::reset () {
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);
		ret |= query.exec("drop table id");
		ret |= query.exec("drop table targets");
		ret |= query.exec("drop table fields");
		ret |= query.exec("drop table softwareVersion");
	}

}

QString DatabaseManager::changePassword (const QString & newPassword) {
	bool ret = false;

	QString result = "";

	ret = backupDB("BACK");

	if (ret) {
		DatabaseManager * oldDB = new DatabaseManager();
		ret = oldDB->_openDB("NEWqtwallet.db.sqlite", "NEW");
		if (ret) {
			oldDB->reset ();
			ret = oldDB->initDB();
			if (ret){
				ret = oldDB->createPassword(newPassword, oldDB->currentVersion);
				if (ret) {
					if (oldDB->migrationFrom(this)) {
						//qDebug () << "succesfull migrated ";
						db.close ();

						ret = oldDB->restoreDB ("NEW");
						if (ret) {
							//qDebug () << "succesfull changed ";

						}
						else {
							result = "DB cannot be restored";
						}

					}
					else
						result = "NEWqtwallet.db.sqlite cannot be migrated";
				}
				else
					result = "NEWqtwallet.db.sqlite cannot be init";
			}
			else
				result = "NEWqtwallet.db.sqlite cannot be reset";

			oldDB->deleteDB("NEW");
		}
		else
			result = "NEWqtwallet.db.sqlite cannot be opened";
		delete oldDB;
	}

	return result;
}


void DatabaseManager::lock () {
	initdb = false;
}

bool DatabaseManager::hasPassword() {
	if (!opendb)
		opendb = _openDB("qtwallet.db.sqlite");

	return opendb && isInit () && hasSecretPhrase();
}

bool DatabaseManager::openDB(const QString & pas) {
	initdb = false;

	//qDebug() << "open db with pas " << SOFT_VERSION;

	if (!opendb)
		opendb = _openDB("qtwallet.db.sqlite");

	if (!opendb) return false;

	//qDebug() << "init db with pas";

	if (db.tables().count() == 0) {
		if (initDB())
			initdb = createPassword(pas, SOFT_VERSION);
		currentVersion = SOFT_VERSION;
	}
	else {
		bool res = false;
		// Check that tables exists
		if (db.tables().count() == 3) {
#ifdef Q_WS_MAEMO_5
			show_banner(NULL, tr("Migrating database from version %1, please wait").arg(0));
#endif
			res = migrateVersion (0, 1, pas);
			if (res)
				res = createTableVersion ();
			if (res)
				res = addVersion (1);
			currentVersion = 1;
		}

		//qDebug() << "init db with pas taules "<< db.tables().count();

		if (db.tables().count() != 4) {
			initdb = false;
			currentVersion = SOFT_VERSION;
		}
		else {
			currentVersion = getVersion ();
			if (currentVersion < SOFT_VERSION) {
				do {
#ifdef Q_WS_MAEMO_5
			show_banner(NULL, tr("Migrating database from version %1, please wait").arg(currentVersion));
#endif
					res = migrateVersion (currentVersion, currentVersion + 1, pas);
					if (res)
						currentVersion =  getVersion ();
				}
				while (res && currentVersion < SOFT_VERSION);
			}
			else
				res = true;


			if (currentVersion != SOFT_VERSION || !res) {
				// TODO show an error!!!
#ifdef Q_WS_MAEMO_5
		show_information_note (NULL, tr("Migration not possible, stop at version %1").arg(currentVersion));
#endif
				qDebug () << "MIGRATION NOT POSSIBLE " << currentVersion << " to " << SOFT_VERSION;
				return false;
			}

			if (hasSecretPhrase()) {
				initdb = checkPassword(pas, currentVersion);
			}
			else {
				initdb = createPassword(pas, currentVersion);
			}
		}
	}

	//qDebug() << "init db with pas " << initdb << " " << currentVersion;

	return initdb;
}

bool DatabaseManager::isReady () {
	return opendb && initdb;
}

bool DatabaseManager::checkPassword (QString _a, int version)
{
	//qDebug () << "checkPassword " << version << "(" << connectionName << ")"; // << _a;

	QByteArray t = _a.toLocal8Bit();
	
	keyLen = 32*8;
	for (int i = 0; i < 32; i++) {
		key [i] = i; 
		
		if (i < t.size ()) {
			key[i] = (unsigned char)t[i];
		}
	}
	/*unsigned char key16[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};
	unsigned char key24[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23};
	unsigned char key32[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31};*/

//	int AES_set_encrypt_key(const unsigned char *userKey, const int bits,
//		AES_KEY *key);

	AES_set_encrypt_key(key, keyLen, &aeskeyE);
	AES_set_decrypt_key(key, keyLen, &aeskeyD);
	
	QByteArray a;
	if (version < 2)
		encrypt(QString(PASS1), a);
	else {
		QString pp(_a);
		pp.append (PASS2);
		encryptSHA1(pp.mid(0, SHA1_LEN), a);
	}

	QByteArray b = selectSecretPhrase ();
	
	//qDebug()<< "checkPassword [[[[[" << b << "]]]]]";
	
	return a == b;
}

bool DatabaseManager::createPassword (QString _a, int version)
{
	//qDebug () << "createPassword " << version << "(" << connectionName << ")";

	QByteArray t = _a.toLocal8Bit();
	
	keyLen = 32*8;
	for (int i = 0; i < 32; i++) {
		key [i] = i; 
		
		if (i < t.size ()) {
			key[i] = (unsigned char)t[i];
		}
	}
	/*unsigned char key16[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};
	unsigned char key24[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23};
	unsigned char key32[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31};*/
	AES_set_encrypt_key(key, keyLen, &aeskeyE);
	AES_set_decrypt_key(key, keyLen, &aeskeyD);
	
	QByteArray a;
	if (version < 2)
		encrypt(QString(PASS1), a);
	else {
		QString pp(_a);
		pp.append (PASS2);
		encryptSHA1(pp.mid(0, SHA1_LEN), a); // encrypt
	}
	
	////qDebug () << "create secret " << a;
	
	return updateSecretPhrase (a);
}

bool DatabaseManager::updateSecretPhrase(QByteArray a) {
	////qDebug () << "update secret " << a;

	bool ret = false;
	if (nextId () > 0) {
		QSqlQuery query (db);
		ret = query.prepare("update id set phrase = :phrase");
		if (ret) {
			query.bindValue(":phrase", a);
			ret = query.exec();
			////qDebug () << "update secret " << ret;
			return ret;
		}
	}
	return ret;
}

bool DatabaseManager::hasSecretPhrase()
{
	//qDebug () << "has secret?";
	
	QSqlQuery query("select phrase from id", db);
	if (query.next()) {
		return !query.value(0).isNull();
	}
	//query.clear();
	return false;
}

QByteArray DatabaseManager::selectSecretPhrase()
{
	//qDebug () << "get secret?";

	QByteArray ret;
	QSqlQuery query("select phrase from id", db);
	if (query.next()) {
		return query.value(0).toByteArray();
	}

	return ret;
}

bool DatabaseManager::createTargetTable()
{
	// Create table
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);
		ret = query.exec("create table targets "
		                 "(id integer primary key, " // this is autoincrement field http://www.sqlite.org/autoinc.html
		                 "name varchar(30), "
		                 "picture BLOB, "
		                 "pictureBack BLOB, "
		                 "type varchar(5), "
		                 "groupName varchar(100), "
		                 "creationdate varchar(24), "
		                 "modificationdate varchar(24))");

	}
	return ret;
}

bool DatabaseManager::createTargetFieldTable()
{
	// Create table
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);
		ret = query.exec("create table fields "
		                 "(id integer primary key, " // this is autoincrement field http://www.sqlite.org/autoinc.html
		                 "targetId integer, "
		                 "orderN integer, "
		                 "encrypted integer, "
		                 "name varchar(30), "
		                 "value varchar(30), "
		                 "type integer)");

	}
	return ret;
}

void DatabaseManager::encryptSHA1 (const QString & a, QByteArray& b) {
	QByteArray t = a.toLocal8Bit();
	int length = t.size();

	SHA512_CTX context;
	unsigned char md[SHA512_DIGEST_LENGTH];

	SHA512_Init(&context);
	SHA512_Update(&context, (unsigned char*)t.constData(), length);
	SHA512_Final(md, &context);

#ifdef BASE64
	b = QByteArray((char*)md, SHA512_DIGEST_LENGTH).toBase64();
#else
	b = QByteArray((char*)md, SHA512_DIGEST_LENGTH);
#endif
}


void DatabaseManager::encrypt (const QString a, QByteArray& b)
{
	unsigned char ivE[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};

	QByteArray t = a.toUtf8();
	int size = t.size();
	int c = size / 16 + 1;
	short p = 16 - size % 16;

	unsigned char pout [c * 16];
	unsigned char pin [c * 16];
	memset (pin, p, c * 16);
	memset(pout, 0, c * 16);
	memcpy(pin, (unsigned char *)t.constData(), size);

	AES_cbc_encrypt(pin, pout, c*16, &aeskeyE, ivE, AES_ENCRYPT);

#ifdef BASE64
	b = QByteArray((char*)pout, c * 16).toBase64();
#else
	b = QByteArray((char*)pout, c * 16);
#endif
}

#ifdef BASE64
void DatabaseManager::decrypt (const QByteArray& aa, QString& b)
{
	QByteArray a = QByteArray::fromBase64(aa);
#else
void DatabaseManager::decrypt (const QByteArray& a, QString& b)
{
#endif
	unsigned char ivD[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};

	int size = a.size ();
	int c = size / 16;
	unsigned char pout [size];
	unsigned char pin [size];
	memset(pout, 0, size);
	memcpy(pin, (unsigned char *)a.constData(), size);

	AES_cbc_encrypt(pin, pout, c*16, &aeskeyD, ivD, AES_DECRYPT);
	int ss = size-(short)(pout[size-1]);
	pout [ss] = 0;

	b = QString::fromUtf8((char*)pout, ss);
}

void DatabaseManager::encrypt (const QByteArray& t, QByteArray& b)
{
	unsigned char ivE[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};

	int size = t.size();
	int c = size / 16 + 1;
	short p = 16 - size % 16;

	unsigned char pout [c * 16];
	unsigned char pin [c * 16];
	memset (pin, p, c * 16);
	memset(pout, 0, c * 16);
	memcpy(pin, (unsigned char *)t.constData(), size);

	AES_cbc_encrypt(pin, pout, c*16, &aeskeyE, ivE, AES_ENCRYPT);

#ifdef BASE64
	b = QByteArray((char*)pout, c * 16).toBase64();
#else
	b = QByteArray((char*)pout, c * 16);
#endif
}

#ifdef BASE64 
void DatabaseManager::decrypt (const QByteArray& aa, QByteArray& b)
{
	QByteArray a = QByteArray::fromBase64(aa);
#else
void DatabaseManager::decrypt (const QByteArray& a, QByteArray& b)
{
#endif
	unsigned char ivD[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};
	int size = a.size ();
	int c = size / 16;
	unsigned char pout [size];
	unsigned char pin [size];
	memset(pout, 0, size);
	memcpy(pin, (unsigned char *)a.constData(), size);

	AES_cbc_encrypt(pin, pout, c*16, &aeskeyD, ivD, AES_DECRYPT);
	int l = size-(short)(pout[size-1]); //pout [l] = 0;

	b = QByteArray ((char*)pout, l);
}

bool DatabaseManager::insertTarget(Target* target)
{
		//qDebug() << "step 0" ;

	bool ret = false;
	if (db.isOpen()) {
		//target->id = nextId(); // We demostrates autoincrement in this case

		// http://www.sqlite.org/autoinc.html
		// NULL = is the keyword for the autoincrement to generate next value
		QSqlQuery query (db);
		ret = query.prepare("INSERT INTO targets (name, type, picture, pictureBack, groupName, creationdate, modificationdate) VALUES (:name, :type, :picture, :pictureBack, :groupName, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)");
		QByteArray t1;

		//qDebug() << "step 1" ;
		if (ret) {
		//qDebug() << "step 2" ;
			encrypt(target->name, t1);
		//qDebug() << "step 3" ;
			query.bindValue(":name", t1);
			query.bindValue(":type", target->type);
			query.bindValue(":groupName", target->groupName);
					//qDebug() << "step 4" ;

			QByteArray im1;
			QBuffer buffer1(&im1);
			QByteArray im_r1;
			QByteArray im2;
			QBuffer buffer2(&im2);
			QByteArray im_r2;
		//qDebug() << "step 5" ;

			if (target->picture != NULL) {
				buffer1.open(QIODevice::WriteOnly);
				target->picture->save(&buffer1, "PNG"); // writes image into ba in PNG format

				encrypt(im1, im_r1);
				query.bindValue(":picture", im_r1);
			} else
				query.bindValue(":picture", QVariant (QVariant::ByteArray));
		//qDebug() << "step 6" ;

			if (target->pictureBack != NULL) {
				buffer2.open(QIODevice::WriteOnly);
				target->pictureBack->save(&buffer2, "PNG"); // writes image into ba in PNG format

				QByteArray im_r2;
				encrypt(im2, im_r2);
				query.bindValue(":pictureBack", im_r2);
			} else
				query.bindValue(":pictureBack", QVariant (QVariant::ByteArray));
						//qDebug() << "step 7" ;

			ret = query.exec();
		}
		// Get database given autoincrement value
		if (ret) {
			// http://www.sqlite.org/c3ref/last_insert_rowid.html
			target->id = query.lastInsertId().toInt();
			//qDebug () << "INSERT " << target->id;

		}
	}
	return ret;
}

bool DatabaseManager::copyTarget(Target* target)
{
		//qDebug() << "step 0" ;

	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);
		ret = query.prepare("INSERT INTO targets (id, name, type, picture, pictureBack, groupName, creationdate, modificationdate) VALUES (:id, :name, :type, :picture, :pictureBack, :groupName, :creationdate, :modificationdate)");
		QByteArray t1;

		if (ret) {
			encrypt(target->name, t1);
			query.bindValue(":id", target->id);
			query.bindValue(":name", t1);
			query.bindValue(":type", target->type);
			query.bindValue(":groupName", target->groupName);
			query.bindValue(":creationdate", target->creationdate);
			query.bindValue(":modificationdate", target->modificationdate);
					//qDebug() << "step 4" ;

			QByteArray im1;
			QBuffer buffer1(&im1);
			QByteArray im_r1;
			QByteArray im2;
			QBuffer buffer2(&im2);
			QByteArray im_r2;

			if (target->picture != NULL) {
				buffer1.open(QIODevice::WriteOnly);
				target->picture->save(&buffer1, "PNG"); // writes image into ba in PNG format

				encrypt(im1, im_r1);
				query.bindValue(":picture", im_r1);
			} else
				query.bindValue(":picture", QVariant (QVariant::ByteArray));

			if (target->pictureBack != NULL) {
				buffer2.open(QIODevice::WriteOnly);
				target->pictureBack->save(&buffer2, "PNG"); // writes image into ba in PNG format

				QByteArray im_r2;
				encrypt(im2, im_r2);
				query.bindValue(":pictureBack", im_r2);
			} else
				query.bindValue(":pictureBack", QVariant (QVariant::ByteArray));
						//qDebug() << "step 7" ;

			ret = query.exec();
		}
	}
	return ret;
}

QList<Target*> DatabaseManager::getTargets(const QString * filter)
{
	QList<Target*> result;

	QSqlQuery query (db);
	bool ret;
	if (filter != NULL) {// && filter.length() > 0) {
		ret = query.prepare("select id, name, type, groupName, creationdate, modificationdate from targets where coalesce(groupName,'') = :groupName");
		query.bindValue(":groupName", *filter);
	}
	else {
		ret = query.prepare("select id, name, type, groupName, creationdate, modificationdate from targets");
	}

	ret = query.exec ();

	while (query.next()) {
		Target * item = new Target();
		item->id = query.value(0).toInt();
		decrypt(query.value(1).toByteArray(), item->name);
		item->type = query.value(2).toString();
		item->groupName = query.value(3).toString();
		item->pictureSmall = NULL;
		item->picture = NULL;
		item->pictureBack = NULL;
		item->creationdate = query.value(4).toString();
		item->modificationdate = query.value(5).toString();
		result.append(item);
	}

	return result;
}

QList<QString> DatabaseManager::getGroups()
{
	QList<QString> result;

	//QSqlQuery query("select distinct coalesce(groupName,'') from targets where groupName != '' and groupName is Not Null order by groupName", db);
	QSqlQuery query("select distinct coalesce(groupName,'') from targets order by coalesce(groupName,'')", db);
	while (query.next()) {
		QString name = query.value(0).toString();

		//qDebug () << "ADDING GROUP " << name;

		result.append(name);
	}

	return result;
}

bool DatabaseManager::deleteTarget(Target * target)
{
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);

		ret = query.prepare("delete from fields where targetId=:code");
		query.bindValue(":code", target->id);
		ret = query.exec();

		if (ret) {
			ret = query.prepare("delete from targets where id=:code");
			if (ret) {
				//qDebug () << "DELETE " << target->id;

				query.bindValue(":code", target->id);
				ret = query.exec();
			}
		}
	}

	return ret;
}

bool DatabaseManager::updateFields(Target* target)
{
	bool ret = false;
	if (db.isOpen()) {

		QSqlQuery query (db);
		ret = query.prepare("delete from fields where targetId=:code");
		query.bindValue(":code", target->id);
		ret = query.exec();

		if (ret)
		for (int i = 0; i < target->fields.count(); i++) {
			TargetField * field = target->fields[i];
	
//			if (field->id == -1) {
				field->targetId = target->id;
				field->id = 0;
				ret &= insertTargetField (field);
//			}
//			else {
//				ret |= updateTargetField (field);
//			}
		}
	}
	
	return ret;
}

bool DatabaseManager::copyFields(Target* target)
{
	bool ret = true;
	for (int i = 0; i < target->fields.count(); i++) {
		TargetField * field = target->fields[i];
		ret &= copyTargetField (field);
	}

	return ret;
}

bool DatabaseManager::updateTarget(Target* target)
{
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);
		ret = query.prepare("UPDATE targets set name=:name, type=:type, groupName=:groupName, modificationdate=CURRENT_TIMESTAMP where id=:code");
		QByteArray t1;
		if (ret) {
			encrypt(target->name, t1);
			query.bindValue(":name", t1);
			query.bindValue(":type", target->type);
			query.bindValue(":groupName", target->groupName);
			query.bindValue(":code", target->id);
			ret = query.exec();

			//qDebug () << "updated " << target->id;

		}
	}
	return ret;
}

bool DatabaseManager::updateTargetPicture(Target* target)
{
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);
		ret = query.prepare("UPDATE targets set picture=:picture, modificationdate=CURRENT_TIMESTAMP where id=:code");
		if (ret) {
			//qDebug () << "updating image " << target->id << " " << target->picture;

			QByteArray im1;
			QBuffer buffer1(&im1);
			QByteArray im_r1;

			if (target->picture != NULL) {
				buffer1.open(QIODevice::WriteOnly);
				target->picture->save(&buffer1, "PNG"); // writes image into ba in PNG format

				encrypt(im1, im_r1);
				query.bindValue(":picture", im_r1);
			}
			else {
				//qDebug () << "delete image ";

				query.bindValue(":picture", QVariant (QVariant::ByteArray));
			}

			query.bindValue(":code", target->id);
			ret = query.exec();

			//qDebug () << "update picturde " << target->id;

		}
	}
	return ret;
}

bool DatabaseManager::updateTargetPictureBack(Target* target)
{
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);
		ret = query.prepare("UPDATE targets set pictureBack=:pictureBack, modificationdate=CURRENT_TIMESTAMP where id=:code");
		if (ret) {
			//qDebug () << "UPDATE " << target->id << " " << target->name << " " << target->type;
			QByteArray im2;
			QBuffer buffer2(&im2);
			QByteArray im_r2;

			if (target->pictureBack != NULL) {
				buffer2.open(QIODevice::WriteOnly);
				target->pictureBack->save(&buffer2, "PNG"); // writes image into ba in PNG format

				QByteArray im_r2;
				encrypt(im2, im_r2);
				query.bindValue(":pictureBack", im_r2);
			}
			else {
				query.bindValue(":pictureBack", QVariant (QVariant::ByteArray));
			}

			query.bindValue(":code", target->id);
			ret = query.exec();
			
			//qDebug () << "update picture back" << target->id;

		}
	}
	return ret;
}

bool DatabaseManager::updateTargetField(TargetField* field)
{
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);
		ret = query.prepare("UPDATE fields set name=:name, value=:value, type=:type, orderN=:order, "
		                     "encrypted=:encrypted where targetId=:targetId and id=:id");
		if (ret) {
			query.bindValue(":targetId", field->targetId);
			query.bindValue(":id", field->id);

			if (field->encrypted) {
				QByteArray a, b;
				encrypt(field->name, a);
				encrypt(field->value, b);
				query.bindValue(":name", a);
				query.bindValue(":value", b);
			}
			else {
				query.bindValue(":name", field->name);
				query.bindValue(":value", field->value);
			}
			query.bindValue(":type", field->type);
			query.bindValue(":order", field->order);
			query.bindValue(":encrypted", field->encrypted ? 1 : 0);
			ret = query.exec();
		}
	}
	return ret;
}

bool DatabaseManager::insertTargetField(TargetField* field)
{
	bool ret = false;
	if (db.isOpen()) {
		//field->id = nextId(); // We demostrates autoincrement in this case

		// http://www.sqlite.org/autoinc.html
		// NULL = is the keyword for the autoincrement to generate next value

		QSqlQuery query (db);
		ret = query.prepare("INSERT INTO fields (targetId, name, value, type, orderN, encrypted) "
		                    "VALUES (:targetId, :name, :value, :type, :order, :encrypted)");
		if (ret) {
			query.bindValue(":targetId", field->targetId);

			if (field->encrypted) {
				QByteArray a, b;
				encrypt(field->name, a);
				encrypt(field->value, b);
				query.bindValue(":name", a);
				query.bindValue(":value", b);
			}
			else {
				query.bindValue(":name", field->name);
				query.bindValue(":value", field->value);
			}
			query.bindValue(":type", field->type);
			query.bindValue(":order", field->order);
			query.bindValue(":encrypted", field->encrypted ? 1 : 0);
			ret = query.exec();
		}

		// Get database given autoincrement value
		if (ret) {
			// http://www.sqlite.org/c3ref/last_insert_rowid.html
			field->id = query.lastInsertId().toInt();
		}
	}
	return ret;
}

bool DatabaseManager::copyTargetField(TargetField* field)
{
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);
		ret = query.prepare("INSERT INTO fields (id, targetId, name, value, type, orderN, encrypted) "
		                    "VALUES (:id, :targetId, :name, :value, :type, :order, :encrypted)");
		if (ret) {
			query.bindValue(":id", field->id);
			query.bindValue(":targetId", field->targetId);

			if (field->encrypted) {
				QByteArray a, b;
				encrypt(field->name, a);
				encrypt(field->value, b);
				query.bindValue(":name", a);
				query.bindValue(":value", b);
			}
			else {
				query.bindValue(":name", field->name);
				query.bindValue(":value", field->value);
			}
			query.bindValue(":type", field->type);
			query.bindValue(":order", field->order);
			query.bindValue(":encrypted", field->encrypted ? 1 : 0);
			ret = query.exec();
		}

	}
	return ret;
}

void DatabaseManager::getFields(Target * target)
{
	if (target->listLoaded) return;

	//qDebug () << "get Fields for " << target->id;

	//target->freeFieldList ();
	
	bool ret = false;
	QSqlQuery query (db);
	ret = query.prepare("select id, name, value, type, targetId, orderN, encrypted from fields where targetId=:targetId order by orderN");
	query.bindValue(":targetId", target->id);
	ret = query.exec ();
	while (query.next()) {
		TargetField* item = new TargetField();
		item->id = query.value(0).toInt();
		item->type= query.value(3).toInt();
		item->targetId = query.value(4).toInt ();
		item->order = query.value(5).toInt ();
		item->encrypted = query.value(6).toInt () != 0;
		if (item->encrypted) {
			decrypt(query.value(1).toByteArray(), item->name);
			decrypt(query.value(2).toByteArray(), item->value);
		}
		else {
			item->name = query.value(1).toString();
			item->value= query.value(2).toString();
		}
		target->fields.append(item);
	}

	target->listLoaded = true;
}

QImage * DatabaseManager::getPicture (Target * target) {
	QImage * result = NULL;
	QSqlQuery query (db);
	bool ret = query.prepare("select picture from targets where id=:targetId");
	query.bindValue(":targetId", target->id);
	ret = query.exec ();
	while (query.next()) {
		if (query.value(0).isNull()) {
			result = NULL;
		}
		else {
			result = new QImage ();
			QByteArray im_s;
			decrypt(query.value(0).toByteArray(), im_s);
			if (!result->loadFromData (im_s)) {
				//qDebug () << "Error loading image " << target->id;
			} else {
				//qDebug () << "loaded image " << target->id;
			}
		}
		break;
	}

	return result;
}
QImage * DatabaseManager::getPictureBack (Target * target) {
	QImage * result = NULL;
	QSqlQuery query (db);
	bool ret = query.prepare("select pictureBack from targets where id=:targetId");
	query.bindValue(":targetId", target->id);
	ret = query.exec ();
	while (query.next()) {
		if (query.value(0).isNull()) {
			result = NULL;
		}
		else {
			result = new QImage ();
			QByteArray im_s;
			decrypt(query.value(0).toByteArray(), im_s);
			if (!result->loadFromData (im_s)) {
				//qDebug () << "Error loading back image " << target->id;
			}
			else {
				////qDebug () << "loaded back image " << target->id;
			}
		}
		break;
	}

	return result;
}

bool DatabaseManager::renameGroup (const QString & from, const QString & to) {
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);
		ret = query.prepare("UPDATE targets set groupName = :newGroupName, modificationdate=CURRENT_TIMESTAMP WHERE coalesce(groupName,'') = :oldGroupName");
		if (ret) {
			query.bindValue(":newGroupName", to);
			query.bindValue(":oldGroupName", from);
			ret = query.exec ();
		};
	}
	return ret;
}

bool DatabaseManager::deleteGroup (const QString & from) {
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query (db);
		ret = query.prepare("UPDATE targets set groupName = '', modificationdate=CURRENT_TIMESTAMP WHERE coalesce(groupName,'') = :oldGroupName");
		if (ret) {
			query.bindValue(":oldGroupName", from);
			ret = query.exec ();
		};
	}
	return ret;
}
