/*
    QtCartoonizer - A cartoon creation application
    Copyright (C) 2009  Ixonos Plc. Authors:

		Yi Zheng - yi.zheng@ixonos.com
        Jari Koski - jari.koski@ixonos.com
        Tero Ojala - tero.ojala@ixonos.com
        Terhi Öljymäki - terhi.oljymaki@ixonos.com

    QtCartoonizer is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    QtCartoonizer is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with QtCartoonizer; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.


	Part of the code reference the maemo code example which is licensed under a MIT-style license.

*/

#include <stdlib.h>
#include <string.h>
#include <QtGui>

#include <gst/gst.h>
#include <gst/interfaces/xoverlay.h>
/* Define save folder and output picture name.
 * If SAVE_FOLDER_DEFAULT is NULL, pictures are
 * written to the default tmp-directory */
#define SAVE_FOLDER_DEFAULT  	 NULL
#define PHOTO_NAME_DEFAULT	 "Picture"
#define PHOTO_NAME_SUFFIX_DEFAULT ".jpg"

/*
gst-launch v4l2camsrc device=/dev/video0 num-buffers=10 ! dspjpegenc ! filesink location=/home/user/MyDocs/fcamtest.jpg
*/

#define VIDEO_SRC "v4l2camsrc"
#define VIDEO_SINK "xvimagesink"

//#ifdef __arm__
///* The device by default supports only
// * vl4l2src for camera and xvimagesink
// * for screen */
//#define VIDEO_SRC "v4l2camsrc"
//#define VIDEO_SINK "xvimagesink"
//#else
///* These are for the X86 SDK. Xephyr doesn't
// * support XVideo extension, so the application
// * must use ximagesink. The video source depends
// * on driver of your Video4Linux device so this
// * may have to be changed */
//#define VIDEO_SRC "videotestsrc"
//#define VIDEO_SINK "ximagesink"
//#endif


///* Define sources and sinks according to
// * running environment
// * NOTE: If you want to run the application
// * in ARM scratchbox, you have to change these*/
//#ifdef __arm__
///* The device by default supports only
// * vl4l2src for camera and xvimagesink
// * for screen */
//#define VIDEO_SRC "v4l2src"
//#define VIDEO_SINK "xvimagesink"
//#else
///* tHESe are for the X86 SDK. Xephyr doesn't
// * support XVideo extension, so the application
// * must use ximagesink. The video source depends
// * on driver of your Video4Linux device so this
// * may have to be changed */
//#define VIDEO_SRC "v4lsrc"
//#define VIDEO_SINK "ximagesink"
//#endif

#include "camera.h"



/* Creates a jpeg file from the buffer's raw image data */

/* This callback will be registered to the image sink
 * after user requests a photo */
gboolean buffer_probe_callback(
		GstElement *image_sink,
		GstBuffer *buffer, GstPad *pad, QCamera *camera)
{
    unsigned char *data_photo =
	    (unsigned char *) GST_BUFFER_DATA(buffer);

	/* Create a JPEG of the data and check the status */
    camera->createJpeg (data_photo);

	/* Disconnect the handler so no more photos
	 * are taken */
    g_signal_handler_disconnect(G_OBJECT(image_sink),
			camera->bufferId());

    return TRUE;
}

gboolean buffer_watch_callback(
		GstElement *image_sink,
		GstBuffer *buffer, GstPad *pad, QCamera *camera)
{
    static int frames = 0;
    //Frame buffer ready now
    frames++;
    if (frames > 1)
    {
        camera->streamReady ();
    //disconnect
        g_signal_handler_disconnect(G_OBJECT(image_sink),
			camera->bufferId());
        frames = 0;
    }
    return TRUE;
}


void QCamera::streamReady ()
{
    if(!m_streamReady)
    {
        m_streamReady = true;
        qDebug() << "camera emit video ready";
        emit videoStreamReady(true);
    }
}

QCamera::QCamera ()
{
    //setSizePolicy (QSizePolicy::Fixed, QSizePolicy::Fixed);
    m_streamReady = false;
    m_mode = video;
    m_image = QImage (64,64,QImage::Format_Invalid);
    if(!initialize_pipeline ())
    {
        qDebug () << "gst init fail";
    };
}

QCamera::~QCamera ()
{
    destroy_pipeline ();
    qDebug() << "exit Camera";
}

void QCamera::photoTaked (QImage* image)
{
    emit photoReady (&m_image);
}


void QCamera::setXWindowId (WId winId)
{
    if ((m_videoSink ) && (GST_IS_X_OVERLAY (m_videoSink)))
    {
        gst_x_overlay_set_xwindow_id (GST_X_OVERLAY (m_videoSink), winId);
        gst_x_overlay_expose (GST_X_OVERLAY (m_videoSink));
    }
}


void QCamera::stop ()
{
    qDebug () << "camera stop";
	gst_element_set_state(m_pipeline, GST_STATE_NULL);
	m_mode = video;
}

void QCamera::play ()
{
	GstElement *image_sink;

	/* Get the image sink element from the pipeline */
	image_sink = gst_bin_get_by_name(GST_BIN(m_pipeline),
			"image_sink");
    qDebug () << "camera play emit video not ready";
    m_streamReady = false;
    emit videoStreamReady(false);
	gst_element_set_state(m_pipeline, GST_STATE_PLAYING);

	m_bufferCbId = g_signal_connect(
			G_OBJECT(image_sink), "handoff",
			G_CALLBACK(buffer_watch_callback), this);

	m_mode = photo;
}

cameraMode QCamera::mode()
{
	return m_mode;
};

void QCamera::setMode(cameraMode mode)
{
	m_mode = mode;
}
/* Set Image*/
void QCamera::createJpeg(unsigned char *data)
{
    guint height, width, bpp;

    width = 320;
    height = 240;
    bpp = 24;

    /* Create a pixbuf object from the data */
    m_image = QImage (data,
            width, height, 
            QImage::Format_RGB888).copy();
    photoTaked (&m_image);
    return ;
}

/* Callback that gets called when user clicks the "Take photo" button */
void QCamera::takePhoto ()
{
	GstElement *image_sink;

	/* Get the image sink element from the pipeline */
	image_sink = gst_bin_get_by_name(GST_BIN(m_pipeline),
			"image_sink");

	/* Connect the "handoff"-signal of the image sink to the
	 * callback. This gets called whenever the sink gets a
	 * buffer it's ready to pass forward on the pipeline */
	m_bufferCbId = g_signal_connect(
			G_OBJECT(image_sink), "handoff",
			G_CALLBACK(buffer_probe_callback), this);
	m_mode = video;
}



/* Initialize the the Gstreamer pipeline. Below is a diagram
 * of the pipeline that will be created:
 *
 *                             |Screen|  |Screen|
 *                           ->|queue |->|sink  |-> Display
 * |Camera|  |CSP   |  |Tee|/
 * |src   |->|Filter|->|   |\  |Image|   |Image |  |Image|
 *                           ->|queue|-> |filter|->|sink |-> JPEG file
 */
bool QCamera::initialize_pipeline()
{
        GstElement *camera_src, *image_sink;
	GstElement *screen_queue, *image_queue;
        GstElement *image_filter, *tee, *csp_filter;
        GstCaps *caps;
	GstBus *bus;


	/* Initialize Gstreamer */
	gst_init( NULL, NULL);

	/* Create pipeline and attach a callback to it's
	 * message bus */
	m_pipeline = gst_pipeline_new("test-camera");

	bus = gst_pipeline_get_bus(GST_PIPELINE(m_pipeline));
	//gst_bus_add_watch(bus, (GstBusFunc)bus_callback, &m_appData);
	gst_object_unref(GST_OBJECT(bus));

	/* Create elements */
	/* Camera video stream comes from a Video4Linux driver */
	camera_src = gst_element_factory_make(VIDEO_SRC, "camera_src");
	/* Colorspace filter is needed to make sure that sinks understands
	 * the stream coming from the camera */
	csp_filter = gst_element_factory_make("ffmpegcolorspace", "csp_filter");
	/* Tee that copies the stream to multiple outputs */
	tee = gst_element_factory_make("tee", "tee");
	/* Queue creates new thread for the stream */
	screen_queue = gst_element_factory_make("queue", "screen_queue");
	/* Sink that shows the image on screen. Xephyr doesn't support XVideo
	 * extension, so it needs to use ximagesink, but the device uses
	 * xvimagesink */
	m_videoSink = gst_element_factory_make(VIDEO_SINK, "screen_sink");
	/* Creates separate thread for the stream from which the image
	 * is captured */
	image_queue = gst_element_factory_make("queue", "image_queue");
	/* Filter to convert stream to use format that the gdkpixbuf library
	 * can use */
	image_filter = gst_element_factory_make("ffmpegcolorspace", "image_filter");
	/* A dummy sink for the image stream. Goes to bitheaven */
	image_sink = gst_element_factory_make("fakesink", "image_sink");

	/* Check that elements are correctly initialized */
	if(!(m_pipeline && camera_src && m_videoSink  && screen_queue && csp_filter
		&& image_queue && image_filter && image_sink))
	{
		qDebug() << "Couldn't create pipeline elements";
		return FALSE;
	}

	/* Set image sink to emit handoff-signal before throwing away
	 * it's buffer */
	g_object_set(G_OBJECT(image_sink),
			"signal-handoffs", TRUE, NULL);

	/* Add elements to the pipeline. This has to be done prior to
	 * linking them */
	gst_bin_add_many(GST_BIN(m_pipeline), camera_src, csp_filter,
			tee, screen_queue, m_videoSink, image_queue,
			image_filter, image_sink, NULL);

	/* Specify what kind of video is wanted from the camera */
	caps = gst_caps_new_simple("video/x-raw-rgb",
			"width", G_TYPE_INT, 320,
			"height", G_TYPE_INT, 240,
			NULL);


	/* Link the camera source and colorspace filter using capabilities
	 * specified */
	if(!gst_element_link_filtered(camera_src, csp_filter, caps))
	{
		return FALSE;
	}
	gst_caps_unref(caps);

	/* Connect Colorspace Filter -> Tee -> Screen Queue -> Screen Sink
	 * This finalizes the initialization of the screen-part of the pipeline */
	if(!gst_element_link_many(csp_filter, tee, screen_queue, m_videoSink, NULL))
	{
        qDebug () << "gst video sink init fail";
		return FALSE;
	}

	/* gdkpixbuf requires 8 bits per sample which is 24 bits per
	 * pixel */
	caps = gst_caps_new_simple("video/x-raw-rgb",
			"width", G_TYPE_INT, 320,
			"height", G_TYPE_INT, 240,
			"bpp", G_TYPE_INT, 24,
			"depth", G_TYPE_INT, 24,
			"framerate", GST_TYPE_FRACTION, 15, 1,
			NULL);

	/* Link the image-branch of the pipeline. The pipeline is
	 * ready after this */
	if(!gst_element_link_many(tee, image_queue, image_filter, NULL))
    {
        qDebug () << "gst tee init fail";
        return FALSE;
    }
	if(!gst_element_link_filtered(image_filter, image_sink, caps))
    {

        qDebug () << "gst filterinit fail";
        return FALSE;
    }

	gst_caps_unref(caps);

	gst_element_set_state(m_pipeline, GST_STATE_NULL);
    qDebug () << "gst init ok";
	return TRUE;
}

/* Destroy the pipeline on exit */
void QCamera::destroy_pipeline ()
{
	/* Free the pipeline. This automatically also unrefs all elements
	 * added to the pipeline */
    gst_x_overlay_set_xwindow_id (GST_X_OVERLAY (m_videoSink), 0);
	gst_element_set_state (m_pipeline, GST_STATE_NULL);
	gst_object_unref(GST_OBJECT (m_pipeline));
}
