/***************************************************************************
 *   Copyright (C) 2010 by Ixonos Plc   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; version 2 of the License.               *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef _SERVERDB_HPP_
#define _SERVERDB_HPP_

#include <QDebug>
#include <QSettings>
#include <QtSql>
#include <QDir>

#include <qtrapids/settings.hpp>

namespace qtrapids
{

class ServerSettings
{

public:
	ServerSettings(QSettings &settings)
			: settings_(settings) { }

	~ServerSettings() { }

	QString getDbEngine() const {
		return getParamAndStore("db_engine", getDefaultDbEngine()).toString();
	}

	QString getDbName() const {
		QString default_db_path(QDir::home().filePath(appName() + ".sqlite"));
		return getParamAndStore("db", default_db_path).toString();
	}

	QString getTorrentsDir() const {
		QString default_dir(QDir::home().filePath(getTorrentsSubDir()));
		return getParamAndStore("torrents_dir", default_dir).toString();

	}

	static QString getTorrentsSubDir() {
		return QString(".") + appName();
	}

	QString getDownloadDir() const {
		QString default_dir(QDir::home().absolutePath());
		QString v = getParamAndStore("download/directory", default_dir).toString();
		if (!v.isEmpty()) {
			return v;
		} else {
			settings_.setValue("download/directory", default_dir);
			return default_dir;
		}
	}

	ports_range_t getListenPorts() const {

		ports_range_t default_ports(13131, 13132);
		default_ports.first = getParamAndStore("net/listen_range_begin", default_ports.first).toUInt();
		default_ports.second = getParamAndStore("net/listen_range_end", default_ports.second).toUInt();
		return default_ports;
	}

	/**
	   @todo deztructor: there is no check for option type yet
	 */
	void setOptions(ParamsMapConst_t &options) {
		for (ParamsMapConstIterator_t p = options.constBegin(); p != options.constEnd(); ++p) {
			settings_.setValue(p.key(), p.value());
		}
	}

	ParamsMap_t getOptions() const {
		ParamsMap_t options;
		QStringList keys = settings_.allKeys();
		for (QStringList::const_iterator p = keys.begin(); p != keys.end(); ++p) {
			options[*p] = settings_.value(*p).toString();
		}
		return options;
	}

private:

	ServerSettings(ServerSettings const&);
	ServerSettings& operator= (ServerSettings const&);

	static inline QString appName() {
		return QCoreApplication::applicationName();
	}

	static QString getDefaultDbEngine() {
		// for (QStringListIterator p = QSqlDatabase::drivers(); p.hasNext();) {
		//     return p.next();
		// }
		return "QSQLITE";
	}

	QVariant getParamAndStore(QString const& name, QVariant default_value) const {
		return GetSettingsStoreDefault(settings_, name, default_value);
	}

	mutable QSettings &settings_;
};

namespace {

class DbAccessor
{
public:
	DbAccessor(QSqlDatabase &db)
			: db_(db) {
		if (!db_.open()) {
			qDebug() << "cant open db";
		}
	}

	~DbAccessor() {
		db_.close();
	}

private:
	QSqlDatabase &db_;
};

}

class ServerDb
{

public:
	ServerDb(ServerSettings *settings)
			: db_(QSqlDatabase::addDatabase(settings->getDbEngine())) {
		QString db_name(settings->getDbName());
		db_.setDatabaseName(db_name);

		qDebug() << "opening db " << db_name;
		if (!db_.open()) {
			qDebug() << "cant open db";
			return;
		}
		qDebug() << "opened " << db_name;

		QSqlQuery q;
		if (!q.exec("create table torrents (hash varchar primary key, path varchar, savepath varchar);\n")) {
			qDebug() << "cant create table: " << q.lastError().text();
		}
	}

	~ServerDb() {
		db_.close();
	}

	void addTorrent(const QString &hash, const QString &path, const QString &save_path) {
		DbAccessor dba(db_);
		QSqlQuery query_add_;
		query_add_.prepare("INSERT INTO torrents (hash, path, savepath) VALUES (?, ?, ?)");
		query_add_.bindValue(0, hash);
		query_add_.bindValue(1, path);
		query_add_.bindValue(2, save_path);
		if (!query_add_.exec()) {
			qDebug() << "cant add torrent info into db: "
			<< query_add_.lastError().text();
		}
	}

	void removeTorrent(const QString &hash) {
		DbAccessor dba(db_);
		QSqlQuery query(QString("DELETE FROM torrents WHERE hash='") + hash + "'");
		if (!query.exec()) {
			qDebug() << "cant delete torrent info from db"
			<< query.lastError().text();
		}
	}

private:

	ServerDb(ServerDb const&);
	ServerDb& operator= (ServerDb const&);

	QSqlDatabase db_;
};

class TorrentsStorage
{
public:

	TorrentsStorage(ServerDb &db)
			: torrents_("SELECT hash, path, savepath from torrents") { }


	bool nextTorrent(TorrentDownloadInfo &info) {
		if (!torrents_.next()) {
			return false;
		}
		info.hash = torrents_.value(0).toString();
		info.path = torrents_.value(1).toString();
		info.download_path = torrents_.value(2).toString();
		return true;
	}


private:
	QSqlQuery torrents_;
};

} // namespace qtrapids

#endif // _SERVERDB_HPP_
