#include <QtGui/QApplication>
#include <QtCore/QTranslator>
#include <QtCore/QLocale>
#include <QtCore/QObject>
#include <QtGui/QMainWindow>
#include <QtGui/QMessageBox>
#include <QtCore/QString>
#include <QtCore/QStringList>
#include <QtCore/QStringListIterator>
#include <QtGui/QCloseEvent>
#include <QtCore/QProcess>
#include <QtCore/QMap>
#include <QtCore/QDir>
#include <QtCore/QPluginLoader>
#include <QtGui/QAction>
#include <QtCore/QList>
#include <QtCore/QListIterator>
#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusMessage>
#include <QtCore/QTimer>
#include "apnhelper.h"
#include "mobilehotspotplugin.h"
#include "dbushelper.h"
#include "hotspothelper.h"
#include "accesspointgui.h"
#include "internetaccessgui.h"
#include "mobilehotspotgui.h"


MobileHotspotGUI::MobileHotspotGUI(QWidget *parent)
    : QMainWindow(parent)
{
	loadPlugins();
	sortPlugins();
	ui.setupUi(this);
	displayPlugins();
	configuration.load();
	updateDisplayedConfiguration();
}

MobileHotspotGUI::~MobileHotspotGUI()
{

}

void MobileHotspotGUI::closeEvent(QCloseEvent* event){
	if(ui.btnStart->isEnabled())
		event->accept();
	else{
		if(QMessageBox::question(this, tr("Closing the program"), tr("Hotspot is still active ! Do you want to stop it ?"), QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel) == QMessageBox::Yes){
			stop();
			event->accept();
		}
		else
			event->ignore();
	}

}

void MobileHotspotGUI::loadPlugins(){
	QDir dir("./plugins");
	QStringList list = dir.entryList(QDir::Files | QDir::NoSymLinks);
	QStringListIterator iterator(list);
	while(iterator.hasNext()){
		QString filename = iterator.next();
		QString path = dir.absoluteFilePath(filename);
		QPluginLoader loader(path);
		QObject *loaded = loader.instance();
		if(loaded != 0){
			MobileHotspotPlugin *plugin = qobject_cast<MobileHotspotPlugin*>(loaded);
			if(plugin != 0){
				QString name = plugin->name();
				if(plugins.contains(name))
					qDebug( (QString("WARNING : plugin name ") + name + " is already taken").toAscii().data());
				else{
					plugins.insert(name, plugin);
					QString lang = (QFileInfo(dir, filename).completeBaseName() + "-") + QLocale::system().name();
					QTranslator *translator = new QTranslator();
					translator->load(lang, "./lang");
					plugin->setTranslator(translator);
					qDebug( (QString("Loaded plugin ") + name).toAscii().data() );
				}
			}
		}
		else
			qDebug(loader.errorString().toAscii().data());
	}
	qDebug( (QString("Loaded ") + QString::number(plugins.size()) + " plugins").toAscii().data() );
}

void MobileHotspotGUI::sortPlugins(){
	sortedPlugins.clear();
	QList<QString> keys = plugins.keys();
	QListIterator<QString> iterator(keys);
	while(iterator.hasNext()){
		QString key = iterator.next();
		int order = plugins.value(key)->order();
		int insertAt = sortedPlugins.size();
		for(int i = 0; i < sortedPlugins.size(); i++){
			QString testedKey = sortedPlugins.at(i);
			int testedOrder = plugins.value(testedKey)->order();
			if(order < testedOrder){
				insertAt = i;
				i = sortedPlugins.size();
			}
		}
		sortedPlugins.insert(insertAt, key);
	}
}

void MobileHotspotGUI::displayPlugins(){
	QList<MobileHotspotPlugin*> values = plugins.values();
	QListIterator<MobileHotspotPlugin*> iterator(values);
	while(iterator.hasNext()){
		MobileHotspotPlugin *plugin = iterator.next();
		if(plugin->displayName()){
			qApp->installTranslator(plugin->translator());
				QAction *action = ui.menuPlugins->addAction(plugin->displayedName());
			qApp->removeTranslator(plugin->translator());
			action->setData(plugin->name());
			QObject::connect(action, SIGNAL(triggered()), this, SLOT(pluginTriggered()));
		}
	}
}

void MobileHotspotGUI::pluginTriggered(){
	QObject *sender = const_cast<QObject*>(QObject::sender());
	QAction *action = static_cast<QAction*>(sender);
	QString name = action->data().toString();
	if(! plugins.contains(name)){
			qDebug("Weird error, plugin action linked to no plugin ?");
			return;
	}
	MobileHotspotPlugin *plugin = plugins.value(name);
	qDebug( (QString("Plugin configure : ") + name).toAscii().data() );
	qApp->installTranslator(plugin->translator());
		plugin->configure(configuration, ui.btnStop->isEnabled());
	qApp->removeTranslator(plugin->translator());
}

void MobileHotspotGUI::about(){
	QString message("== "); message += QApplication::applicationName(); message += " ==\n";
	message += tr("Developped by"); message += " Loic Burtin (dest)\n";
	message += tr("Some elements are inspired from MobileHotSpot by"); message += " Eero af Heurlin (Rambo)\n\n";
	message += tr("This program allows your N900 to be used as a hotspot."); message += " ";
	message += tr("Share your data connection with your other devices !");
	QMessageBox::information(this, tr("About this program..."), message);
}

void MobileHotspotGUI::updateDisplayedConfiguration(){
	ui.lstAccessPoint->clear();
	ui.lstAccessPoint->addItem(configuration.hotspotName);
	ui.lstAccessPoint->addItem(configuration.encryptionType == configuration.ENCRYPTION_WEP ? tr("WEP") : tr("No encryption"));
	if(configuration.disablePowerSaving)
		ui.lstAccessPoint->addItem(tr("Disable power-saving"));

	ui.lstInternetAccess->clear();
	QString s = configuration.internetAPName;
	if(s != "" && configuration.internetInterface() == configuration.INTERFACE_GPRS){
		if(configuration.enforcement2g3g != IGNORE){
			s += " (";
			if(configuration.enforcement2g3g == ENFORCE_2G)
				s += tr("2G");
			else if(configuration.enforcement2g3g == ENFORCE_3G)
				s += tr("3G");
			else
				s += tr("2G/3G");
			s += ")";
		}
	}
	ui.lstInternetAccess->addItem(s);
	ui.lstInternetAccess->addItem(configuration.internetEnabled ? tr("With internet") : tr("Without internet"));
	ui.lstInternetAccess->addItem(configuration.lanNetwork + "0");

	ui.lstInterfaces->clear();
	ui.lstInterfaces->addItem(tr("Wifi"), configuration.INTERFACE_WLAN);
	ui.lstInterfaces->addItem(tr("USB"), configuration.INTERFACE_USB);
	ui.lstInterfaces->setCurrentIndex(configuration.lanInterface == configuration.INTERFACE_WLAN ? 0 : 1);
}

void MobileHotspotGUI::configureInternetAccess(){
	/** Find available networks **/
	//TODO
	/** For now we just call doConfigureInternetAccess() **/
	doConfigureInternetAccess();
}

void MobileHotspotGUI::availableAPNsSignal(const QDBusMessage &message){
	//TODO
}

void MobileHotspotGUI::availableAPNsEnableSignal(){
	//TODO
}

void MobileHotspotGUI::availableAPNsDisableSignal(){
	//TODO
}

void MobileHotspotGUI::availableAPNsRequest(){
	//TODO
}

void MobileHotspotGUI::doConfigureInternetAccess(){
	InternetAccessGUI gui(this);
	gui.setConfiguration(configuration);
	if(gui.exec() != QDialog::Accepted)
		return;

	gui.configuration(&configuration);
	configuration.save();
	updateDisplayedConfiguration();
}

void MobileHotspotGUI::configureAccessPoint(){
	AccessPointGUI gui(this);
	gui.setConfiguration(configuration);
	if(gui.exec() != QDialog::Accepted)
		return;

	gui.configuration(&configuration);
	configuration.save();
	updateDisplayedConfiguration();
}

void MobileHotspotGUI::configureInterface(int index){
	configuration.lanInterface = ui.lstInterfaces->itemData(index).toString();
	if(configuration.lanInterface == configuration.INTERFACE_WLAN && configuration.internetAPType != GCONF_APN_TYPE_GPRS){
		configuration.internetAPID = "";
		configuration.internetAPType = "";
		configuration.internetAPName = "";
	}
	configuration.save();
	updateDisplayedConfiguration();
}

void MobileHotspotGUI::start(bool notification){
	int ret;
	/** Find previous enforcement **/
	if(configuration.internetEnabled && configuration.enforcement2g3g != IGNORE){
		ret = DBUSHelper::internetEnforcement2G3G(&previousEnforcement);
		if(ret != 0){
			qDebug("WARNING : Cannot get the current level of 2G/3G enforcement, using default value");
			previousEnforcement = ENFORCEMENT_NONE;
		}
	}

	/** Find previous connection **/
	previousConnectionID = "";
	previousConnectionType = "";
	waitingForPreviousConnection = true;
	previousConnectionEnableSignal();
	previousConnectionRequest();

	/** Start the hotspot a bit later (hopefully the previous connection will be known) **/
	showNotifications = notification;
	QTimer::singleShot(800, this, SLOT(startProcedure()));
}

void MobileHotspotGUI::startProcedure(){
	/* Credits to Rambo for the procedure */
	bool success = true;
	int ret;
	qDebug("*** STARTING PROCESS ***");

	/** Disable previous connection search **/
	waitingForPreviousConnection = false;
	previousConnectionDisableSignal();

	/** Obvious sanity checks **/
	if(configuration.lanInterface == configuration.INTERFACE_WLAN && configuration.internetInterface() == configuration.INTERFACE_WLAN){
		/** Both LAN and INET are on WLAN, that is not possible **/
		QString fail = tr("Error : LAN and Internet are both using the Wifi interface, please use a LAN on USB");
		QMessageBox::critical(this, tr("Invalid parameters"), fail);
		return;
	}

	/** Save scan interval & powersave (if there is one) **/
	scanInterval = APNHelper::scanInterval();
	apnPowerSave = APNHelper::apnPowerSave(configuration.internetAPID);

	/** Notify plugins **/
	QListIterator<QString> pluginsIterator = QListIterator<QString>(sortedPlugins);
	while(pluginsIterator.hasNext()){
		MobileHotspotPlugin *plugin = plugins.value(pluginsIterator.next());
		qDebug( (QString("Plugin beforeStarting : ") + plugin->name()).toAscii().data() );
		qApp->installTranslator(plugin->translator());
			plugin->beforeStarting(configuration);
		qApp->removeTranslator(plugin->translator());
	}

	try{
		/** Load system modules if necessary **/
		alreadyLoadedModules.clear();
		ret = HotspotHelper::loadSystemModules(&alreadyLoadedModules);
		if(ret != 0)
			throw QString(tr("Error while loading system modules, are you running the power kernel ?"));

		/** Internet connection handling **/
		if(configuration.internetEnabled){

			if(previousConnectionType == configuration.internetAPType && previousConnectionID == configuration.internetAPID && configuration.internetInterface() != configuration.INTERFACE_WLAN)
				/** Nothing to do to the internet connection **/
				/** In case of USB-Lan and Wifi-Inet we have to disconnect, change powersave & reconnect to prevent route disappearing problem **/
				qDebug("** Internet connection is already good, skipping this step **");
			else{
				/** Disable current internet connection **/
				ret = HotspotHelper::disableInternetConnection(configuration);
				if(ret != 0)
					throw QString(tr("Error while disconnecting previous internet connection"));

				/** USB-Lan Wifi-Inet **/
				if(configuration.internetInterface() == configuration.INTERFACE_WLAN){
					/** Change powersave if needed **/
					if(configuration.disablePowerSaving)
						APNHelper::setAPNPowerSave(configuration.internetAPID, GCONF_APN_POWER_NOECO);
				}

				/** Connect to the chosen APN **/
				ret = HotspotHelper::enableInternetConnection(configuration);
				if(ret != 0)
					throw QString(tr("Error while connecting to the chosen APN"));
			}
		}
		else if(configuration.lanInterface == configuration.INTERFACE_WLAN && previousConnectionType.indexOf("WLAN") >= 0){
			/** WLAN must be disabled in order to get the adhoc WLAN working **/
			ret = HotspotHelper::disableInternetConnection(configuration);
			if(ret != 0)
				throw QString(tr("Error while disconnecting previous internet connection"));
		}
		else
			qDebug("** User requested no internet connection, ignoring it then **");

		/** Disable network automatic search **/
		APNHelper::setScanInterval(0);

		/** USB vs Wifi **/
		if(configuration.lanInterface == configuration.INTERFACE_USB){

			/** USB mode **/
			ret = HotspotHelper::checkUSBMode();
			if(ret != 0)
				throw QString(tr("Error while checking USB mode, are you in PC Suite mode ?"));

			/** Prepare USB interface **/
			ret = HotspotHelper::prepareUSBInterface(configuration);
			if(ret != 0)
				throw QString(tr("Error while preparing the USB interface"));

		}
		else{

			/** Prepare Wifi interface **/
			ret = HotspotHelper::prepareWifiInterface(configuration);
			if(ret != 0)
				throw QString(tr("Error while preparing the Wifi interface"));
		}

		/** Stop dnsmasq - may fail, not fatal unless timeout **/
		ret = HotspotHelper::stopDnsMasq(configuration);
		if(ret == -2)
			throw QString(tr("Error (timeout) while stopping the DHCP/DNS server"));
		else if(ret != 0)
			qDebug("WARNING : Could not stop dnsmasq (maybe it was not running - its fine)");

		/** Start dnsmasq **/
		ret = HotspotHelper::startDnsMasq(configuration, &dnsmasqProcess);
		if(ret != 0)
			throw QString("Error while starting dnsmasq");

		/** Set iptables **/
		ret = HotspotHelper::setIPTables(configuration, &ipforward);
		if(ret != 0)
			throw QString("Error while setting iptables");


		// There :) Finished
	}
	catch(QString error){
		qDebug(error.toAscii().data());
		success= false;
		if(showNotifications)
			QMessageBox::critical(this, tr("Failed to start the hotspot"), error);
	}

	/** Notify plugins **/
	successStarting = success;
	pluginsIterator = QListIterator<QString>(sortedPlugins);
	while(pluginsIterator.hasNext()){
		MobileHotspotPlugin *plugin = plugins.value(pluginsIterator.next());
		qDebug( (QString("Plugin afterStarting : ") + plugin->name()).toAscii().data() );
		qApp->installTranslator(plugin->translator());
			plugin->afterStarting(configuration, successStarting);
		qApp->removeTranslator(plugin->translator());
	}

	/** Enforcement 2G/3G **/
	if(configuration.internetEnabled && configuration.enforcement2g3g != IGNORE && configuration.internetInterface() == configuration.INTERFACE_GPRS){
		qDebug("Changing 2g/3g enforcement");
		ret = 0;
		if(configuration.enforcement2g3g == ENFORCE_2G && previousEnforcement != ENFORCEMENT_2G)
			ret = DBUSHelper::setInternetEnforcement2G3G(ENFORCEMENT_2G);
		else if(configuration.enforcement2g3g == ENFORCE_3G && previousEnforcement != ENFORCEMENT_3G)
			ret = DBUSHelper::setInternetEnforcement2G3G(ENFORCEMENT_3G);
		else if(previousEnforcement != ENFORCEMENT_NONE)
			ret = DBUSHelper::setInternetEnforcement2G3G(ENFORCEMENT_NONE);
		if(ret != 0)
			qDebug("WARNING : Cannot change 2g/3g enforcement");
	}

	/** GUI **/
	if(success){
		if(showNotifications)
			DBUSHelper::notificationSimple(tr("Hotspot started successfully"));
		ui.btnAccessPoint->setEnabled(false);
		ui.btnInternetAccess->setEnabled(false);
		ui.lstInterfaces->setEnabled(false);
		ui.btnStart->setEnabled(false);
		ui.btnStop->setEnabled(true);
		qDebug("*** STARTING PROCESS TERMINATED SUCCESSFULLY ***");
	}
	else{
		qDebug("*** STARTING PROCESS FAILED ***");
		stop(false);
	}
}

void MobileHotspotGUI::stop(bool notification){
	/** Nothing to do here, its simply for symetry with start() **/
	showNotifications = notification;
	stopProcedure();
}

void MobileHotspotGUI::stopProcedure(){
	/* Credits to Rambo for the procedure */
	bool success = true;
	int ret;
	qDebug("*** STOPPING PROCESS ***");

	/** Notify plugins **/
	QListIterator<QString> pluginsIterator = QListIterator<QString>(sortedPlugins);
	pluginsIterator.toBack();
	while(pluginsIterator.hasPrevious()){
		MobileHotspotPlugin *plugin = plugins.value(pluginsIterator.previous());
		qDebug( (QString("Plugin beforeStopping : ") + plugin->name()).toAscii().data() );
		qApp->installTranslator(plugin->translator());
			plugin->beforeStopping(configuration, successStarting);
		qApp->removeTranslator(plugin->translator());
	}

	/** Unset iptables **/
	ret = HotspotHelper::unsetIPTables(configuration, ipforward);
	if(ret != 0){
		qDebug("Error while unsetting iptables");
		success = false;
	}

	/** Stop dnsmasq **/
	ret = HotspotHelper::stopDnsMasq(configuration);
	if(ret != 0){
		qDebug("Error while stopping dnsmasq");
		success = false;
	}

	/** USB vs Wifi **/
	if(configuration.lanInterface == configuration.INTERFACE_USB){

		/** Unprepare USB interface **/
		ret = HotspotHelper::unprepareUSBInterface(configuration);
		if(ret != 0){
			qDebug("Error while unpreparing USB interface");
			success = false;
		}
	}
	else{

		/** Unprepare Wifi interface **/
		ret = HotspotHelper::unprepareWifiInterface(configuration);
		if(ret != 0){
			qDebug("Error while unpreparing Wifi interface");
			success = false;
		}
	}

	/** Restore scan interval **/
	APNHelper::setScanInterval(scanInterval);

	/** Internet connection handling **/
	if(configuration.internetEnabled){

		if(previousConnectionType == configuration.internetAPType && previousConnectionID == configuration.internetAPID && configuration.internetInterface() != configuration.INTERFACE_WLAN)
			/** Current connection == previous connection **/
			/** In case of USB-Lan and Wifi-Inet we have to disconnect, change powersave & reconnect **/
			qDebug("** Previous connection was the same, nothing to do to the internet connection **");
		else{
			/** Disable current internet connection **/
			ret = HotspotHelper::disableInternetConnection(configuration);
			if(ret != 0){
				qDebug("Error while disconnecting internet connection");
				success = false;
			}

			/** USB-Lan Wifi-Inet **/
			if(configuration.internetInterface() == configuration.INTERFACE_WLAN){
				/** Change powersave if needed **/
				if(configuration.disablePowerSaving)
					APNHelper::setAPNPowerSave(configuration.internetAPID, apnPowerSave);
			}

			if(previousConnectionID != ""){
				/** Restore previous internet connection **/
				qDebug("** Restoring previous internet connection **");
				ret = DBUSHelper::internetConnect(previousConnectionID);
				if(ret != 0){
					qDebug("Error while reconnecting previous internet connection");
					success = false;
				}
			}
			else
				qDebug("** There was no previous connection, so no reconnection to do **");
		}
	}
	else if(configuration.lanInterface == configuration.INTERFACE_WLAN && previousConnectionType.indexOf("WLAN") >= 0){
		/** We need to restore disabled WLAN **/
		ret = DBUSHelper::internetConnect(previousConnectionID);
		if(ret != 0){
			qDebug("Error while reconnecting previous internet connection");
			success = false;
		}
	}
	else
		qDebug("** User requested no internet connection, ignoring it then **");

	/** Unload system modules if we loaded them ourselves **/
	ret = HotspotHelper::unloadSystemModules(alreadyLoadedModules);
	if(ret != 0){
		qDebug("Error while unloading system modules");
		success = false;
	}


	// There :) Finished

	/** Notify plugins **/
	pluginsIterator = QListIterator<QString>(sortedPlugins);
	pluginsIterator.toBack();
	while(pluginsIterator.hasPrevious()){
		MobileHotspotPlugin *plugin = plugins.value(pluginsIterator.previous());
		qDebug( (QString("Plugin afterStopping : ") + plugin->name()).toAscii().data() );
		qApp->installTranslator(plugin->translator());
			plugin->afterStopping(configuration, successStarting, success);
		qApp->removeTranslator(plugin->translator());
	}

	/** Enforcement 2G/3G **/
	if(configuration.internetEnabled && configuration.enforcement2g3g != IGNORE && configuration.internetInterface() == configuration.INTERFACE_GPRS){
		qDebug("Restoring original 2g/3g enforcement");
		ret = DBUSHelper::setInternetEnforcement2G3G(previousEnforcement);
		if(ret != 0)
			qDebug("WARNING : Cannot restore previous 2g/3g enforcement");
	}

	/** GUI **/
	if(showNotifications)
		DBUSHelper::notificationSimple(success ? tr("Hotspot stopped successfully") : tr("Hotspot stopped with errors"));
	ui.btnAccessPoint->setEnabled(true);
	ui.btnInternetAccess->setEnabled(true);
	ui.lstInterfaces->setEnabled(true);
	ui.btnStart->setEnabled(true);
	ui.btnStop->setEnabled(false);

	updateDisplayedConfiguration();
	qDebug("*** STOPPING PROCESS FINISHED ***");
}

void MobileHotspotGUI::previousConnectionSignal(const QDBusMessage &message){
	if(! waitingForPreviousConnection)
		return;
	QList<QVariant> args = message.arguments();
	if(args.size() != 8)
		return;
	QString errorMessage = args.at(6).toString();
	if(errorMessage != ""){
		qDebug((QString("Error while testing previous connection : ") + errorMessage).toAscii().data());
		return;
	}
	previousConnectionID = QString(args.at(5).toByteArray());
	previousConnectionType = args.at(3).toString();
}

void MobileHotspotGUI::previousConnectionEnableSignal(){
	QDBusConnection dbus = QDBusConnection::systemBus();
	dbus.connect("com.nokia.icd2", "/com/nokia/icd2", "com.nokia.icd2", "state_sig", this, SLOT(previousConnectionSignal(const QDBusMessage&)));
}

void MobileHotspotGUI::previousConnectionDisableSignal(){
	QDBusConnection dbus = QDBusConnection::systemBus();
	dbus.disconnect("com.nokia.icd2", "/com/nokia/icd2", "com.nokia.icd2", "state_sig", this, SLOT(previousConnectionSignal(const QDBusMessage&)));
}

void MobileHotspotGUI::previousConnectionRequest(){
	QDBusConnection dbus = QDBusConnection::systemBus();
	QDBusMessage msg = QDBusMessage::createMethodCall("com.nokia.icd2", "/com/nokia/icd2", "com.nokia.icd2", "state_req");
	dbus.send(msg);
}
