#include <QtCore/QObject>
#include <QtGui/QMessageBox>
#include <QtGui/QMainWindow>
#include <QtCore/qplugin.h>
#include <QtGui/QIcon>
#include <QtCore/QString>
#include <QtCore/QTranslator>
#include "../../qtMobileHotspot/src/mobilehotspotpluginhelper.h"
#include "../../qtMobileHotspot/src/mobilehotspotplugin.h"
#include "../../qtMobileHotspot/src/mobilehotspotconfiguration.h"
#include "../../qtMobileHotspot/src/hotspothelper.h"
#include "pluginconfiguration.h"
#include "plugingui.h"
#include "plugin.h"


Plugin::Plugin(){
	helper = new MobileHotspotPluginHelper(this); // create the helper for this plugin
	conf = new PluginConfiguration(helper); // create the configuration of this plugin
	conf->load(); // load plugin's configuration (see PluginConfiguration::load())
	gui = 0; // no plugin GUI for now
}

Plugin::~Plugin(){
	delete conf; // delete this plugin's configuration
	delete helper; // delete this plugin's helper
}

QString Plugin::name(){
	return "org.dest.qtmh.winusb"; // unique name of the plugin, wont be displayed
}

QString Plugin::displayedName(){
	return tr("Windows USB Setup"); // displayed name in the plugins control window
}

QString Plugin::displayedAuthor(){
	return "Loic Burtin"; // displayed author name in the plugins control window
}

QString Plugin::displayedContact(){
	return "speedoflight@hotmail.fr"; // displayed contact-info in the plugins control window
}

QString Plugin::displayedDescription(){
	QString desc = tr("This assistant will help you into installing the USB networking driver for Windows.") + "\n";
	desc += tr("Hit the Configure / Reset buttons to start it.");
	return desc; // displayed description in the plugins control window
}

QIcon Plugin::displayedIcon(){
	return QIcon(); // displayed icon in the plugins control window
}

bool Plugin::allowEnableDisableStopped(){
	return false; // allow (or not) the user to enable & disable the plugin while the hotspot is stopped
}

bool Plugin::allowEnableDisableRunning(){
	return false; // allow (or not) the user to enable & disable the plugin while the hotspot is running
}

bool Plugin::allowConfigureStopped(){
	return true; // allow (or not) the user to configure the plugin while hotspot is stopped
}

bool Plugin::allowConfigureRunning(){
	return false; // allow (or not) the user to configure the plugin while hotspot is running
}

int Plugin::order(){
	// Plugins call ordering
	// Range should be 0..100 (included)
	// Startup calls are done in natural order (0, 1, ..., 100)
	// Shutdown calls are done in reverse order (100, 99, ..., 0)
	// Ranges 0..20 and 80..100 (included) should be reserved to 'official' plugins.
	return 0; // order is not important for WinUSB (it is not supposed to be enabled)
}

bool Plugin::isPluginEnabled(){
	// Return the enable status of the plugin
	// Should work for 99% of the plugins, feel free to tweak it if you want
	// return helper->isPluginEnabled(); // return true if the plugin is enabled, false if it is not
	return false; // not supposed to be enabled at all
}

void Plugin::setPluginEnabled(bool enabled, const MobileHotspotConfiguration *configuration, bool running){
	// Set the enable status of the plugin
	// Should work for all plugins whose enable status cannot be changed while running
	this->running = running; // Save running
	this->configuration = configuration; // Save configuration
	helper->setPluginEnabled(enabled); // true if the plugin is enabled, false if it is not

	// If your plugin can be enabled/disabled while running, you might want to add some code in this method (stop a process...)
	// eg if(running){ do_some_things(configuration); }
}

void Plugin::configure(const QMainWindow *parentWindow, const MobileHotspotConfiguration *configuration, bool running, bool reset){
	// User clicked the Configure or the Reset button in the control window
	this->running = running; // Save running
	this->configuration = configuration; // Save configuration
	QMainWindow *window = const_cast<QMainWindow*>(parentWindow); // Unconst the parentWindow
	helper->installTranslator(); // Enable our translations - Usually QTMH does it itself, but as we are using our own windows, it is our job this time
	usbWindowsAssistant(window);
	configureFinished();
}

void Plugin::configureFinished(){
	// Configuration GUI has been closed
	gui->configuration(conf); // Get plugin configuration from the GUI
	conf->save(); // Save plugin's configuration (see PluginConfiguration::save())
	helper->removeTranslator(); // Disable plugin translations, usually QTMH does it but this time its our job - see configure()

	// You may want to add here some things to do, usually there are some if configuration has been changed while the hotspot was running
	// eg if(this->running){ do_some_things(this->configuration); }
}

void Plugin::beforeStarting(const MobileHotspotConfiguration *configuration){
	// Run before the startup of the hotspot
	this->running = false; // Hotspot is not running for now
	this->configuration = configuration; // Save configuration
	this->successStarting = false; // Hotspot is not started
	this->successFinishing = false; // Hotspot is not terminated

	// Add your own code here
}

void Plugin::afterStarting(const MobileHotspotConfiguration *configuration, bool successStarting){
	// Run after the startup of the hotspot
	this->running = true; // Hotspot is running
	this->configuration = configuration; // Save configuration
	this->successStarting = successStarting; // Save success starting
	this->successFinishing = false; // Hotspot is not terminated

	// Add your own code here
}

void Plugin::beforeStopping(const MobileHotspotConfiguration *configuration, bool successStarting){
	// Run before the shutdown of the hotspot
	this->running = true; // Hotspot is running
	this->configuration = configuration; // Save configuration
	this->successStarting = successStarting; // Save success starting
	this->successFinishing = false; // Hotspot is not terminated

	// Add your own code here
}

void Plugin::afterStopping(const MobileHotspotConfiguration *configuration, bool successStarting, bool successFinishing){
	// Run after the shutdown of the hotspot
	this->running = false; // Hotspot is running
	this->configuration = configuration; // Save configuration
	this->successStarting = successStarting; // Save success starting
	this->successFinishing = successFinishing; // Save success finishing

	// Add your own code here
}

QTranslator* Plugin::translator(){
	// Translations need
	// Should be kept as is, even if there are no translations
	return &translater;
}

void Plugin::usbWindowsAssistant(QMainWindow *parent){
	QString title = tr("USB For Windows Setup");
	bool failed = false;
	QString text;
	QString usbMAC = configuration->usbMAC;
	QString lanNetwork = configuration->lanNetwork;

	try{
		text = tr("This will help you into installing the USB networking driver into Windows. Please disconnect your N900 from your computer before continuing.");
		if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::Cancel) != QMessageBox::Yes)
			throw QString();

		text = tr("First, ensure that you have copied the windows-usb-driver folder from /opt/qtmobilehotspot/docs (or QtMobileHotspot-Driver when connected in mass-storage mode) to your computer. If this is not the case, stop this assistant, and copy the folder.");
		if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::Cancel) != QMessageBox::Yes)
			throw QString();

		text = tr("We are now going to unload every USB module and load the one for USB networking. Ensure that you are disconnected from your PC and that your N900 isn't in any USB mode (no PC-Suite, no mass-storage).");
		if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::Cancel) != QMessageBox::Yes)
			throw QString();

		HotspotHelper::unloadStandardUSB();
		HotspotHelper::unloadNokiaUSB();
		if(HotspotHelper::checkUSBMode() != 0){
			text = tr("It seems that the USB device is busy. Usually it is due to a USB-mode being active (mass storage / pc-suite). Please stop this mode and try again.");
			QMessageBox::critical(parent, title, text);
			throw QString();
		}
		if(HotspotHelper::loadStandardUSB() != 0){
			text = tr("The system failed to load the standard g_ether USB module. It may be due to a missing power-kernel or to a system in an inconsistent state. Please try again later.");
			QMessageBox::critical(parent, title, text);
			throw QString();
		}
		if(HotspotHelper::prepareUSBInterface(usbMAC, lanNetwork) != 0){
			text = tr("The system failed to turn on the USB interface. It may be due to a system in an inconsistent state, please try again later.");
			QMessageBox::critical(parent, title, text);
			throw QString();
		}

		text = tr("Everything seems to be fine on your N900. You can now connect it to your computer, then continue. Do not select any USB-mode when asked.");
		if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::Cancel) != QMessageBox::Yes)
			throw QString();

		text = tr("Are you running Windows 7 / Windows Vista ?");
		if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::No) != QMessageBox::Yes){
			text = tr("The computer will ask you about drivers. Tell it that you want to do an expert install (pick the driver yourself).");
			if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::Cancel) != QMessageBox::Yes)
				throw QString();

			text = tr("Now edit the settings in order to make Windows use your windows-usb-driver folder. Make sure not to forget any useful checkbox.");
			if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::Cancel) != QMessageBox::Yes)
				throw QString();
		}
		else{
			text = tr("The computer will try to install a driver automatically (usually). It will fail, that is normal.");
			if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::Cancel) != QMessageBox::Yes)
				throw QString();

			text = tr("Now go to the device manager (control panel, system, device manager). You should see an unknown device called Linux Ethernet RNDIS. Right-click on it, and choose Update driver.");
			if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::Cancel) != QMessageBox::Yes)
				throw QString();

			text = tr("Choose to pick the driver yourself, and tell the program about your windows-usb-driver folder. Do not miss any important checkbox.");
			if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::Cancel) != QMessageBox::Yes)
				throw QString();
		}

		text = tr("Your computer will now install the USB networking driver. If everything goes fine, congratulations, you have a new network device available in the devices-list.");
		if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::Cancel) != QMessageBox::Yes)
			throw QString();

		text = tr("You are now ready to use the program in USB mode with your Windows. Just remember to never use pc-suite/mass-storage modes when you enable USB networking.");
		if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::Cancel) != QMessageBox::Yes)
			throw QString();

		text = tr("Windows should not need more configuration in order to be used with qtmobilehotspot. Simply run qtmobilehotspot, start in USB mode, plug your N900 into your PC, and enjoy.");
		if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::Cancel) != QMessageBox::Yes)
			throw QString();

		text = tr("Please note that if you are already using MADDE on your computer, you may have to reset your new network card settings to automatic.");
		if(QMessageBox::question(parent, title, text, QMessageBox::Yes | QMessageBox::Cancel) != QMessageBox::Yes)
			throw QString();
	}
	catch(QString fail){
		qDebug("WARNING : Interrupted Windows USB Assistant");
		failed = true;
	}

	text = tr("Please disconnect your N900 from your computer. We are going to unload the USB module and load the normal Nokia one. Do not choose any USB mode if you plan to use qtmobilehotspot.");
	if(! failed)
		QMessageBox::information(parent, title, text);
	HotspotHelper::unprepareUSBInterface();
	HotspotHelper::unloadStandardUSB();
	HotspotHelper::loadNokiaUSB();
}


Q_EXPORT_PLUGIN2(qtmhwinusb, Plugin); // 1st is target name, 2nd is class name
