#include <QtCore/QObject>
#include <QtGui/QMessageBox>
#include <QtCore/qplugin.h>
#include <QtCore/QString>
#include <QtCore/QTranslator>
#include <QtCore/QTimer>
#include <QtCore/QProcess>
#include "../../qtMobileHotspot/src/mobilehotspotpluginhelper.h"
#include "../../qtMobileHotspot/src/mobilehotspotplugin.h"
#include "../../qtMobileHotspot/src/mobilehotspotconfiguration.h"
#include "../../qtMobileHotspot/src/apnhelper.h"
#include "../../qtMobileHotspot/src/commandhelper.h"
#include "pluginconfiguration.h"
#include "plugingui.h"
#include "plugin.h"


Plugin::Plugin(){
	helper = new MobileHotspotPluginHelper(this); // create the helper for this plugin
	conf = new PluginConfiguration(helper); // create the configuration of this plugin
	conf->load(); // load plugin's configuration (see PluginConfiguration::load())
	internetPingEnabled = false;
	internetPingProcess = 0;
}

Plugin::~Plugin(){
	delete conf; // delete this plugin's configuration
	delete helper; // delete this plugin's helper
	delete translater; // delete this plugin's translator (even if there is no translation)
}

QString Plugin::name(){
	return "com.dest.qtmh.netfix"; // unique name of the plugin, wont be displayed
}

QString Plugin::displayedName(){
	return tr("Net Fixer"); // displayed name in the application menu (if applyable)
}

bool Plugin::displayName(){
	return true; // get (or not) an entry in the application menu
}

int Plugin::order(){
	// Plugins call ordering
	// Range should be 0..100 (included)
	// Startup calls are done in natural order (0, 1, ..., 100)
	// Shutdown calls are done in reverse order (100, 99, ..., 0)
	// Ranges 0..20 and 80..100 (included) should be reserved to 'official' plugins.
	return 20; // not important, should be run before 21..79 plugins thought
}

void Plugin::configure(const MobileHotspotConfiguration &configuration, const bool running){
	// Plugin's entry in the application menu has been clicked
	if(running){
		QMessageBox::information(0, tr("Configure"), tr("Please stop the hotspot before configuring."));
		return; // disallow configuration of the plugin if hotspot is running, its up to you to remove this check
	}

	PluginGUI gui; // create the GUI
	gui.setConfiguration(*conf); // load plugin configuration into the GUI
	if(gui.exec() != gui.Accepted) // display the GUI
		return;
	gui.configuration(conf); // get plugin configuration from the GUI
	conf->save(); // save plugin's configuration (see PluginConfiguration::save())
}

void Plugin::beforeStarting(const MobileHotspotConfiguration &configuration){
	// Run before the startup of the hotspot
	if(configuration.internetEnabled){
		if(configuration.internetInterface() == configuration.INTERFACE_WLAN){
			if(conf->disablePowerSaving){
				apnPowerSave = APNHelper::apnPowerSave(configuration.internetAPID);
				APNHelper::setAPNPowerSave(configuration.internetAPID, GCONF_APN_POWER_NOECO);
			}
		}
	}
}

void Plugin::afterStarting(const MobileHotspotConfiguration &configuration, const bool successStarting){
	// Run after the startup of the hotspot
	if(! successStarting)
		return;
	if(configuration.internetEnabled){
		if(configuration.internetInterface() == configuration.INTERFACE_WLAN){
			if(conf->disablePowerSaving)
				setPowerManagement(configuration.INTERFACE_WLAN, false);
		}
		else if(configuration.internetInterface() == configuration.INTERFACE_GPRS){
			if(conf->internetPing){
				internetPingEnabled = true;
				QTimer::singleShot(TIME_PING_START_FIRST, this, SLOT(mustStartPing()));
			}
		}
	}
}

void Plugin::beforeStopping(const MobileHotspotConfiguration &configuration, const bool successStarting){
	// Run before the shutdown of the hotspot
	if(configuration.internetEnabled){
		if(configuration.internetInterface() == configuration.INTERFACE_WLAN){
			if(conf->disablePowerSaving)
				APNHelper::setAPNPowerSave(configuration.internetAPID, apnPowerSave);
		}
		else if(configuration.internetInterface() == configuration.INTERFACE_GPRS){
			if(conf->internetPing){
				internetPingEnabled = false;
				if(internetPingProcess != 0){
					internetPingProcess->kill();
					internetPingProcess->waitForFinished(TIME_PING_WAIT_STOP);
					delete internetPingProcess;
					internetPingProcess = 0;
				}
			}
		}
	}
}

void Plugin::afterStopping(const MobileHotspotConfiguration & configuration, const bool successStarting, const bool successFinishing){
	// Run after the shutdown of the hotspot
	if(! successStarting)
		return;
	if(configuration.internetEnabled){
		if(configuration.internetInterface() == configuration.INTERFACE_WLAN){
			// HANDLED BY QTMH ITSELF : setPowerManagement(configuration.INTERFACE_WLAN, true);
		}
	}
}

QTranslator* Plugin::translator(){
	// Translations need
	// Should be kept as is, even if there are no translations
	return translater;
}

void Plugin::setTranslator(QTranslator *translater){
	// Translations need
	// Should be kept as is, even if there are no translations
	this->translater = translater;
}

int Plugin::setPowerManagement(QString interface, bool management){
	/* -1=fail, -2=timeout, 0=ok */
	qDebug("NetFix :: Modifying power management of %s to %s", interface.toAscii().data(), management ? "ON" : "OFF");
	int ret;
	bool cmdret;
	QString cmd("/sbin/iwconfig "); cmd += interface;
	cmd += management ? " power on" : " power off";
	cmdret = CommandHelper::executeCommand(cmd, &ret);
	if( (! cmdret) || (ret != 0) ){
		qDebug("WARNING : NetFix :: Cannot modify power management");
		return -1;
	}
	return 0;
}

void Plugin::mustStartPing(){
	if(! internetPingEnabled)
		return;
	qDebug("NetFix :: Starting ping process to %s for %d ms", conf->internetPingIP.toAscii().data(), TIME_PING_STOP);
	internetPingProcess = new QProcess();
	bool ret = CommandHelper::executeCommandAsync(QString("ping ") + conf->internetPingIP, internetPingProcess);
	if(! ret)
		qDebug("WARNING : NetFix :: Failed to start the ping process");
	QTimer::singleShot(TIME_PING_STOP, this, SLOT(mustStopPing()));

}

void Plugin::mustStopPing(){
	if(! internetPingEnabled)
		return;
	qDebug("NetFix :: Stopping ping process for %d ms", TIME_PING_START_RUNNING);
	if(internetPingProcess != 0){
		internetPingProcess->kill();
		internetPingProcess->waitForFinished(TIME_PING_WAIT_STOP);
		delete internetPingProcess;
		internetPingProcess = 0;
	}
	QTimer::singleShot(TIME_PING_START_RUNNING, this, SLOT(mustStartPing()));
}


Q_EXPORT_PLUGIN2(qtmhnetfix, Plugin); // 1st is target name, 2nd is class name
