/* Copyright (c) 2010, Nokia Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * * Neither the name of the Nokia Corporation nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Object to handle async, complex header search from modest / tinymail
 */

/* System includes */
#include <tny-simple-list.h>
#include <tny-account.h>
#include <tny-folder.h>
#include <tny-folder-store.h>
#include <tny-folder-store-query.h>

/* Own includes */
#include "qtmm-folder-search.h"
#include "qtmm-defs.h"
#include "qtmm-debug.h"
#include "qtmm-marshal.h"

#define SEARCH_THRESHOLD 50

enum {
    SIG_HEADERS_RECEIVED = 0,
    SIG_DONE,
    SIGNALS
};
static gulong signals[SIGNALS] = {0};

/* Private data definition */
struct _QtmMFolderSearchPrivate {
    gboolean   disposed;

    TnyFolder *folder;
    time_t     begin_time;
    time_t     end_time;
    gboolean   sync;

    gint       progress;
    gint       max;

    TnyList   *headers;

    gulong     result_source;

    /* Search state */
    QtmmSearchState status;
};

#define GET_PRIVATE(o) ((QtmMFolderSearchPrivate *)           \
                        ((QtmMFolderSearch *)(o))->priv)

G_DEFINE_TYPE (QtmMFolderSearch, qtmm_folder_search, G_TYPE_OBJECT);

/*
 * Private function prototypes
 */

/* Reply for listing headers */
static void
get_headers_reply (TnyFolder *folder,
                   gboolean   cancelled,
                   TnyList   *headers,
                   GError    *error,
                   gpointer   user_data);

/* Status update for listing headers */
static void
header_status_cb (GObject   *folder,
                  TnyStatus *status,
                  gpointer   user_data);

/* Idle function to handle results */
static gboolean
handle_headers_func (gpointer data);

/* Creates a header info Map from TnyHeader */
static GHashTable *
create_header_info (TnyHeader *header);

/* Frees a GValue struch with g_slice_free */
static void
value_slice_free (GValue *value);

/*
 * GObject funcs
 */

/* Instance init */
static void
qtmm_folder_search_init (QtmMFolderSearch *search)
{
    QtmMFolderSearchPrivate *priv;

    QTMM_DEBUG();

    priv = search->priv = G_TYPE_INSTANCE_GET_PRIVATE (
        search, QTMM_TYPE_FOLDER_SEARCH, QtmMFolderSearchPrivate);
}

/* Instance disposal, unref all other GObjects here */
static void
qtmm_folder_search_dispose (GObject *object)
{
    QtmMFolderSearchPrivate *priv;

    QTMM_DEBUG();

    priv = GET_PRIVATE (object);

    if (priv == NULL) return;

    if (priv->disposed) {
        QTMM_DEBUG ("Already disposed");
        return;
    }

    priv->disposed = TRUE;

    if (priv->folder) {
        g_object_unref (priv->folder);
        priv->folder = NULL;
    }

    if (priv->headers) {
        g_object_unref (priv->headers);
        priv->headers = NULL;
    }

    G_OBJECT_CLASS (qtmm_folder_search_parent_class)->dispose (object);
}

/* Instance disposal, unref all other GObjects here */
static void
qtmm_folder_search_finalize (GObject *object)
{
    QtmMFolderSearchPrivate *priv;

    QTMM_DEBUG();

    priv = GET_PRIVATE (object);

    if (priv == NULL) return;

    if (priv->result_source) {
        g_source_remove (priv->result_source);
        priv->result_source = 0;
    }

    G_OBJECT_CLASS (qtmm_folder_search_parent_class)->finalize (object);
}

/* Class (type) init */
static void
qtmm_folder_search_class_init (QtmMFolderSearchClass *klass)
{
    GObjectClass *object_class;

    QTMM_DEBUG();

    object_class = G_OBJECT_CLASS (klass);

    object_class->dispose  = qtmm_folder_search_dispose;
    object_class->finalize = qtmm_folder_search_finalize;

    /* Signals */
    signals[SIG_HEADERS_RECEIVED] = g_signal_new (
        "headers-received",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST,
        0, NULL, NULL,
        qtmm_marshal_VOID__STRING_POINTER,
        G_TYPE_NONE, 2,
        G_TYPE_STRING,
        G_TYPE_POINTER);

    signals[SIG_DONE] = g_signal_new (
        "done",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST,
        0, NULL, NULL,
        g_cclosure_marshal_VOID__VOID,
        G_TYPE_NONE, 0);

    g_type_class_add_private (klass, sizeof (QtmMFolderSearchPrivate));
}

/*
 * Public functions
 */

/* Create a new instance with appropriate search parameters */
GObject *
qtmm_folder_search_new (TnyFolder *folder,
                        time_t     begin_time,
                        time_t     end_time,
                        gboolean   sync)
{
    QtmMFolderSearchPrivate *priv;
    GObject *self;

    QTMM_DEBUG();

    g_return_val_if_fail (folder != NULL, NULL);

    self = g_object_new (QTMM_TYPE_FOLDER_SEARCH, NULL);

    priv = GET_PRIVATE (self);
    priv->folder       = g_object_ref (folder);
    priv->begin_time   = begin_time;
    priv->end_time     = end_time;
    priv->sync         = sync;

    return self;
}

/* Start the search, all results are signalled async */
gboolean
qtmm_folder_search_start (QtmMFolderSearch *self)
{
    QtmMFolderSearchPrivate *priv;

    priv = GET_PRIVATE (self);

    g_return_val_if_fail (priv != NULL, FALSE);

    if (priv->status != QTMM_SEARCH_IDLE) {
        QTMM_DEBUG ("Can't start as the state is not idle %d", priv->status);
        return FALSE;
    }

    if (!priv->folder) {
        priv->status = QTMM_SEARCH_CANCELLED;
        return FALSE;
    }

    QTMM_DEBUG ("Listing headers.");

    priv->status = QTMM_SEARCH_RUNNING;

    priv->headers = tny_simple_list_new ();

    tny_folder_get_headers_async (
        priv->folder,
        priv->headers,
        priv->sync,
        get_headers_reply,
        header_status_cb,
        self);

    return TRUE;
}

/* Get the status of the search op */
QtmmSearchState
qtmm_folder_search_get_status (QtmMFolderSearch *search)
{
    QtmMFolderSearchPrivate *priv;

    g_return_val_if_fail (search != NULL, QTMM_SEARCH_CANCELLED);

    priv = GET_PRIVATE (search);

    g_return_val_if_fail (priv != NULL, QTMM_SEARCH_CANCELLED);

    QTMM_DEBUG ("-> %d", priv->status);

    return priv->status;
}

/*
 * Private functions
 */

/* Reply for listing headers */
static void
get_headers_reply (TnyFolder *folder,
                   gboolean   cancelled,
                   TnyList   *headers,
                   GError    *error,
                   gpointer   user_data)
{
    QtmMFolderSearchPrivate *priv;

    QTMM_DEBUG ();

    priv = GET_PRIVATE (user_data);

    if (!priv || priv->disposed) return;

    if (cancelled) {
        QTMM_DEBUG ("Cancelled...");
        priv->status = QTMM_SEARCH_CANCELLED;
    }

    if (error) {
        QTMM_DEBUG_ERR (error, "Failed to list headers");
    }

    /* If we had any results, fire an idle function to handle them, otherwise
     * signal were done
     */
    if (tny_list_get_length (headers) != 0) {
        if (!priv->result_source) {
            priv->result_source = g_idle_add (handle_headers_func, user_data);
        }
    } else {
        QTMM_DEBUG ("No headers in folder \"%s\"", tny_folder_get_id (folder));

        priv->status = QTMM_SEARCH_DONE;

        /* Ref and unref the search object, as it's likely the caller will
         * dispose us in the signal
         */
        g_object_ref (user_data);
        g_signal_emit (user_data, signals[SIG_DONE], 0);
        g_object_unref (user_data);
    }
}

/* Status update for listing headers */
static void
header_status_cb (GObject   *folder,
                  TnyStatus *status,
                  gpointer   user_data)
{
}

/* Idle function to handle results */
static gboolean
handle_headers_func (gpointer data)
{
    QtmMFolderSearchPrivate *priv;
    GPtrArray *results = NULL;
    guint i;

    if (!QTMM_IS_FOLDER_SEARCH (data)) return FALSE;

    priv = GET_PRIVATE (data);

    if (!priv || priv->disposed) return FALSE;

    QTMM_DEBUG ();

    /* We're done, emit done signal and stop the loop */
    if (!priv->headers || tny_list_get_length (priv->headers) == 0) {
        priv->result_source = 0;
        g_object_ref (data);
        g_signal_emit (data, signals[SIG_DONE], 0);
        g_object_unref (data);
        return FALSE;
    }

    for (i = 0; i < SEARCH_THRESHOLD; i++) {
        TnyIterator *iter;
        GObject     *header;
        time_t       time_stamp = 0;

        iter   = tny_list_create_iterator (priv->headers);
        header = tny_iterator_get_current (iter);
        g_object_unref (iter);

        if (!header) continue;

        /* Let's remove the object from results, it's handled, no matter
         * if it's ending into the final set or not.
         */
        tny_list_remove (priv->headers, header);

        time_stamp = tny_header_get_date_received (TNY_HEADER (header));
        if (!time_stamp) {
            time_stamp = tny_header_get_date_sent (
                TNY_HEADER (header));
        }

        /* Assumption is, that begin time is smaller than end time. If it's
         * not, we could throw an error in GetHeaders right off the start...
         */
        if (time_stamp) {
            if (priv->begin_time) {
                if (time_stamp < priv->begin_time) {
                    g_object_unref (header);
                    continue;
                }
            }
            if (priv->end_time) {
                if (time_stamp > priv->end_time) {
                    g_object_unref (header);
                    continue;
                }
            }
        }

        if (!results) {
            results = g_ptr_array_new ();
        }

        g_ptr_array_add (results, create_header_info (TNY_HEADER (header)));

        g_object_unref (header);
    }

    if (results) {
        g_signal_emit (
            data, signals[SIG_HEADERS_RECEIVED], 0,
            tny_folder_get_id (priv->folder),
            results);

        g_ptr_array_foreach (results, (GFunc)g_hash_table_unref, NULL);
        g_ptr_array_free (results, TRUE);
    }

    return TRUE;
}

/* Creates a header info Map from TnyHeader */
static GHashTable *
create_header_info (TnyHeader *header)
{
    GHashTable *info       = NULL;
    GValue     *variant    = NULL;
    gchar      *str_value  = NULL;
    time_t      time_stamp = 0;
    guint       size       = 0;
    guint       flags      = 0;
    guint       priority   = 0;

    info = g_hash_table_new_full (
        g_str_hash, g_str_equal, NULL, (GDestroyNotify)value_slice_free);

    str_value = tny_header_dup_uid (header);
    if (str_value) {
        variant = g_slice_new0 (GValue);
        g_value_init (variant, G_TYPE_STRING);
        g_value_take_string (variant, str_value);
        g_hash_table_insert (info, "message-uid", variant);
    }

    str_value = tny_header_dup_bcc (header);
    if (str_value) {
        variant = g_slice_new0 (GValue);
        g_value_init (variant, G_TYPE_STRING);
        g_value_take_string (variant, str_value);
        g_hash_table_insert (info, "bcc", variant);
    }

    str_value = tny_header_dup_cc (header);
    if (str_value) {
        variant = g_slice_new0 (GValue);
        g_value_init (variant, G_TYPE_STRING);
        g_value_take_string (variant, str_value);
        g_hash_table_insert (info, "cc", variant);
    }

    time_stamp = tny_header_get_date_received (header);
    if (time_stamp) {
        variant = g_slice_new0 (GValue);
        g_value_init (variant, G_TYPE_UINT64);
        g_value_set_uint64 (variant, time_stamp);
        g_hash_table_insert (info, "date-received", variant);
    }

    time_stamp = tny_header_get_date_sent (header);
    if (time_stamp) {
        variant = g_slice_new0 (GValue);
        g_value_init (variant, G_TYPE_UINT64);
        g_value_set_uint64 (variant, time_stamp);
        g_hash_table_insert (info, "date-sent", variant);
    }

    /* XXX: Unlike documentation says, this is not returning a dynamically
     *      allocated value, so don't return this value. NB#160412
     *
     * As a matter of fact, wonder if this is even implemented?
     */
#if 0
    str_value = tny_header_dup_message_id (header);
    if (str_value) {
        variant = g_slice_new0 (GValue);
        g_value_init (variant, G_TYPE_STRING);
        g_value_set_string (variant, str_value);
        g_hash_table_insert (info, "message-mime-id", variant);
    }
#endif

    size = tny_header_get_message_size (header);
    variant = g_slice_new0 (GValue);
    g_value_init (variant, G_TYPE_UINT64);
    g_value_set_uint64 (variant, size);
    g_hash_table_insert (info, "size", variant);

    str_value = tny_header_dup_from (header);
    if (str_value) {
        variant = g_slice_new0 (GValue);
        g_value_init (variant, G_TYPE_STRING);
        g_value_take_string (variant, str_value);
        g_hash_table_insert (info, "from", variant);
    }

    str_value = tny_header_dup_to (header);
    if (str_value) {
        variant = g_slice_new0 (GValue);
        g_value_init (variant, G_TYPE_STRING);
        g_value_take_string (variant, str_value);
        g_hash_table_insert (info, "to", variant);
    }

    str_value = tny_header_dup_subject (header);
    if (str_value) {
        variant = g_slice_new0 (GValue);
        g_value_init (variant, G_TYPE_STRING);
        g_value_take_string (variant, str_value);
        g_hash_table_insert (info, "subject", variant);
    }

    str_value = tny_header_dup_replyto (header);
    if (str_value) {
        variant = g_slice_new0 (GValue);
        g_value_init (variant, G_TYPE_STRING);
        g_value_take_string (variant, str_value);
        g_hash_table_insert (info, "reply-to", variant);
    }

    flags = tny_header_get_flags (header);
    if (flags) {
        variant = g_slice_new0 (GValue);
        g_value_init (variant, G_TYPE_UINT);
        g_value_set_uint (variant, flags);
        g_hash_table_insert (info, "flags", variant);
    }

    priority = tny_header_get_priority (header);
    if (priority) {
        variant = g_slice_new0 (GValue);
        g_value_init (variant, G_TYPE_UINT);
        g_value_set_uint (variant, priority);
        g_hash_table_insert (info, "priority", variant);
    }

    return info;
}

/* Frees a GValue struct with g_slice_free */
static void
value_slice_free (GValue *value)
{
    g_value_unset (value);
    g_slice_free (GValue, value);
}
