//    QtBatteryWidget
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "mainwindow.h"
#include "ui_mainwindow.h"
#include <QUuid>
#include <QDebug>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <QtGui/QX11Info>
#include <QProcess>
#include <QPainter>
#include <QFile>
#include "commonstatic.h"
#include "settingsdialog.h"
#include "widgetmanager.h"
#include <mce/dbus-names.h>
#include <mce/mode-names.h>

MainWindow::MainWindow(QWidget *parent, QString appletId) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    setAttribute(Qt::WA_TranslucentBackground);
    // Get required atoms
    Atom winTypeAtom = XInternAtom(QX11Info::display(), "_NET_WM_WINDOW_TYPE", false);
    Atom homeAppletAtom = XInternAtom(QX11Info::display(), "_HILDON_WM_WINDOW_TYPE_HOME_APPLET", false);
    Atom appletIDAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_ID", false);
    Atom utf8Atom = XInternAtom(QX11Info::display(), "UTF8_STRING", false);

    // Set correct window type
    XChangeProperty(QX11Info::display(), winId(), winTypeAtom, XA_ATOM, 32,
            PropModeReplace, (unsigned char *) &homeAppletAtom, 1);

    if (!appletId.isEmpty())
        m_AppletId = appletId;
    else
        m_AppletId = QString("%1_%2").arg(QCoreApplication::instance()->applicationName().remove(' ')).arg(QUuid::createUuid().toString().remove("{").remove("}"));
    qDebug() << QString("AppletId = %1").arg(m_AppletId);
    QByteArray id (m_AppletId.toUtf8());
    XChangeProperty(QX11Info::display(), winId(), appletIDAtom, utf8Atom, 8,
                    PropModeReplace, (unsigned char *)id.constData(), id.length());

    // Add setting button. This button is shown when hildon-desktop is in edit mode.
    Atom appletSettingAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_SETTINGS", false);
    int settings = 0;
    XChangeProperty(QX11Info::display(), winId(), appletSettingAtom, XA_CARDINAL, 32,
            PropModeReplace, (unsigned char*)&settings, 1);

    ui->centralWidget->layout()->setSizeConstraint(QLayout::SetMinAndMaxSize);

    m_Bq27x00 = CommonStatic::Bq27x00Present(this);
    qDebug() << "Module bq27x00" << m_Bq27x00;

    m_Pressed = false;
    m_PbCharge = NULL;
    m_Timer = new QTimer(this);
    m_Timer->setInterval(1000);
    m_Timer->setSingleShot(false);
    connect(m_Timer, SIGNAL(timeout()), this, SLOT(ChargingTimerSlot()));

    m_WasCharging = false;
    m_LastRemainingSecs = 0;
    m_LastPercentage = 0;
    m_BackgroundColor = QApplication::palette().window().color();

    ApplySettings();

#ifdef Q_WS_MAEMO_5
    QDBusConnection conn = QDBusConnection::systemBus();

    if (!conn.connect(QString(), "/org/freedesktop/Hal/devices/bme", "org.freedesktop.Hal.Device",
                      "PropertyModified", this, SLOT(BmePropertyModifiedSlot(QDBusMessage)))){
        qCritical() << QString("Failed to connect to org.freedesktop.Hal.Device");
    }

    conn.connect(QString(), MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
                 MCE_DISPLAY_SIG, this, SLOT(DisplayStateChangedSlot(QString)));
#endif
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    qDebug() << "Close event: removing widget";
    WidgetManager::RemoveWidget(m_AppletId);
    QMainWindow::closeEvent(event);
}

void MainWindow::paintEvent(QPaintEvent *event)
{
    QPainter p(this);
    p.setBrush(m_BackgroundColor);
    p.setPen(Qt::NoPen);
    QRect myRect(rect().left(), rect().top(), rect().width(), rect().height());
    p.drawRoundedRect(myRect, 15.0, 15.0);

    QWidget::paintEvent(event);
}

QString MainWindow::GetAppletId()
{
    return m_AppletId;
}

//Filter show setting requests
bool MainWindow::x11Event(XEvent* event)
{
  static Atom appletShowSettingAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_SHOW_SETTINGS", false);

  if (event->xclient.message_type == appletShowSettingAtom ) {
    showSettingsDialog();
    return true;
  }
  return false;
}

void MainWindow::showSettingsDialog()
{
    SettingsDialog* frm = new SettingsDialog();
    if (frm->exec() == QDialog::Accepted)
        ApplySettings();
    delete frm;
}

void MainWindow::BmePropertyModifiedSlot(QDBusMessage)
{
#ifdef Q_WS_MAEMO_5
    qDebug() << "------------ BmePropertyModifiedSlot ------------";
    QString command = "hal-device bme";

    float temperature = 0.0;
    float vcurrent = 0;
    float perc = 0.0;
    float consumption = 0.0;

    if (CommonStatic::CurrentSettings->PercType == Settings::SourceBq27200){
        qDebug() << "Getting data from bq27200";
        perc = GetValue("/sys/class/power_supply/bq27200-0/capacity");
        vcurrent = GetValue("/sys/class/power_supply/bq27200-0/voltage_now");
        //Fix voltage for kernel v47
        if (vcurrent > 9999)
             vcurrent /= 1000;
    }

    temperature = GetTemperature();
    qDebug() << "Battery temperature: " << temperature;
    ui->lblTemp->setText(QString("%1C").arg(QString::number(temperature, 'f', 1)));

    if (CommonStatic::CurrentSettings->TimeLeft == Settings::TimeLeftConsumption){
        consumption = GetPowerConsumption();
        qDebug() << "Power consumption: " << consumption << "mA";
    }

    qDebug() << "Getting data from omap34xx";
    QProcess* proc = new QProcess(this);
    proc->start(command, QIODevice::ReadOnly);
    proc->waitForFinished();
    if (proc->exitCode() == 0){
        QByteArray ba = proc->readAll();
        QString output(ba);
        QStringList lines = output.split("\n", QString::SkipEmptyParts);

        float current = 0;
        float design = 0;
        float vdesign = 0;        

        bool charging = false;
        bool chargingChanged = false;

        foreach (QString line, lines){
            line = line.trimmed();
            //qDebug() << line;

            QStringList e = line.split("=", QString::SkipEmptyParts);
            if (e.count() > 1){
                QString name = e.at(0).trimmed();
                QString value = e.at(1).trimmed();
                int pos = value.indexOf(" ");
                if (pos != -1)
                    value = value.left(pos);

                if (name == "battery.reporting.current"){
                   current = value.toFloat();
                }else if (name == "battery.reporting.design"){
                   design = value.toFloat();
                }else if (name == "battery.voltage.current" && vcurrent == 0){
                   vcurrent = value.toFloat();
                }else if (name == "battery.voltage.design"){
                   vdesign = value.toFloat();
                }else if (name == "battery.rechargeable.is_charging"){
                   charging = value == "true";
                }
            }
        }

        chargingChanged = m_WasCharging != charging;
        if (chargingChanged)
            qDebug() << "Charging state changed";

        //When charging use the bq27x00 percentage, if possible
        if (charging && m_Bq27x00 && CommonStatic::CurrentSettings->PercType != Settings::SourceBq27200) {
            qDebug() << "Getting percentage from bq27200 (charging)";
            perc = GetValue("/sys/class/power_supply/bq27200-0/capacity");
        }

        if (perc == 0.0){
            if (CommonStatic::CurrentSettings->PercType == Settings::SourceDrFrost){
                qDebug() << "Getting percentage (dr_frost)";
                if (vcurrent >= 4050)
                    perc = 85.0 + (vcurrent - 4050.0) / 12.0;
                else if (vcurrent >= 3900)
                    perc = 70.0 + (vcurrent - 3900.0) / 10.0;
                else if (vcurrent >= 3800)
                    perc = 50.0 + (vcurrent - 3800.0) / 5.0;
                else if (vcurrent >= 3660)
                    perc = 15.0 + (vcurrent - 3660.0) / 4.0;
                else if (vcurrent >= 3600)
                    perc = 5.0 + (vcurrent - 3600.0) / 6.0;
                else
                    perc = (vcurrent - 3300.0) / 60.0;
            }else{
                qDebug() << "Getting percentage (standard)";
                if (design > 0)
                    perc = current / design * 100.0;
                else
                    perc = 0;
            }
        }

        //Reset last values if charging state changed:
        if (chargingChanged){
            qDebug() << "Resetting last data (chargingChanged)";
            m_LastPercentage = 0;
            m_LastPercentageTime = QDateTime::currentDateTime();
            m_LastRemainingSecs = 0;
        }

        int remainingSecs = 0;
        if (m_LastPercentage != 0 || consumption != 0){
            qDebug() << "Calculating time left";
            if (!charging){
                if (consumption > 0){
                    remainingSecs = current / consumption * 3600.0;
                }else if (perc == m_LastPercentage || perc > m_LastPercentage){
                    remainingSecs = m_LastRemainingSecs;
                }else{
                    float delta = m_LastPercentage - perc;
                    if (delta > 0){
                        int secs = m_LastPercentageTime.secsTo(QDateTime::currentDateTime());

                        remainingSecs = perc / delta * (float)secs;
                        m_LastRemainingSecs = remainingSecs;
                    }
                }
            }else{
                if (consumption < 0){
                    remainingSecs = (design - current) / (consumption * -1.0) * 3600.0;
                }else if (perc == m_LastPercentage || perc < m_LastPercentage){
                    remainingSecs = m_LastRemainingSecs;
                }else{
                    float delta = perc - m_LastPercentage;
                    if (delta > 0){
                        int secs = m_LastPercentageTime.secsTo(QDateTime::currentDateTime());

                        remainingSecs = (100.0 - perc) / delta * (float)secs;
                        m_LastRemainingSecs = remainingSecs;
                    }
                }
            }
        }
        qDebug() << "Last battery percentage: " << m_LastPercentage;
        qDebug() << "Battery percentage: " << perc;
        qDebug() << "Battery charge: " << current << "/" << design;
        qDebug() << "Battery voltage: " << vcurrent << "/" << vdesign;
        qDebug() << "Battery time left: " << CommonStatic::FormatHourMinutes(remainingSecs);
        qDebug() << "Battery is charging: " << charging;

        m_WasCharging = charging;

        if (!charging){
            if (m_PbCharge)
                m_PbCharge->SetValue(perc * 100);
            ui->lblPerc->setText(QString("%1%").arg(QString::number(perc, 'f', 1)));
        }

        if (chargingChanged && m_PbCharge)
            m_PbCharge->SetValue(perc * 100);
        ui->lblCharge->setText(QString("%1 mAh").arg((int)current));
        ui->lblVoltage->setText(QString("%1 mV").arg((int)vcurrent));
        ui->lblRemaining->setText(CommonStatic::FormatHourMinutes(remainingSecs));

        if (m_PbCharge)
            m_PbCharge->SetIsCharging(charging);

        if (!charging){
            m_Timer->stop();
            if (perc < m_LastPercentage || m_LastPercentage == 0){
                m_LastPercentageTime = QDateTime::currentDateTime();
                m_LastPercentage = perc;
            }
        }else{
            if (perc > m_LastPercentage || m_LastPercentage == 0){
                m_LastPercentageTime = QDateTime::currentDateTime();
                m_LastPercentage = perc;
            }
            if(!m_Timer->isActive())
                m_Timer->start();
        }
    }else{
        qWarning() << "Command returned" << proc->exitCode();
        if (m_PbCharge)
            m_PbCharge->SetValue(0);
        ui->lblPerc->setText(QString("%1%").arg(QString::number(0, 'f', 1)));
        ui->lblCharge->setText(QString("%1 mAh").arg(0));
        ui->lblVoltage->setText(QString("%1 mV").arg(0));
        ui->lblRemaining->setText("-");
    }
    delete proc;
#endif
}

void MainWindow::ApplySettings()
{
    m_LastPercentage = 0;
    m_LastPercentageTime = QDateTime();
    m_LastRemainingSecs = 0;

    if (CommonStatic::CurrentSettings->PercStyle == Settings::StyleGraphical ||
        CommonStatic::CurrentSettings->PercStyle == Settings::StyleStandard) {

        if (m_PbCharge == NULL){
            m_PbCharge = new BatteryWidget(ui->frame);
            m_PbCharge->SetNormalLevelImage(QPixmap(":/images/green.png"));
            m_PbCharge->SetLowLevelImage(QPixmap(":/images/orange.png"));
            m_PbCharge->SetCriticalLevelImage(QPixmap(":/images/red.png"));
            m_PbCharge->setFixedSize(166, 98);
            m_PbCharge->SetRange(0, 10000);            
            m_PbCharge->SetLevels(50, 30);
            m_PbCharge->SetValue(0);
            ui->frame->layout()->addWidget(m_PbCharge);
        }
    }else{
        if (m_PbCharge){
            ui->frame->layout()->removeWidget(m_PbCharge);
            delete m_PbCharge;
            m_PbCharge = NULL;
        }
    }

    if (CommonStatic::CurrentSettings->PercStyle == Settings::StyleGraphical){
        m_PbCharge->SetBackgroundImage(QPixmap(":/images/battery.png"));
        m_PbCharge->setFixedHeight(QWIDGETSIZE_MAX);
        m_PbCharge->SetInsideMargins(8, 8, 16, 8);

        QSize bkgSize(166, 98);
        bkgSize.scale(166*CommonStatic::CurrentSettings->PercScale, 98*CommonStatic::CurrentSettings->PercScale, Qt::KeepAspectRatio);

        ui->lblPerc->setVisible(false);
        ui->frame->setVisible(true);

        qDebug() << "Percentage size:" << bkgSize.width() << "x" << bkgSize.height();
        ui->frame->setFixedSize(bkgSize);
        m_PbCharge->setFixedSize(bkgSize);
        m_PbCharge->SetInsideMargins(8*CommonStatic::CurrentSettings->PercScale, 8*CommonStatic::CurrentSettings->PercScale, 16*CommonStatic::CurrentSettings->PercScale, 8*CommonStatic::CurrentSettings->PercScale);
    }else if (CommonStatic::CurrentSettings->PercStyle == Settings::StyleStandard){
        m_PbCharge->SetBackgroundImage(QPixmap(":/images/grey.png"));

        QSize bkgSize(166, 98);
        bkgSize.scale(166*CommonStatic::CurrentSettings->PercScale, 98*CommonStatic::CurrentSettings->PercScale, Qt::KeepAspectRatio);
        bkgSize.setHeight(48);
        m_PbCharge->setFixedSize(bkgSize);

        ui->lblPerc->setVisible(false);
        ui->frame->setVisible(true);
        qDebug() << "Percentage size:" << bkgSize.width() << "x" << bkgSize.height();
        ui->frame->setFixedSize(bkgSize);
        m_PbCharge->setFixedSize(bkgSize);
        m_PbCharge->SetInsideMargins(0,0,0,0);
    }else if (CommonStatic::CurrentSettings->PercStyle == Settings::StyleTextual){
        ui->lblPerc->setVisible(true);
        ui->frame->setVisible(false);
    }

    ui->label->setVisible(CommonStatic::CurrentSettings->ShowCharge && CommonStatic::CurrentSettings->ShowLabels);
    ui->lblCharge->setVisible(CommonStatic::CurrentSettings->ShowCharge);

    ui->label_2->setVisible(CommonStatic::CurrentSettings->ShowVoltage && CommonStatic::CurrentSettings->ShowLabels);
    ui->lblVoltage->setVisible(CommonStatic::CurrentSettings->ShowVoltage);

    ui->label_3->setVisible(CommonStatic::CurrentSettings->ShowRemaining && CommonStatic::CurrentSettings->ShowLabels);
    ui->lblRemaining->setVisible(CommonStatic::CurrentSettings->ShowRemaining);

    ui->label_4->setVisible(CommonStatic::CurrentSettings->ShowTemp && CommonStatic::CurrentSettings->ShowLabels);
    ui->lblTemp->setVisible(CommonStatic::CurrentSettings->ShowTemp);

    Qt::Alignment align = Qt::AlignRight | Qt::AlignVCenter;
    if (!CommonStatic::CurrentSettings->ShowLabels)
        align = Qt::AlignCenter;
    ui->lblCharge->setAlignment(align);
    ui->lblVoltage->setAlignment(align);
    ui->lblRemaining->setAlignment(align);
    ui->lblTemp->setAlignment(align);

    m_BackgroundColor.setAlpha(CommonStatic::CurrentSettings->Opacity);

    BmePropertyModifiedSlot(QDBusMessage());

    SizeToFit();
    update();
}

float MainWindow::GetValue(QString file)
{
    float value = 0.0;
    QFile in(file);
    if (in.open(QIODevice::ReadOnly)){
        QString temp(in.readAll());
        value = temp.trimmed().toFloat();
        in.close();
    }
    return value;
}

void MainWindow::SizeToFit()
{
    int spacing = ui->centralWidget->layout()->spacing();
    int height = ui->centralWidget->layout()->contentsMargins().top() + ui->centralWidget->layout()->contentsMargins().bottom();    

    if (CommonStatic::CurrentSettings->PercStyle == Settings::StyleTextual)
        height += ui->lblPerc->sizeHint().height() + spacing;
    else{
        height += ui->frame->sizeHint().height() + spacing;
        height += ui->frame->layout()->contentsMargins().top() + ui->frame->layout()->contentsMargins().bottom();
    }
    if (CommonStatic::CurrentSettings->ShowCharge)
        height += ui->lblCharge->sizeHint().height() + spacing;
    if (CommonStatic::CurrentSettings->ShowRemaining)
        height += ui->lblRemaining->sizeHint().height() + spacing;
    if (CommonStatic::CurrentSettings->ShowTemp)
        height += ui->lblTemp->sizeHint().height() + spacing;
    if (CommonStatic::CurrentSettings->ShowVoltage)
        height += ui->lblVoltage->sizeHint().height() + spacing;

    setFixedHeight(height);
}

void MainWindow::ChargingTimerSlot()
{
    if (m_PbCharge){
        if (m_PbCharge->Value() == m_PbCharge->Maximum() || m_PbCharge->Value() < m_LastPercentage * 100){
            m_PbCharge->SetValue(m_LastPercentage * 100);
            ui->lblPerc->setText(tr("%1% Charging").arg(QString::number(m_LastPercentage, 'f', 1)));
        }else{
            if (m_PbCharge->Value() + 500 > m_PbCharge->Maximum())
                m_PbCharge->SetValue(m_PbCharge->Maximum(), false);
            else
                m_PbCharge->SetValue(m_PbCharge->Value() + 500, false);
        }
    }
}

void MainWindow::DisplayStateChangedSlot(QString displayState)
{
    qDebug() << "Display state changed:" << displayState;
    if (displayState == QLatin1String(MCE_DISPLAY_OFF_STRING)) {
        if (m_Timer->isActive())
            m_Timer->stop();
    }else if (displayState == QLatin1String(MCE_DISPLAY_ON_STRING)) {
        if (m_WasCharging)
            m_Timer->start();
        else
            BmePropertyModifiedSlot(QDBusMessage());
    }
}

float MainWindow::GetTemperature()
{
    float temperature = GetValue("/sys/class/power_supply/bq27200-0/temp");
    if (temperature == 0)
        temperature = GetValue("/sys/devices/platform/omap34xx_temp/temp1_input");

    //Fix temperature for kernel v47
    if (temperature > 100)
        temperature /= 10.0;
    return temperature;
}

float MainWindow::GetPowerConsumption()
{
    float res = 0.0;

    if (QFile::exists("/usr/sbin/i2cget")){
        //With i2cget:
        qDebug() << "Getting power consumption with i2cget";
        QProcess* proc = new QProcess(this);
        proc->start("/usr/sbin/i2cget -y 2 0x55 0x14 w", QIODevice::ReadOnly);
        proc->waitForFinished();
        if (proc->exitCode() == 0){
            QByteArray ba = proc->readAllStandardOutput();
            QString output(ba);
            output = output.trimmed();

            qDebug() << "i2cget output: " << output;
            bool ok = false;
            res = output.toInt(&ok,16);
            if (ok)
                res = res * 3570.0 / 22.0 / 1000.0;
        }
        delete proc;
    }

    if (res == 0 && QFile::exists("/sys/class/power_supply/bq27200-0/current_now")){
        //From bq27200:
        qDebug() << "Getting power consumption from bq27200";
        float curr = GetValue("/sys/class/power_supply/bq27200-0/current_now");
        res =  curr * 3570.0 / 22.0 / 1000.0;
    }

    return res;
}

void MainWindow::mousePressEvent(QMouseEvent* event)
{
    qDebug() << "mousePressEvent";
    m_Pressed = true;
    QMainWindow::mousePressEvent(event);
}

void MainWindow::mouseReleaseEvent(QMouseEvent* event)
{
    qDebug() << "mouseReleaseEvent";
    if (m_Pressed){
        qDebug() << "Updating data (user request)";
        m_Pressed = false;
        BmePropertyModifiedSlot(QDBusMessage());
    }
    QMainWindow::mouseReleaseEvent(event);
}
