/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef TEXT_P_H
#define TEXT_P_H

#include "item_p.h"

class TextPrivate;

class Text : public Item
{
    Q_OBJECT

    Q_PROPERTY(QColor color READ color WRITE setColor NOTIFY colorChanged)
    Q_PROPERTY(ElideMode elide READ elide WRITE setElide NOTIFY elideChanged)
    Q_PROPERTY(HAlignment horizontalAlignment READ horizontalAlignment WRITE setHorizontalAlignment NOTIFY alignmentChanged)
    Q_PROPERTY(VAlignment verticalAlignment READ verticalAlignment WRITE setVerticalAlignment NOTIFY alignmentChanged)
    Q_PROPERTY(int lineCount READ lineCount NOTIFY textChanged)
    Q_PROPERTY(int maximumLineCount READ maximumLineCount WRITE setMaximumLineCount NOTIFY maximumLineCountChanged)
    Q_PROPERTY(qreal lineHeight READ lineHeight WRITE setLineHeight NOTIFY lineHeightChanged)
    Q_PROPERTY(LineHeightMode lineHeightMode READ lineHeightMode WRITE setLineHeightMode NOTIFY lineHeightModeChanged)
    Q_PROPERTY(bool smooth READ smooth WRITE setSmooth NOTIFY smoothChanged)
    Q_PROPERTY(TextStyle style READ textStyle WRITE setTextStyle NOTIFY textStyleChanged)
    Q_PROPERTY(TextFormat textFormat READ textFormat WRITE setTextFormat NOTIFY textFormatChanged)
    Q_PROPERTY(QColor styleColor READ styleColor WRITE setStyleColor NOTIFY styleColorChanged)
    Q_PROPERTY(QString text READ text WRITE setText NOTIFY textChanged)
    Q_PROPERTY(bool truncated READ truncated NOTIFY textChanged)
    Q_PROPERTY(WrapMode wrapMode READ wrapMode WRITE setWrapMode NOTIFY wrapModeChanged)

    Q_ENUMS(ElideMode HAlignment VAlignment)

public:
    enum ElideMode {
        ElideLeft = 0,
        ElideRight,
        ElideMiddle,
        ElideNone
    };

    enum HAlignment {
        AlignLeft = Qt::AlignLeft,
        AlignRight = Qt::AlignRight,
        AlignHCenter = Qt::AlignHCenter,
        AlignJustify = Qt::AlignJustify
    };

    enum VAlignment {
        AlignTop = Qt::AlignTop,
        AlignBottom = Qt::AlignBottom,
        AlignVCenter = Qt::AlignVCenter
    };

    enum LineHeightMode {
        ProportionalHeight = 0,
        FixedHeight
    };

    enum TextStyle {
        Normal = 0,
        Outline,
        Raised,
        Sunken
    };

    enum TextFormat {
        AutoText = 0,
        PlainText,
        RichText,
        StyledText
    };

    enum WrapMode {
        NoWrap = 0,
        WordWrap,
        ManualWrap,
        WrapAnywhere,
        WrapAtWordBoundaryOrAnywhere
    };

    explicit Text(QWidget *parent = 0);
    ~Text();

    QColor color() const;
    void setColor(const QColor &color);

    QColor styleColor() const;
    void setStyleColor(const QColor &color);

    ElideMode elide() const;
    void setElide(ElideMode elide);

    HAlignment horizontalAlignment() const;
    void setHorizontalAlignment(HAlignment alignment);
    VAlignment verticalAlignment() const;
    void setVerticalAlignment(VAlignment alignment);

    int lineCount() const;
    int maximumLineCount() const;
    void setMaximumLineCount(int count);

    qreal lineHeight() const;
    void setLineHeight(qreal height);

    LineHeightMode lineHeightMode() const;
    void setLineHeightMode(LineHeightMode mode);

    bool smooth() const;
    void setSmooth(bool smooth);

    TextStyle textStyle() const;
    void setTextStyle(TextStyle style);

    TextFormat textFormat() const;
    void setTextFormat(TextFormat format);

    QString text() const;
    void setText(const QString &text);

    bool truncated() const;

    WrapMode wrapMode() const;
    void setWrapMode(WrapMode mode);

signals:
    void colorChanged();
    void elideChanged();
    void alignmentChanged();
    void textChanged();
    void maximumLineCountChanged();
    void lineHeightChanged();
    void lineHeightModeChanged();
    void smoothChanged();
    void textStyleChanged();
    void textFormatChanged();
    void styleColorChanged();
    void textChanged();
    void truncatedChanged();
    void wrapModeChanged();

protected:
    Text(TextPrivate &dd, QWidget *parent = 0);

    Q_DECLARE_PRIVATE(Text)

private:
    Q_DISABLE_COPY(Text)
};

QML_DECLARE_TYPE(Text)

#endif // TEXT_P_H
