/*
 * Copyright (C) 2015 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
 
#include "qchvideoplayer.h"
#include "metadatawatcher.h"
#include "missioncontrol.h"
#include "mafw/mafwregistryadapter.h"
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <QApplication>
#include <QDBusConnection>
#include <QDBusMessage>
#include <QDeclarativeInfo>
#include <QVBoxLayout>

class QchVideoPlayerPrivate
{

public:
    QchVideoPlayerPrivate(QchVideoPlayer *parent) :
        q_ptr(parent),
        videoWidget(0),
        layout(0),
        mafwRegistry(0),
        mafwRenderer(0),
        mafwPlaylist(0),
        mafwTrackerSource(0),
        metadataWatcher(0),
        missionControl(0),
        autoLoad(true),
        bufferProgress(0.0),
        seekable(true),
        position(0),
        duration(0),
        status(QchMediaStatus::Stopped),
        muted(false),
        volume(0),
        muteVolume(0),
        tickInterval(1000),
        positionTimerId(-1),
        sourceLoaded(true),
        readyToPlay(false),
        playWhenReady(false)
    {
    }
    
    void init() {
        Q_Q(QchVideoPlayer);
        videoWidget = new QWidget(q);
        videoWidget->setAttribute(Qt::WA_NativeWindow);
        videoWidget->setStyleSheet("background-color: rgb(3, 13, 3);");
        layout = new QVBoxLayout(q);
        layout->addWidget(videoWidget);
        layout->setContentsMargins(0, 0, 0, 0);
        
        QApplication::syncX();
        mafwRenderer->setColorKey(QColor(3, 13, 3).rgb() & 0xffffff);
        mafwRenderer->setWindowXid(videoWidget->winId());
    
        if (mafwRenderer->isRendererReady()) {
            mafwRenderer->getStatus();
            mafwRenderer->getVolume();
        }
        else {
            q->connect(mafwRenderer, SIGNAL(rendererReady()), mafwRenderer, SLOT(getStatus()));
            q->connect(mafwRenderer, SIGNAL(rendererReady()), mafwRenderer, SLOT(getVolume()));
        }
    }
    
    void loadSource() {
        QString uriToPlay = source.startsWith("/") ? "file://" + source : source;
        QString objectId = mafwTrackerSource->createObjectId(uriToPlay);
        
        if (uriToPlay.startsWith("file://")) {
            QString mime(gnome_vfs_get_mime_type_for_name(uriToPlay.toUtf8()));
            objectId = objectId.remove(0, 18) // "urisource::file://"
                               .replace("/", "%2F")
                               .prepend(QString("localtagfs::%1/")
                               .arg(mime.startsWith("video") ? "videos" : "music/songs"));
        }
        
        mafwPlaylist->assignVideoPlaylist();
        mafwPlaylist->clear();
        mafwPlaylist->appendItem(objectId);
        sourceLoaded = true;
    }
    
    void startPositionTimer() {
        if (positionTimerId == -1) {
            Q_Q(QchVideoPlayer);
            positionTimerId = q->startTimer(tickInterval);
        }
        
        mafwRenderer->getPosition();
    }
    
    void stopPositionTimer() {
        if (positionTimerId != -1) {
            Q_Q(QchVideoPlayer);
            q->killTimer(positionTimerId);
            positionTimerId = -1;
        }
    }
    
    void _q_onStatusReady(MafwPlaylist*, uint index, MafwPlayState state, const char*, const QString &error) {
        Q_Q(QchVideoPlayer);
        q->disconnect(mafwRenderer, SIGNAL(signalGetStatus(MafwPlaylist*,uint,MafwPlayState,const char*,QString)),
                      q, SLOT(_q_onStatusReady(MafwPlaylist*,uint,MafwPlayState,const char*,QString)));

        q->connect(mafwRenderer, SIGNAL(stateChanged(int)), q, SLOT(_q_onStateChanged(int)));
        
        _q_onStateChanged(state);
        
        if (!error.isEmpty()) {
            qmlInfo(q) << error;
        }
    }
    
    void _q_onMetaDataChanged() {
        Q_Q(QchVideoPlayer);
        int dur = metadataWatcher->metadata().value(MAFW_METADATA_KEY_DURATION).toInt();
        bool seek = metadataWatcher->metadata().value(MAFW_METADATA_KEY_IS_SEEKABLE).toBool();
        QString uri = metadataWatcher->metadata().value(MAFW_METADATA_KEY_URI).toString();
        
        if (dur != duration) {
            duration = dur;
            emit q->durationChanged();
        }
        
        if (seek != seekable) {
            seekable = seek;
            emit q->seekableChanged();
        }
        
        if (uri != source) {
            source = uri;
            sourceLoaded = true;
            emit q->sourceChanged();
        }
        
        emit q->hasAudioChanged();
        emit q->hasVideoChanged();
    }
    
    void _q_onBufferProgressChanged(float progress) {
        Q_Q(QchVideoPlayer);
        bufferProgress = progress;
        emit q->bufferProgressChanged();
    }
    
    void _q_onPositionChanged(int pos) {
        Q_Q(QchVideoPlayer);
        position = pos;
        emit q->positionChanged();
    }
    
    void _q_onVolumeChanged(int vol) {
        Q_Q(QchVideoPlayer);
        volume = vol;
        emit q->volumeChanged();
    }
    
    void _q_onStateChanged(int state) {
        Q_Q(QchVideoPlayer);
        QchMediaStatus::Status oldStatus = status;
        
        switch (state) {
        case Transitioning:
            readyToPlay = false;
            status = QchMediaStatus::Loading;
            break;
        case Playing:
            readyToPlay = false;
            status = QchMediaStatus::Playing;
            
            if (tickInterval > 0) {
                startPositionTimer();
            }
            
            if (oldStatus == QchMediaStatus::Paused) {
                emit q->resumed();
            }
            else {
                emit q->started();
            }
            
            break;
        case Paused:
            readyToPlay = true;
            
            if (playWhenReady) {
                playWhenReady = false;
                q->play();
            }
            else {
                status = QchMediaStatus::Paused;
                mafwRenderer->getPosition();
                stopPositionTimer();
                emit q->paused();
            }
            
            break;
        case Stopped:
            readyToPlay = true;
            
            if (playWhenReady) {
                playWhenReady = false;
                q->play();
            }
            else {
                if ((duration > 0) && (position >= duration)) {
                    status = QchMediaStatus::EndOfMedia;
                }
                else {
                    status = QchMediaStatus::Stopped;
                }
                
                position = 0;
                stopPositionTimer();
                emit q->positionChanged();
                emit q->stopped();
            }
            
            break;
        default:
            readyToPlay = false;
            break;
        }
        
        emit q->statusChanged();
    }
    
    void _q_onPropertyChanged(const QDBusMessage &msg) {
        if (msg.arguments()[0].toString() == MAFW_PROPERTY_RENDERER_VOLUME) {
            Q_Q(QchVideoPlayer);
            volume = qdbus_cast<QVariant>(msg.arguments()[1]).toInt();
            emit q->volumeChanged();
        }
    }
    
    void _q_onError(const QDBusMessage &msg) {
        if (msg.arguments()[0] == "com.nokia.mafw.error.renderer") {
            status = QchMediaStatus::Error;
            errorString.clear();
            errorString.append(QchVideoPlayer::tr("Unable to play media"));
            errorString.append("\n");

            if (msg.arguments()[1] == MAFW_RENDERER_ERROR_NO_MEDIA) {
                errorString.append(QchVideoPlayer::tr("Media not found"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_URI_NOT_AVAILABLE) {
                errorString.append(QchVideoPlayer::tr("URI not available"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_INVALID_URI) {
                errorString.append(QchVideoPlayer::tr("Invalid URI"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_MEDIA_NOT_FOUND) {
                errorString.append(QchVideoPlayer::tr("Unable to open media"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_STREAM_DISCONNECTED) {
                errorString.append(QchVideoPlayer::tr("Playback stream no longer available"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_TYPE_NOT_AVAILABLE) {
                errorString.append(QchVideoPlayer::tr("Could not determine MIME-type"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_PLAYBACK) {
                errorString.append(QchVideoPlayer::tr("General error occured, unable to continue playback"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_UNABLE_TO_PERFORM) {
                errorString.append(QchVideoPlayer::tr("General error occured"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_UNSUPPORTED_TYPE) {
                errorString.append(QchVideoPlayer::tr("Unsupported media"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_UNSUPPORTED_RESOLUTION) {
                errorString.append(QchVideoPlayer::tr("Unsupported resolution"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_UNSUPPORTED_FPS) {
                errorString.append(QchVideoPlayer::tr("Unsupported framerate"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_DRM) {
                errorString.append(QchVideoPlayer::tr("Media is protected by DRM"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_DEVICE_UNAVAILABLE) {
                errorString.append(QchVideoPlayer::tr("System sound device is unavailable"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CORRUPTED_FILE) {
                errorString.append(QchVideoPlayer::tr("Media corrupted"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_PLAYLIST_PARSING) {
                errorString.append(QchVideoPlayer::tr("Error while parsing playlist"));
                errorString.append(QchVideoPlayer::tr("Playlist may be corrupt or empty"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CODEC_NOT_FOUND) {
                errorString.append(QchVideoPlayer::tr("Codec not found:") + "\n");
                errorString.append(msg.arguments()[2].toString());
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_VIDEO_CODEC_NOT_FOUND) {
                errorString.append(QchVideoPlayer::tr("Video codec not found:") + "\n");
                errorString.append(msg.arguments()[2].toString());
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_AUDIO_CODEC_NOT_FOUND) {
                errorString.append(QchVideoPlayer::tr("Video codec not found:") + "\n");
                errorString.append(msg.arguments()[2].toString());
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_NO_PLAYLIST) {
                errorString.append(QchVideoPlayer::tr("No playlist assigned"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_INDEX_OUT_OF_BOUNDS) {
                errorString.append(QchVideoPlayer::tr("Media index is not in bound with playlist items"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CANNOT_PLAY) {
                errorString.append(QchVideoPlayer::tr("Unable to start playback"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CANNOT_STOP) {
                errorString.append(QchVideoPlayer::tr("Unable to stop playback"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CANNOT_PAUSE) {
                errorString.append(QchVideoPlayer::tr("Unable to pause playback"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CANNOT_SET_POSITION) {
                errorString.append(QchVideoPlayer::tr("Unable to seek position in media"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CANNOT_GET_POSITION) {
                errorString.append(QchVideoPlayer::tr("Unable to retrieve current position in media"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CANNOT_GET_STATUS) {
                errorString.append(QchVideoPlayer::tr("Unable to get current playback status"));
            }
                
            Q_Q(QchVideoPlayer);
            emit q->statusChanged();
            emit q->error(errorString);
        }
    }
    
    QchVideoPlayer *q_ptr;
    QWidget *videoWidget;
    QVBoxLayout *layout;
    MafwRegistryAdapter *mafwRegistry;
    MafwRendererAdapter *mafwRenderer;
    MafwPlaylistAdapter *mafwPlaylist;
    MafwSourceAdapter *mafwTrackerSource;
    MetadataWatcher *metadataWatcher;
    MissionControl *missionControl;
    
    bool autoLoad;
    
    qreal bufferProgress;
    
    QString errorString;
    
    bool seekable;
    
    int position;
    int duration;
    
    QString source;
    
    QchMediaStatus::Status status;
    
    bool muted;
    
    int volume;
    int muteVolume;
    
    int tickInterval;
    
    int positionTimerId;
    
    bool sourceLoaded;
    
    bool readyToPlay;
    bool playWhenReady;
    
    Q_DECLARE_PUBLIC(QchVideoPlayer)
};

QchVideoPlayer::QchVideoPlayer(QWidget *parent) :
    QWidget(parent),
    d_ptr(new QchVideoPlayerPrivate(this))
{
    Q_D(QchVideoPlayer);
    setAttribute(Qt::WA_OpaquePaintEvent);
    
    d->mafwRegistry = MafwRegistryAdapter::get();
    d->mafwRenderer = d->mafwRegistry->renderer();
    d->mafwPlaylist = d->mafwRegistry->playlist();
    d->mafwTrackerSource = d->mafwRegistry->source(MafwRegistryAdapter::Tracker);
    d->metadataWatcher = MetadataWatcher::acquire();
    d->missionControl = MissionControl::acquire();
    
    d->mafwRenderer->setErrorPolicy(MAFW_RENDERER_ERROR_POLICY_STOP);
    
    connect(d->metadataWatcher, SIGNAL(metadataChanged()), this, SLOT(_q_onMetaDataChanged()));
    connect(d->mafwRenderer, SIGNAL(signalGetStatus(MafwPlaylist*,uint,MafwPlayState,const char*,QString)),
            this, SLOT(_q_onStatusReady(MafwPlaylist*,uint,MafwPlayState,const char*,QString)));
    connect(d->mafwRenderer, SIGNAL(signalGetPosition(int,QString)), this, SLOT(_q_onPositionChanged(int)));
    connect(d->mafwRenderer, SIGNAL(signalGetVolume(int)), this, SLOT(_q_onVolumeChanged(int)));
    connect(d->mafwRenderer, SIGNAL(bufferingInfo(float)), this, SLOT(_q_onBufferProgressChanged(float)));
            
    QDBusConnection::sessionBus().connect("com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer",
                                          "/com/nokia/mafw/renderer/gstrenderer",
                                          "com.nokia.mafw.extension",
                                          "property_changed",
                                          this, SLOT(_q_onPropertyChanged(const QDBusMessage &)));
                                          
    QDBusConnection::sessionBus().connect("",
                                          "/com/nokia/mafw/renderer/gstrenderer",
                                          "com.nokia.mafw.extension",
                                          "error",
                                          this, SLOT(_q_onError(const QDBusMessage &)));
}

QchVideoPlayer::~QchVideoPlayer() {
    Q_D(QchVideoPlayer);
    d->mafwRenderer->setErrorPolicy(MAFW_RENDERER_ERROR_POLICY_CONTINUE);
    d->mafwRenderer->enablePlayback(false);
    d->mafwRenderer->stop();
}

bool QchVideoPlayer::autoLoad() const {
    Q_D(const QchVideoPlayer);
    return d->autoLoad;
}

void QchVideoPlayer::setAutoLoad(bool enable) {
    if (enable != autoLoad()) {
        Q_D(QchVideoPlayer);
        d->autoLoad = enable;
        emit autoLoadChanged();
    }
}

qreal QchVideoPlayer::bufferProgress() const {
    Q_D(const QchVideoPlayer);
    return d->bufferProgress;
}

QString QchVideoPlayer::errorString() const {
    Q_D(const QchVideoPlayer);
    return d->errorString;
}

bool QchVideoPlayer::hasAudio() const {
    return !metaData()->audioCodec().isEmpty();
}

bool QchVideoPlayer::hasVideo() const {
    return !metaData()->videoCodec().isEmpty();
}

MetadataWatcher* QchVideoPlayer::metaData() const {
    Q_D(const QchVideoPlayer);
    return d->metadataWatcher;
}

bool QchVideoPlayer::isMuted() const {
    Q_D(const QchVideoPlayer);
    return d->muted;
}

void QchVideoPlayer::setMuted(bool muted) {
    if (muted != isMuted()) {
        Q_D(QchVideoPlayer);
        d->muted = muted;
        emit mutedChanged();
        
        if (muted) {
            d->muteVolume = volume();
            setVolume(0);
        }
        else {
            setVolume(d->muteVolume);
        }
    }
}

bool QchVideoPlayer::isPaused() const {
    return status() == QchMediaStatus::Paused;
}

void QchVideoPlayer::setPaused(bool paused) {
    if (paused) {
        pause();
    }
    else if (isPaused()) {
        play();
    }
}

bool QchVideoPlayer::isPlaying() const {
    switch (status()) {
    case QchMediaStatus::Loading:
    case QchMediaStatus::Playing:
        return true;
    default:
        return false;
    }
}

void QchVideoPlayer::setPlaying(bool playing) {
    if (playing) {
        play();
    }
    else if (isPlaying()) {
        pause();
    }
}

bool QchVideoPlayer::isSeekable() const {
    Q_D(const QchVideoPlayer);
    return d->seekable;
}

int QchVideoPlayer::position() const {
    Q_D(const QchVideoPlayer);
    return d->position;
}

void QchVideoPlayer::setPosition(int pos) {
    if (pos != position()) {
        Q_D(QchVideoPlayer);
        d->mafwRenderer->setPosition(SeekAbsolute, pos);
        d->mafwRenderer->getPosition();
    }
}

int QchVideoPlayer::duration() const {
    Q_D(const QchVideoPlayer);
    return d->duration;
}

QString QchVideoPlayer::source() const {
    Q_D(const QchVideoPlayer);
    return d->source;
}

void QchVideoPlayer::setSource(const QString &uri) {
    if (uri != source()) {
        Q_D(QchVideoPlayer);
        d->source = uri;
        
        if (autoLoad()) {
            d->loadSource();
        }
        else {
            d->sourceLoaded = false;
        }
    }
}

QchMediaStatus::Status QchVideoPlayer::status() const {
    Q_D(const QchVideoPlayer);
    return d->status;
}

int QchVideoPlayer::tickInterval() const {
    Q_D(const QchVideoPlayer);
    return d->tickInterval;
}

void QchVideoPlayer::setTickInterval(int interval) {
    if (interval != tickInterval()) {
        Q_D(QchVideoPlayer);
        d->tickInterval = qMax(0, interval);
        emit tickIntervalChanged();
        
        d->stopPositionTimer();
        
        if ((interval > 0) && (isPlaying())) {
            d->startPositionTimer();
        }
    }
}

int QchVideoPlayer::volume() const {
    Q_D(const QchVideoPlayer);
    return d->volume;
}

void QchVideoPlayer::setVolume(int vol) {
    if (vol != volume()) {
        Q_D(QchVideoPlayer);
        d->volume = qBound(0, vol, 100);
        d->mafwRenderer->setVolume(d->volume);
        d->mafwRenderer->getVolume();
    }
}

void QchVideoPlayer::play() {
    Q_D(QchVideoPlayer);
    
    if (isPaused()) {
        d->mafwRenderer->resume();
    }
    else {
        if (!d->sourceLoaded) {
            d->loadSource();
        }
        
        if (d->readyToPlay) {
            d->mafwRenderer->play();
        }
        else {
            d->playWhenReady = true;
        }
    }
}

void QchVideoPlayer::pause() {
    if (!isPlaying()) {
        Q_D(QchVideoPlayer);
        d->mafwRenderer->pause();
    }
}

void QchVideoPlayer::stop() {
    if ((isPlaying()) || (isPaused())) {
        Q_D(QchVideoPlayer);
        d->mafwRenderer->stop();
    }
}

void QchVideoPlayer::showEvent(QShowEvent *e) {
    QWidget::showEvent(e);
    emit visibleChanged();
}

void QchVideoPlayer::resizeEvent(QResizeEvent *e) {
    QWidget::resizeEvent(e);
    emit sizeChanged();
}

void QchVideoPlayer::hideEvent(QHideEvent *e) {
    QWidget::hideEvent(e);
    emit visibleChanged();
}

void QchVideoPlayer::timerEvent(QTimerEvent *) {
    Q_D(QchVideoPlayer);
    d->mafwRenderer->getPosition();
}

void QchVideoPlayer::classBegin() {}

void QchVideoPlayer::componentComplete() {
    Q_D(QchVideoPlayer);
    d->init();
}

#include "moc_qchvideoplayer.cpp"
