/*
 * Copyright (C) 2015 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "qchdbusadaptor.h"
#include "qchdbusutils.h"
#include "qmetaobjectbuilder_p.h"
#include <QDeclarativeInfo>
#include <QDBusArgument>
#include <QDBusConnection>
#include <QDBusMessage>

class QchDBusAdaptorPrivate
{

public:
    QchDBusAdaptorPrivate(QchDBusAdaptor *parent) :
        q_ptr(parent),
        adaptor(0),
        bus(QchDBus::SessionBus),
        path("/"),
        enabled(true),
        complete(false)
    {
    }
    
    virtual ~QchDBusAdaptorPrivate() {
        unregisterObject();
        unregisterService();
    }
    
    bool serviceRegistered() const {
        return services.contains(service);
    }

    void registerService() {
        if (service.isEmpty()) {
            return;
        }

        if (!serviceRegistered()) {
            if (QchDBus::connection(bus).registerService(service)) {
                services[service] = 1;
            }
            else {
                Q_Q(QchDBusAdaptor);
                qmlInfo(q) << QchDBusAdaptor::tr("Cannot register service %1").arg(service);
            }
        }
        else {
            services[service]++;
        }
    }

    void unregisterService() {
        if (services[service] == 1) {
            if (QchDBus::connection(bus).unregisterService(service)) {
                services.remove(service);
            }
            else {
                Q_Q(QchDBusAdaptor);
                qmlInfo(q) << QchDBusAdaptor::tr("Cannot unregister service %1").arg(service);
            }
        }
        else {
            services[service]--;
        }
    }

    void registerObject() {
        Q_Q(QchDBusAdaptor);
        
        if (!adaptor) {
            adaptor = new QchDBusAdaptorInternal(q);
        }
        
        if (!QchDBus::connection(bus).registerObject(path.isEmpty() ? "/" : path, adaptor,
                                                     QDBusConnection::ExportAllContents)) {
            qmlInfo(q) << QchDBusAdaptor::tr("Cannot register object");
        }
    }

    void unregisterObject() {
        Q_Q(QchDBusAdaptor);
        QchDBus::connection(bus).unregisterObject(path.isEmpty() ? "/" : path);
    }

    static QHash<QString, int> services;

    QchDBusAdaptor *q_ptr;
    QchDBusAdaptorInternal *adaptor;
    
    QchDBus::BusType bus;

    QString interface;
    QString path;
    QString service;
    QString xml;
    
    bool enabled;
    bool complete;

    Q_DECLARE_PUBLIC(QchDBusAdaptor)
};

QHash<QString, int> QchDBusAdaptorPrivate::services;

/*!
    \class DBusAdaptor
    \brief Registers the object with DBus.
    
    \ingroup dbus
    
    \snippet dbus.qml DBusAdaptor
        
    \sa DBusConnections, DBusMessage
*/
QchDBusAdaptor::QchDBusAdaptor(QObject *parent) :
    QObject(parent),
    d_ptr(new QchDBusAdaptorPrivate(this))
{
}

QchDBusAdaptor::QchDBusAdaptor(QchDBusAdaptorPrivate &dd, QObject *parent) :
    QObject(parent),
    d_ptr(&dd)
{
}

QchDBusAdaptor::~QchDBusAdaptor() {}

/*!
    \brief The bus on which the object is registered.
    
    Possible values are:
    
    <table>
        <tr>
            <th>Value</th>
            <th>Description</th>
        </tr>
        <tr>
            <td>DBus.SessionBus</td>
            <td>Registers the object on the session bus (default).</td>
        </tr>
        <tr>
            <td>DBus.SystemBus</td>
            <td>Registers the object on the system bus.</td>
        </tr>
    </table>
*/
QchDBus::BusType QchDBusAdaptor::bus() const {
    Q_D(const QchDBusAdaptor);
    return d->bus;
}

void QchDBusAdaptor::setBus(QchDBus::BusType b) {
    Q_D(QchDBusAdaptor);

    if (!d->complete) {
        d->bus = b;
        emit busChanged();
    }
    else {
        qmlInfo(this) << tr("Bus can only be set during initialisation");
    }
}

bool QchDBusAdaptor::isEnabled() const {
    Q_D(const QchDBusAdaptor);
    return d->enabled;
}

void QchDBusAdaptor::setEnabled(bool enabled) {
    if (enabled != isEnabled()) {
        Q_D(QchDBusAdaptor);
        d->enabled = enabled;
        emit enabledChanged();
        
        if (d->complete) {
            if (enabled) {
                d->registerObject();
            }
            else {
                d->unregisterObject();
            }
        }
    }
}

/*!
    \brief The DBus interface on which the object is registered.
*/
QString QchDBusAdaptor::interfaceName() const {
    Q_D(const QchDBusAdaptor);
    return d->interface;
}

void QchDBusAdaptor::setInterfaceName(const QString &name) {
    Q_D(QchDBusAdaptor);

    if (!d->complete) {
        d->interface = name;
        emit interfaceNameChanged();
    }
    else {
        qmlInfo(this) << tr("Interface name can only be set during initialisation");
    }
}

/*!
    \brief The DBus path on which the object is registered.
*/
QString QchDBusAdaptor::path() const {
    Q_D(const QchDBusAdaptor);
    return d->path;
}

void QchDBusAdaptor::setPath(const QString &path) {
    Q_D(QchDBusAdaptor);

    if (!d->complete) {
        d->path = path;
        emit pathChanged();
    }
    else {
        qmlInfo(this) << tr("Path can only be set during initialisation");
    }
}

/*!
    \brief The DBus service on which the object is registered.
*/
QString QchDBusAdaptor::serviceName() const {
    Q_D(const QchDBusAdaptor);
    return d->service;
}

void QchDBusAdaptor::setServiceName(const QString &name) {
    Q_D(QchDBusAdaptor);

    if (!d->complete) {
        d->service = name;
        emit serviceNameChanged();
    }
    else {
        qmlInfo(this) << tr("Service name can only be set during initialisation");
    }
}

/*!
    \brief The introspection XML for the DBus interface.
*/
QString QchDBusAdaptor::xml() const {
    Q_D(const QchDBusAdaptor);
    return d->xml;
}

void QchDBusAdaptor::setXml(const QString &x) {
    Q_D(QchDBusAdaptor);

    if (!d->complete) {
        d->xml = x;
        emit xmlChanged();
    }
    else {
        qmlInfo(this) << tr("Introspection XML can only be set during initialisation");
    }
}

void QchDBusAdaptor::classBegin() {}

void QchDBusAdaptor::componentComplete() {
    Q_D(QchDBusAdaptor);
    d->complete = true;
    d->registerService();
    d->registerObject();
}

class QchDBusAdaptorInternalPrivate
{

public:
    QchDBusAdaptorInternalPrivate(QchDBusAdaptorInternal *parent) :
        q_ptr(parent),
        object(0),
        dynamicMetaObject(0)
    {
    }
    
    virtual ~QchDBusAdaptorInternalPrivate() {        
        if (dynamicMetaObject) {
            free(dynamicMetaObject);
        }
    }
    
    void init(QchDBusAdaptor *obj) {
        if (!obj) {
            return;
        }
        
        Q_Q(QchDBusAdaptorInternal);
        object = obj;
        QMetaObjectBuilder builder(&QchDBusAdaptorInternal::staticMetaObject);
        builder.addClassInfo("D-Bus Interface", object->interfaceName().toUtf8());
        builder.addClassInfo("D-Bus Introspection", object->xml().toUtf8());
        
        QList<QByteArray> objectSignals;
        const QMetaObject *mo = object->metaObject();
        
        for (int i = QchDBusAdaptor::staticMetaObject.methodCount(); i < mo->methodCount(); i++) {
            const QMetaMethod mm = mo->method(i);            
            
            if (mm.methodType() == QMetaMethod::Signal) {
                builder.addMethod(mm);
                objectSignals << QMetaObject::normalizedSignature(mm.signature());
            }
            else {
                QByteArray signature = mm.signature();
                const QByteArray name = signature.left(signature.indexOf('('));
                signature = name + "(QBusMessage)";
                QMetaMethodBuilder methodBuilder = builder.addSlot(signature);
                methodBuilder.setReturnType(mm.typeName());
            }
        }
                
        for (int i = QchDBusAdaptor::staticMetaObject.propertyCount(); i < mo->propertyCount(); i++) {
            builder.addProperty(mo->property(i));
        }
        
        if (dynamicMetaObject) {
            free(dynamicMetaObject);
        }
        
        dynamicMetaObject = builder.toMetaObject();
        
        foreach (const QByteArray &signal, objectSignals) {
            q->connect(object, signal, q, signal);
        }
        
        const QMetaObject *qmo = q->metaObject();
        
        for (int i = QchDBusAdaptorInternal::staticMetaObject.methodCount(); i < qmo->methodCount(); i++) {
            qDebug() << qmo->method(i).signature();
        }
        
        for (int i = qmo->classInfoOffset(); i < qmo->classInfoCount(); i++) {
            const QMetaClassInfo info = qmo->classInfo(i);
            qDebug() << info.name() << ":" << info.value();
        }
    }

    QchDBusAdaptorInternal *q_ptr;
    QchDBusAdaptor *object;
    QMetaObject *dynamicMetaObject;
    
    QHash<QString, int> objectMethods;

    Q_DECLARE_PUBLIC(QchDBusAdaptorInternal)
};

QchDBusAdaptorInternal::QchDBusAdaptorInternal(QchDBusAdaptor *parent) :
    QObject(parent),
    d_ptr(new QchDBusAdaptorInternalPrivate(this))
{
    Q_D(QchDBusAdaptorInternal);
    
    if (parent) {
        d->init(parent);
    }
    
    //setAutoRelaySignals(true);
}

QchDBusAdaptorInternal::~QchDBusAdaptorInternal() {}

const QMetaObject* QchDBusAdaptorInternal::metaObject() const {
    Q_D(const QchDBusAdaptorInternal);
    return d->dynamicMetaObject ? d->dynamicMetaObject : &staticMetaObject;
}

void QchDBusAdaptorInternal::handleMessage(const QDBusMessage &message) const {
    Q_D(const QchDBusAdaptorInternal);
    
    if (!d->objectMethods.contains(message.member())) {
        QchDBus::connection(d->object->bus()).send(message.createErrorReply(QDBusError::UnknownMethod,
                                                   QDBusError::errorString(QDBusError::UnknownMethod)));
        return;
    }
    
    QVariantList arguments;
    
    foreach (const QVariant &argument, message.arguments()) {
        if (argument.canConvert<QDBusArgument>()) {
            arguments << QchDBusUtils::dbusArgumentToVariant(argument.value<QDBusArgument>());
        }
        else {
            arguments << argument;
        }
    }
    
    QGenericArgument args[10];
    
    for (int i = 0; i < qMin(arguments.size(), 10); i++) {
        const QVariant &arg = arguments.at(i);
        args[i] = Q_ARG(QVariant, arg);
    }
        
    const QMetaMethod &method = d->object->metaObject()->method(d->objectMethods.value(message.member()));
    
    if (message.isReplyRequired()) {
        QVariant returnArg;
        method.invoke(d->object, Q_RETURN_ARG(QVariant, returnArg), args[0], args[1], args[2], args[3], args[4],
                      args[5], args[6], args[7], args[8], args[9]);
        
        QchDBus::connection(d->object->bus()).send(message.createReply(returnArg));
    }
    else {
        method.invoke(d->object, args[0], args[1], args[2], args[3], args[4], args[5], args[6], args[7], args[8],
                      args[9]);
    }
}

int QchDBusAdaptorInternal::qt_metacall(QMetaObject::Call call, int methodId, void **arguments) {
    Q_D(QchDBusAdaptorInternal);
    const QMetaMethod method = metaObject()->method(methodId);
    qDebug() << method.signature();
    
    if (method.methodType() == QMetaMethod::Signal) {
        const int signalId = metaObject()->indexOfSignal(QMetaObject::normalizedSignature(method.signature()));
        
        if (signalId != -1) {
            QMetaObject::activate(this, metaObject(), signalId, arguments);
            methodId = -1;
        }
    }
    else if (d->object) {
        //handleMessage((*reinterpret_cast< QDBusMessage(*)>(arguments[1])));
        return -1;
    }
    
    return QObject::qt_metacall(call, methodId, arguments);
}

#include "moc_qchdbusadaptor.cpp"
