#ifndef PROCESS_P_H
#define PROCESS_P_H

class Process
{
    Q_OBJECT

    Q_PROPERTY(string command READ command WRITE setCommand)
    Q_PROPERTY(string workingDirectory READ workingDirectory WRITE setWorkingDirectory)
    Q_PROPERTY(int pid READ pid)
    Q_PROPERTY(enumeration error READ error)
    Q_PROPERTY(enumeration state READ state)
    Q_PROPERTY(int exitCode READ exitCode)
    Q_PROPERTY(enumeration exitStatus READ exitStatus)
    Q_PROPERTY(enumeration processChannelMode READ processChannelMode WRITE setProcessChannelMode)
    Q_PROPERTY(jsobject processEnvironment READ processEnvironment WRITE setProcessEnvironment RESET resetProcessEnvironment)
    Q_PROPERTY(enumeration readChannel READ readChannel WRITE setReadChannel)
    Q_PROPERTY(string standardError READ standardError)
    Q_PROPERTY(string standardOutput READ standardOutput)
    Q_PROPERTY(string standardErrorFile READ standardErrorFile WRITE setStandardErrorFile)
    Q_PROPERTY(string standardInputFile READ standardInputFile WRITE setStandardInputFile)
    Q_PROPERTY(string standardOutputFile READ standardOutputFile WRITE setStandardOutputFile)
    Q_PROPERTY(Process standardOutputProcess READ standardOutputProcess WRITE setStandardOutputProcess)

public:
    void start();
    void abort();

signals:
    void started();
    void finished();

/*!
    \class Process

    \brief The Process component is used to start external programs and
    to communicate with them.

    \ingroup org-hildon-utils
    
    \code
    Process {
        id: process
        
        command: "echo 'Hello world!'"
        onFinished: console.log(standardOutput) // "Hello world!"
        
        Component.onCompleted: start()
    }
    \endcode
*/

/*!
    \property string Process::command
    \brief The command to be executed by the process
*/


/*!
    \property string Process::workingDirectory
    \brief The directory from which \link command\endlink is executed
*/

/*!
    \property int Process::pid
    \brief The unique process id
*/

/*!
    \property enumeration Process::error
    \brief The error that last occurred
    
    Possible values are:
    
    \li \c FailedToStart - The process failed to start. 
    Either the invoked program is missing, or you may have 
    insufficient permissions to invoke the program.
    \li \c Crashed - The process crashed some time after starting successfully.
    \li \c WriteError - An error occurred when attempting to write to the process. 
    For example, the process may not be running, or it may have closed its input channel.
    \li \c ReadError - An error occurred when attempting to read from the process. 
    For example, the process may not be running.
    \li \c UnknownError - An unknown error occurred.
    
    The default value is \c UnknownError.
*/

/*!
    \property enumeration Process::state
    \brief The current state of the process
    
    Possible values are:
    
    \li \c NotRunning - The process is not running.
    \li \c Starting - The process is starting, but the 
    program has not yet been invoked.
    \li \c Running - The process is running and is ready for 
    reading and writing.
    
    The default value is \c NotRunning.
*/

/*!
    \property int Process::exitCode
    \brief The last exit code of the process
    
    \sa exitStatus
*/

/*!
    \property enumeration Process::exitStatus
    \brief The last exit status of the process
    
    Possible values are:
    
    \li \c NormalExit - The process exited normally.
    \li \c CrashExit - The process crashed.
    
    \sa exitCode
*/

/*!
    \property enumeration Process::processChannelMode
    \brief The channel mode of the process standard output and standard error channels

    Possible values are:
    
    \li \c SeparateChannels - Process manages the output of the running process, 
    keeping standard output and standard error data in separate internal buffers. 
    You can select the Process's current read channel by setting \link readChannel\endlink.
    \li \c MergedChannels - Process merges the output of the running process 
    into the standard output channel (stdout). The standard error channel (stderr) 
    will not receive any data. The standard output and standard error data of 
    the running process are interleaved.
    \li \c ForwardedChannels - Process forwards the output of the running process 
    onto the main process. Anything the child process writes to its standard output 
    and standard error will be written to the standard output and standard error 
    of the main process.
    
    The default value is \c SeparateChannels.
    
    \sa readChannel
*/

/*!
    \property jsobject Process::processEnvironment
    \brief The environment variables of the process
*/

/*!
    \property enumeration Process::readChannel
    \brief The read channel of the process
    
    Possible values are:
    
    \li \c StandardOutput - The standard output (stdout) of the running process.
    \li \c StandardError - The standard error (stderr) of the running process.
*/

/*!
    \property string Process::standardError
    \brief The output from standard error (stderr)
*/

/*!
    \property string Process::standardOutput
    \brief The output from standard output (stdout)
*/

/*!
    \property string Process::standardErrorFile
    \brief The file to which standard error output is redirected
    
    \sa standardError
*/

/*!
    \property string Process::standardOutputFile
    \brief The file to which standard output is redirected
    
    \sa standardOutput
*/

/*!
    \property string Process::standardInputFile
    \brief The file to which standard input is redirected
*/

/*!
    \property Process Process::standardOutputProcess
    \brief The Process to which standard output is redirected
    
    Setting this property to a valid Process is the equivalent of
    
    \code
    $ process1 | process2
    \endcode
    
    Set the property to \c null to reset.
    
    \sa standardOutput, standardOutputFile
*/

/*!
    \fn void Process::start()
    
    Starts the process. The started() signal will be emitted 
    if the process was started successfully.
    
    \sa started(), state
*/

/*!
    \fn void Process::abort()
    
    Aborts a running process.
*/

/*!
    \fn void Process::started()
    
    This signal is emitted when the process is started.
    
    \sa start(), state
*/


/*!
    \fn void Process::finished()
    
    This signal is emitted when the process is finished.
    
    \sa exitCode, exitStatus, error, state
*/

};

#endif // PROCESS_P_H
