#ifndef DIRECTORY_P_H
#define DIRECTORY_P_H

class Directory
{
    Q_OBJECT

    Q_PROPERTY(string path READ path WRITE setPath)
    Q_PROPERTY(string absolutePath READ absolutePath)
    Q_PROPERTY(string canonicalPath READ canonicalPath)
    Q_PROPERTY(string currentPath READ currentPath WRITE setCurrentPath)
    Q_PROPERTY(string homePath READ homePath CONSTANT)
    Q_PROPERTY(string rootPath READ rootPath CONSTANT)
    Q_PROPERTY(string tempPath READ tempPath CONSTANT)
    Q_PROPERTY(int count READ count)
    Q_PROPERTY(string dirName READ dirName)
    Q_PROPERTY(bool exists READ exists)
    Q_PROPERTY(flags filter READ filter WRITE setFilter)
    Q_PROPERTY(bool absolute READ isAbsolute)
    Q_PROPERTY(bool readable READ isReadable)
    Q_PROPERTY(bool relative READ isRelative)
    Q_PROPERTY(bool root READ isRoot)
    Q_PROPERTY(stringlist nameFilters READ nameFilters WRITE setNameFilters)
    Q_PROPERTY(flags sorting READ sorting WRITE setSorting)

public:
    string absoluteFilePath(string fileName);
    string filePath(string fileName);
    string relativeFilePath(string fileName);

    bool cd(string dirName);
    bool cdUp();

    stringlist entryList();

    bool makeAbsolute();

    bool mkdir(string dirName);
    bool mkpath(string dirPath);
    bool remove(string dirName);
    bool rename(string oldName, string newName);
    bool rmdir(string dirName);
    bool rmpath(string dirPath);

    void refresh();

    string cleanPath(string path);

    bool match(string filter, string fileName);
    bool match(stringlist filters, string fileName);

    stringlist searchPaths(string prefix);
    void addSearchPath(string prefix, string path);
    void setSearchPaths(string prefix, stringlist searchPaths);

/*!
    \class Directory
    \brief The Directory component provides access to directory structures and their contents
    
    \ingroup org-hildon-utils
    
    A Directory is used to manipulate path names, access information
    regarding paths and files, and manipulate the underlying file
    system. It can also be used to access Qt's resource system.

    A Directory can point to a file using either a relative or an absolute
    path. Absolute paths begin with the directory separator.
    
    Relative file names begin with a directory name or a file name and
    specify a path relative to the current directory.
    
    \code
    Page {
        id: page
        
        windowTitle: "Directory page"
        
        Column {
            id: column
            
            anchors.fill: parent
            
            Label {
                text: "Directory"
            }
            
            TextField {
                id: dirField
                
                text: "/home/user/MyDocs/"
            }
            
            Button {
                id: button
                
                text: "List files"
                onClicked: edit.text = dir.entryList().join(", ")
            }
            
            Label {
                text: "Results"
            }
            
            TextEdit {
                id: edit
                
                readOnly: true
            }
        }
        
        Directory {
            id: dir
            
            filter: Directory.Files
            path: dirField.text ? dirField.text : "/home/user/MyDocs/"
        }
    }
    \endcode
*/

/*!
    \property string Directory::path
    \brief The path of the directory
*/

/*!
    \property string Directory::absolutePath
    \brief The absolute path of the directory
*/

/*!
    \property string Directory::canonicalPath
    \brief The canonical path of the directory
*/

/*!
    \property string Directory::path
    \brief The current path of the directory
*/

/*!
    \property string Directory::currentPath
    \brief The current path of the directory
*/

/*!
    \property string Directory::homePath
    \brief The home path of the system
*/

/*!
    \property string Directory::rootPath
    \brief The root path of the system
*/

/*!
    \property string Directory::tempPath
    \brief The temp path of the system
*/

/*!
    \property int Directory::count
    \brief The number of results in the directory
*/

/*!
    \property string Directory::dirName
    \brief The name of the directory
*/

/*!
    \property string Directory::path
    \brief The current path of the directory
*/

/*!
    \property bool Directory::exists
    \brief Whether the directory exists
*/

/*!
    \property flags Directory::filter
    \brief The filter used by entryList()
    
    The value of this property is a bitwise OR combination of 
    the following:
    
    \li \c Dirs - List directories that match the filters.
    \li \c AllDirs - List all directories; i.e. 
    don't apply the filters to directory names.
    \li \c Files - List files.
    \li \c NoSymLinks - Do not list symbolic links.
    \li \c NoDotAndDotDot - Do not list the special entries "." and "..".
    \li \c NoDot - Do not list the special entry ".".
    \li \c NoDotDot - Do not list the special entry "..".
    \li \c AllEntries - List directories, files, drives and symlinks 
    (this does not list broken symlinks unless you specify System).
    \li \c Readable - List files for which the application has read access. 
    The \c Readable value needs to be combined with \c Dirs or \c Files.
    \li \c Writable - List files for which the application has write access. 
    The \c Writable value needs to be combined with \c Dirs or \c Files.
    \li \c Executable - List files for which the application has execute access. 
    The \c Executable value needs to be combined with \c Dirs or \c Files.
    \li \c Hidden - List hidden files.
    \li \c System - List system files.
    \li \c CaseSensitive - The filter should be case sensitive.
*/

/*!
    \property bool Directory::absolute
    \brief Whether the directory's path is absolute
*/

/*!
    \property bool Directory::readable
    \brief Whether the directory is readable
*/

/*!
    \property bool Directory::relative
    \brief Whether the directory's path is relative
*/

/*!
    \property bool Directory::root
    \brief Whether the directory's path is the root path
*/

/*!
    \property stringlist Directory::nameFilters
    \brief The name filters used by entryList()
    
    The name filters are a list of wildcards, e.g:
    
    /code
    nameFilters: [ "*.jpg", "*.jpeg", "*.png", "*.svg" ]
    /endcode
*/

/*!
    \property flags Directory::sorting
    \brief The sorting used by entryList()
    
    The value of this property is a bitwise OR combination of 
    the following:
    
    \li \c Name - Sort by name.
    \li \c Time - Sort by modification time.
    \li \c Size - Sort by file size.
    \li \c Type - Sort by file extension.
    \li \c NoDotAndDotDot - Do not list the special entries "." and "..".
    \li \c Unsorted - Do not sort.
    \li \c NoSort - Not sorted by default.
    \li \c DirsFirst - Put directories first, then the files.
    \li \c DirsLast - Put files first, then the directories. 
    \li \c Reversed - Reverse the sort order. 
    \li \c IgnoreCase - Sort case-insensitively.
    \li \c LocalAware - Sort items appropriately using the current locale settings.
*/

/*!
    \fn string Directory::absolutePath(string fileName)
    
    Returns the absolute path name of a file in the directory. Does \e
    not check if the file actually exists in the directory. 
    Redundant multiple separators or "." and ".."
    directories in \a fileName are not removed (see cleanPath()).

    \sa relativeFilePath(), filePath(), canonicalPath()
*/

/*!
    \fn string Directory::filePath(string fileName)
    
    Returns the path name of a file in the directory. Does \e not
    check if the file actually exists in the directory. 
    If the Directory is relative the returned path name will also
    be relative. Redundant multiple separators or "." and ".."
    directories in \a fileName are not removed (see cleanPath()).

    \sa dirName, absoluteFilePath(), relative, canonicalPath()
*/

/*!
    \fn string Directory::relativeFilePath(string fileName)
    
    Returns the path to \a fileName relative to the directory.

    \sa absoluteFilePath(), filePath(), canonicalPath()
*/

/*!
    \fn bool Directory::cd(string dirName)
    
    Changes the Directory's directory to \a dirName.

    Returns true if the new directory exists and is readable;
    otherwise returns false. Note that the logical cd() operation is
    not performed if the new directory does not exist.

    Calling cd("..") is equivalent to calling cdUp().

    \sa cdUp(), readable, exists, path
*/

/*!
    \fn bool Directory::cdUp()
    
    Changes directory by moving one directory up from the Directory's
    current directory.

    Returns true if the new directory exists and is readable;
    otherwise returns false. Note that the logical cdUp() operation is
    not performed if the new directory does not exist.

    \sa cd(), readable, exists, path
*/

/*!
    \fn stringlist Directory::entryList()
    
    Returns a list of the names of all the files and
    directories in the directory, ordered according to \link filter\endlink 
    and \link nameFilters\endlink, and sorted according to \link sorting\endlink.

    Returns an empty list if the directory is unreadable, does not
    exist, or if nothing matches the specification.

    \sa nameFilters, sorting, filter
*/

/*!
    \fn bool Directory::makeAbsolute()
    
    Converts the directory path to an absolute path. If it is already
    absolute nothing happens. Returns true if the conversion
    succeeded; otherwise returns false.

    \sa absolute, relative, cleanPath()
*/

/*!
    \fn bool Directory::mkdir(string dirName)
    
    Creates a sub-directory called \a dirName.

    Returns true on success; otherwise returns false.

    If the directory already exists when this function is called, it will return false.

    \sa rmdir()
*/

/*!
    \fn bool Directory::rmdir(string dirName)
    
    Removes the directory specified by \a dirName.

    The directory must be empty for rmdir() to succeed.

    Returns true if successful; otherwise returns false.

    \sa mkdir()
*/

/*!
    \fn bool Directory::mkpath(string dirPath)
    
    Creates the directory path \a dirPath.

    The function will create all parent directories necessary to
    create the directory.

    Returns true if successful; otherwise returns false.

    If the path already exists when this function is called, it will return true.

    \sa rmpath()
*/

/*!
    \fn bool Directory::rmpath(string dirPath)
    
    Removes the directory path \a dirPath.

    The function will remove all parent directories in \a dirPath,
    provided that they are empty. This is the opposite of
    mkpath(dirPath).

    Returns true if successful; otherwise returns false.

    \sa mkpath()
*/

/*!
    \fn bool Directory::remove(string fileName)
    
    Removes the file, \a fileName.

    Returns true if the file is removed successfully; otherwise
    returns false.
*/

/*!
    \fn bool Directory::rename(string oldName, string newName)
    
    Renames a file or directory from \a oldName to \a newName, and returns
    true if successful; otherwise returns false.

    On most file systems, rename() fails only if \a oldName does not
    exist, if \a newName and \a oldName are not on the same
    partition or if a file with the new name already exists.
    However, there are also other reasons why rename() can
    fail. For example, on at least one file system rename() fails if
    \a newName points to an open file.
*/

/*!
    \fn void Directory::refresh()
    
    Refreshes the directory information.
*/

/*!
    \fn string Directory::cleanPath(string path)
    
    Removes all multiple directory separators "/" and resolves any
    "."s or ".."s found in the path, \a path.

    Symbolic links are kept. This function does not return the
    canonical path, but rather the simplest version of the input.
    For example, "./local" becomes "local", "local/../bin" becomes
    "bin" and "/local/usr/../bin" becomes "/local/bin".

    \sa absolutePath(), canonicalPath()
*/

/*!
    \fn bool Directory::match(stringlist filters, string fileName)
    \overload

    Returns true if the \a fileName matches any of the wildcard (glob)
    patterns in the list of \a filters; otherwise returns false. The
    matching is case insensitive.

    \sa entryList()
*/

/*!
    \fn bool Directory::match(string filter, string fileName)
    
    Returns true if the \a fileName matches the wildcard (glob)
    pattern \a filter; otherwise returns false. The \a filter may
    contain multiple patterns separated by spaces or semicolons.
    The matching is case insensitive.

    \sa entryList()
*/

/*!
    \fn stringlist Directory::searchPaths(string prefix)

    Returns the search paths for \a prefix.

    \sa setSearchPaths(), addSearchPath()
*/

/*!
    \fn void Directory::addSearchPath(string prefix, string path)

    Adds \a path to the search path for \a prefix.

    \sa setSearchPaths()
*/

/*!
    \fn void Directory::setSearchPaths(string prefix, stringlist searchPaths)

    Sets or replaces Qt's search paths for file names with the prefix \a prefix
    to \a searchPaths.

    To specify a prefix for a file name, prepend the prefix followed by a single
    colon (e.g., "images:undo.png", "xmldocs:books.xml"). \a prefix can only
    contain letters or numbers (e.g., it cannot contain a colon, nor a slash).

    Qt uses this search path to locate files with a known prefix. The search
    path entries are tested in order, starting with the first entry.

    Search paths may contain paths to Qt Resource System.
*/

};

#endif // DIRECTORY_P_H
