/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef SCREEN_P_H
#define SCREEN_P_H

class Screen
{

    Q_PROPERTY(int width READ width)
    Q_PROPERTY(int height READ height)
    Q_PROPERTY(enumeration orientationLock READ orientationLock WRITE setOrientationLock)
    Q_PROPERTY(enumeration currentOrientation READ currentOrientation)
    Q_PROPERTY(bool covered READ covered)

/*!
    \class Screen
    \brief Screen is a context property providing information about the display
    
    \ingroup org-hildon-components
    \ingroup org-hildon-components-properties
    
    The Screen object is accessible as a global 
    context property (named 'screen') in QML, and provides 
    information about the display geometry, orientation and state.
    
    \code
    Page {
        windowTitle: "Screen info"
        
        Column {
        
            Label {
                text: "Width: " + screen.width + " px"
            }
            
            Label {
                text: "Height: " + screen.height + " px"
            }
        
            Label {
                text: "Orientation: " + (screen.currentOrientation == Screen.PortraitOrienation ? "Portrait" : "Landscape")
            }
        }
    }
    \endcode
    
    \note Screen is a context property and cannot be created in QML.
    
    \sa Page, Window
*/

/*!
    \property int Screen::width
    \brief The width of the display
*/

/*!
    \property int Screen::height
    \brief The height of the display
*/

/*!
    \property enumeration Screen::orientationLock
    \brief The allowed orientation(s) of the screen
    
    Possible values are:
    
    \li \c Screen.LandscapeOrientation - The screen is locked to landscape.
    \li \c Screen.PortraitOrientation - The screen is locked to portrait.
    \li \c Screen.AutoOrientation - The screen's orientation changes when 
    the device is rotated.
    
    The default value is \c Screen.LandscapeOrientation.
*/

/*!
    \property enumeration Screen::currentOrientation
    \brief The current orientation of the screen
    
    Possible values are:
    
    \li \c Screen.LandscapeOrientation - The screen is in landscape.
    \li \c Screen.PortraitOrientation - The screen is in portrait.
*/

/*!
    \property bool Screen::covered
    \brief Whether the display is active
    
    This property is useful for suspending timers 
    or user-interface updates when the display is 
    inactive:
    
    \code
    Timer {
        interval: 1000
        repeat: true
        running: !screen.covered
        onTriggered: doSomething()
    }
    \endcode
    
    \code
    Page {
        id: page
        
        windowTitle: "Download progress"
        
        MyDownloadObject {
            id: downloader
        }
        
        ProgressBar {
            id: progressBar
        }
        
        Binding {
            target: progressBar
            property: "value"
            value: downloader.progress
            when: !screen.covered
        }
    }
    \endcode
*/

};

#endif // SCREEN_P_H
