/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef ITEM_P_H
#define ITEM_P_H

class Item
{

    Q_PROPERTY(bool modal READ isModal)
    Q_PROPERTY(enumeration windowModality READ windowModality WRITE setWindowModality)
    Q_PROPERTY(bool enabled READ isEnabled WRITE setEnabled)
    Q_PROPERTY(rect geometry READ geometry WRITE setGeometry)
    Q_PROPERTY(rect frameGeometry READ frameGeometry)
    Q_PROPERTY(rect normalGeometry READ normalGeometry)
    Q_PROPERTY(point pos READ pos WRITE move)
    Q_PROPERTY(size frameSize READ frameSize)
    Q_PROPERTY(size size READ size WRITE resize)
    Q_PROPERTY(rect rect READ rect)
    Q_PROPERTY(rect childrenRect READ childrenRect)
    Q_PROPERTY(size minimumSize READ minimumSize WRITE setMinimumSize)
    Q_PROPERTY(size maximumSize READ maximumSize WRITE setMaximumSize)
    Q_PROPERTY(int x READ x WRITE setX)
    Q_PROPERTY(int y READ y WRITE setY)
    Q_PROPERTY(int width READ width WRITE setFixedWidth)
    Q_PROPERTY(int height READ height WRITE setFixedHeight)
    Q_PROPERTY(int minimumWidth READ minimumWidth WRITE setMinimumWidth)
    Q_PROPERTY(int minimumHeight READ minimumHeight WRITE setMinimumHeight)
    Q_PROPERTY(int maximumWidth READ maximumWidth WRITE setMaximumWidth)
    Q_PROPERTY(int maximumHeight READ maximumHeight WRITE setMaximumHeight)
    Q_PROPERTY(size sizeIncrement READ sizeIncrement WRITE setSizeIncrement)
    Q_PROPERTY(size baseSize READ baseSize WRITE setBaseSize)
    Q_PROPERTY(font font READ font WRITE setFont)
    Q_PROPERTY(bool isActiveWindow READ isActiveWindow)
    Q_PROPERTY(enumeration focusPolicy READ focusPolicy WRITE setFocusPolicy)
    Q_PROPERTY(bool focus READ hasFocus WRITE setFocus)
    Q_PROPERTY(enumeration contextMenuPolicy READ contextMenuPolicy WRITE setContextMenuPolicy)
    Q_PROPERTY(bool visible READ isVisible WRITE setVisible)
    Q_PROPERTY(bool minimized READ isMinimized)
    Q_PROPERTY(bool maximized READ isMaximized)
    Q_PROPERTY(bool fullScreen READ isFullScreen)
    Q_PROPERTY(size sizeHint READ sizeHint)
    Q_PROPERTY(size minimumSizeHint READ minimumSizeHint)
    Q_PROPERTY(string windowTitle READ windowTitle WRITE setWindowTitle)
    Q_PROPERTY(string toolTip READ toolTip WRITE setToolTip)
    Q_PROPERTY(string statusTip READ statusTip WRITE setStatusTip)
    Q_PROPERTY(string whatsThis READ whatsThis WRITE setWhatsThis)
    Q_PROPERTY(bool windowModified READ isWindowModified WRITE setWindowModified)
    Q_PROPERTY(bool autoFillBackground READ autoFillBackground WRITE setAutoFillBackground)
    Q_PROPERTY(string styleSheet READ styleSheet WRITE setStyleSheet)
    Q_PROPERTY(flags inputMethodHints READ inputMethodHints WRITE setInputMethodHints)
    Q_PROPERTY(Item parent READ parentWidget WRITE setParent)
    Q_PROPERTY(real opacity READ opacity WRITE setOpacity)
    Q_PROPERTY(Anchors anchors READ anchors CONSTANT)
    Q_PROPERTY(AnchorLine left READ left CONSTANT)
    Q_PROPERTY(AnchorLine right READ right CONSTANT)
    Q_PROPERTY(AnchorLine top READ top CONSTANT)
    Q_PROPERTY(AnchorLine bottom READ bottom CONSTANT)
    Q_PROPERTY(AnchorLine horizontalCenter READ horizontalCenter CONSTANT)
    Q_PROPERTY(AnchorLine verticalCenter READ verticalCenter CONSTANT)
    Q_PROPERTY(list<Object> data READ data)
    Q_PROPERTY(list<Item> children READ children)
    Q_PROPERTY(list<Object> actions READ actions)
    Q_PROPERTY(Style style READ style WRITE setStyle)
    
signals:
    void customContextMenuRequested(point pos);
    
/*!
    \class Item
    \brief The Item component is the base of all visual components

    \ingroup org-hildon-components
    \ingroup org-hildon-components-basic

    The item is the atom of the user interface: it receives mouse, keyboard
    and other events from the window system, and paints a representation of
    itself on the screen. Every item is rectangular, and they are sorted in a
    Z-order. An item is clipped by its parent and by the items in front of
    it.

    An item that is not embedded in a parent item is called a window.
    Usually, windows have a frame and a title bar, although it is also possible
    to create windows without such decoration using suitable
    window flags. In Qt Components Hildon, Window and the various
    types of Dialog are the most common window types.
*/

/*!
    \property bool Item::autoFillBackground
    \brief Whether the item background is filled automatically

    If enabled, this property will cause Qt to fill the background of the
    item before invoking the paint event.

    This property cannot be turned off (i.e., set to false) if an item's
    parent has a static gradient for its background.

    \warning Use this property with caution in conjunction with
    styleSheet or style. When an item has a style sheet with a valid
    background or a border-image, this property is automatically disabled.

    By default, this property is false.

    \sa styleSheet, style
*/

/*!
    \property string Item::styleSheet
    \brief The item's style sheet

    The style sheet contains a textual description of customizations to the
    item's style.

    \sa style
*/

/*!
    \property Style Item::style
    \brief The item's Style
    
    Setting the style property applies the styling properties of the 
    Style to the item:
    
    \code
    Item {
        id: item
        
        style: Style {
            backgroundImage: "background-image.png"
        }
    }
    \endcode
*/

/*!
    \property bool Item::modal
    \brief Whether the item is a modal item

    This property only makes sense for windows. A modal item
    prevents items in all other windows from getting any input.

    By default, this property is false.

    \sa windowModality, Dialog
*/

/*!
    \property enumeration Item::windowModality
    \brief Which windows are blocked by the modal item

    This property only makes sense for windows. A modal item
    prevents items in other windows from getting input. The value of
    this property controls which windows are blocked when the item
    is visible. Changing this property while the window is visible has
    no effect.

    By default, this property is Qt.NonModal.

    \sa modal, Dialog
*/

/*!
    \property bool Item::minimized
    \brief Whether this item is minimized (iconified)

    This property is only relevant for windows.

    By default, this property is false.

    \sa visible, maximized
*/

/*!
    \property bool Item::maximized
    \brief Whether this item is maximized

    This property is only relevant for windows.

    \note Due to limitations on some window systems, this does not always
    report the expected results (e.g., if the user on X11 maximizes the
    window via the window manager, Qt has no way of distinguishing this
    from any other resize). This is expected to improve as window manager
    protocols evolve.

    By default, this property is false.

    \sa visible, minimized
*/

/*!
    \property bool Item::fullScreen
    \brief Whether the item is shown in full screen mode

    A item in full screen mode occupies the whole screen area and does not
    display window decorations, such as a title bar.

    By default, this property is false.

    \sa minimized, maximized
*/

/*!
    \property bool Item::enabled
    \brief Whether the item is enabled

    In general an enabled item handles keyboard and mouse events; a disabled
    item does not.

    Some items display themselves differently when they are
    disabled. For example a button might draw its label grayed out.

    Disabling a item implicitly disables all its children. Enabling
    respectively enables all child items unless they have been
    explicitly disabled.

    By default, this property is true.
*/

/*!
    \property rect Item::frameGeometry
    \brief geometry of the item relative to its parent including any
    window frame

    By default, this property contains a value that depends on the screen geometry.

    \sa geometry, x, y, pos
*/

/*!
    \property int Item::x

    \brief The x coordinate of the item relative to its parent including
    any window frame

    By default, this property has a value of 0.

    \sa frameGeometry, y, pos
*/

/*!
    \property int Item::y
    \brief The y coordinate of the item relative to its parent and
    including any window frame

    By default, this property has a value of 0.

    \sa frameGeometry, x, pos
*/

/*!
    \property point Item::pos
    \brief The position of the item within its parent item

    If the item is a window, the position is that of the item on
    the desktop, including its frame.

    By default, this property contains a position that refers to the
    origin.

    \sa frameGeometry, size x, y
*/

/*!
    \property rect Item::geometry
    \brief The geometry of the item relative to its parent and
    excluding the window frame

    By default, this property contains a value that depends on the user's
    platform and screen geometry.

    \sa frameGeometry, rect, size, minimumSize, maximumSize
*/

/*!
    \property rect Item::normalGeometry

    \brief The geometry of the item as it will appear when shown as
    a normal (not maximized or full screen) top-level item

    For child items this property always holds an empty rectangle.

    By default, this property contains an empty rectangle.

    \sa windowState, geometry
*/

/*!
    \property size Item::size
    \brief The size of the item excluding any window frame

    The size is adjusted if it lies outside the range defined by
    minimumSize and maximumSize.

    By default, this property contains a value that depends on the user's
    platform and screen geometry.

    \note Setting the size to \c size(0, 0) will cause the item to not
    appear on screen. This also applies to windows.

    \sa pos, geometry, minimumSize, maximumSize
*/

/*!
    \property int Item::width
    \brief The width of the item excluding any window frame

    By default, this property contains a value that depends on the screen geometry.

    \sa geometry, height, size
*/

/*!
    \property int Item::height
    \brief The height of the item excluding any window frame

    By default, this property contains a value that depends on the screen geometry.

    \sa geometry, width, size
*/

/*!
    \property rect Item::rect
    \brief The internal geometry of the item excluding any window
    frame

    The rect property equals rect(0, 0, width, height).

    By default, this property contains a value that depends on the screen geometry.

    \sa size
*/

/*!
    \property rect Item::childrenRect
    \brief The bounding rectangle of the item's children

    Hidden children are excluded.

    By default, for a item with no children, this property contains a
    rectangle with zero width and height located at the origin.

    \sa geometry
*/

/*!
    \property size Item::minimumSize
    \brief The item's minimum size

    The item cannot be resized to a smaller size than the minimum
    item size. The item's size is forced to the minimum size if
    the current size is smaller.

    The minimum size set by this function will override the minimum size
    defined by QLayout. In order to unset the minimum size, use a
    value of \c size(0, 0).

    By default, this property contains a size with zero width and height.

    \sa minimumWidth, minimumHeight, maximumSize, sizeIncrement
*/

/*!
    \property size Item::maximumSize
    \brief The item's maximum size in pixels

    The item cannot be resized to a larger size than the maximum
    item size.

    By default, this property contains a size in which both width and height
    have values of 16777215.

    \sa maximumWidth, maximumHeight, minimumSize, sizeIncrement
*/

/*!
    \property int Item::minimumWidth
    \brief The item's minimum width in pixels

    This property corresponds to the width held by the minimumSize property.

    By default, this property has a value of 0.

    \sa minimumSize, minimumHeight
*/

/*!
    \property int Item::minimumHeight
    \brief The item's minimum height in pixels

    This property corresponds to the height held by the minimumSize property.

    By default, this property has a value of 0.

    \sa minimumSize, minimumWidth
*/

/*!
    \property int Item::maximumWidth
    \brief The item's maximum width in pixels

    This property corresponds to the width held by the maximumSize property.

    By default, this property contains a value of 16777215.

    \sa maximumSize, maximumHeight
*/

/*!
    \property int Item::maximumHeight
    \brief The item's maximum height in pixels

    This property corresponds to the height held by the maximumSize property.

    By default, this property contains a value of 16777215.

    \sa maximumSize, maximumWidth
*/

/*!
    \property size Item::sizeIncrement
    \brief The size increment of the item

    When the user resizes the window, the size will move in steps of
    sizeIncrement.width pixels horizontally and
    sizeIncrement.height pixels vertically, with baseSize as the
    basis.

    Note that while you can set the size increment for all items, it
    only affects windows.

    By default, this property contains a size with zero width and height.

    \sa size, minimumSize, maximumSize
*/

/*!
    \property size Item::baseSize
    \brief The base size of the item

    The base size is used to calculate a proper item size if the
    item defines sizeIncrement.

    By default, for a newly-created item, this property contains a size with
    zero width and height.

    \sa sizeIncrement
*/

/*!
    \property font Item::font
    \brief The font currently set for the item

    This property describes the item's requested font. The font is used by
    the item's style when rendering standard components, and is available as
    a means to ensure that custom items can maintain consistency with the
    native platform's look and feel. It's common that different platforms, or
    different styles, define different fonts for an application.

    When you assign a new font to a item, the properties from this font are
    combined with the item's default font to form the item's final
    font.

    The default depends on the system environment. QApplication maintains a
    system/theme font which serves as a default for all items.  There may
    also be special font defaults for certain types of items. You can also
    define default fonts for items yourself by passing a custom font and the
    name of a item to QApplication::setFont(). Finally, the font is matched
    against Qt's font database to find the best match.

    Item propagates explicit font properties from parent to child. If you
    change a specific property on a font and assign that font to a item,
    that property will propagate to all the item's children, overriding any
    system defaults for that property.

    \note If styleSheet or style is set on the same item as font,
    styles will take precedence if the settings conflict.
*/

/*!
    \property string Item::windowTitle
    \brief The window title (caption)

    This property only makes sense for top-level items, such as
    windows and dialogs.

    If you use the windowModified mechanism, the window title must
    contain a "[*]" placeholder, which indicates where the '*' should
    appear. Normally, it should appear right after the file name
    (e.g., "document1.txt[*] - Text Editor"). If the 
    windowModified property is false (the default), the placeholder
    is simply removed.

    \sa windowModified
*/

/*!
    \property bool Item::focus
    \brief Whether this item (or its focus proxy) has the keyboard
    input focus

    By default, this property is false.
*/

/*!
    \property bool Item::isActiveWindow
    \brief Whether this item's window is the active window

    The active window is the window that contains the item that has
    keyboard focus (The window may still have focus if it has no
    items or none of its items accepts keyboard focus).

    When popup windows are visible, this property is true for both the
    active window \e and for the popup.

    By default, this property is false.
*/

/*!
    \property size Item::frameSize
    \brief The size of the item including any window frame

    By default, this property contains a value that depends on the screen geometry.
*/

/*!
    \property enumeration Item::contextMenuPolicy
    \brief How the item shows a context menu

    The default value of this property is Qt.DefaultContextMenu.
    
    Other values are:
    
    \li \c Qt.NoContextMenu
    \li \c Qt.PreventContextMenu
    \li \c Qt.ActionsContextMenu
    \li \c Qt.CustomContextMenu. 
    
    With Qt.ActionsContextMenu, the context menu contains the item's 
    \link actions\endlink.
    
    With Qt.CustomContextMenu, the signal customContextMenuRequested() is
    emitted.

    \sa actions, customContextMenuRequested()
*/

/*!
    \property enumeration Item::focusPolicy
    \brief The way the item accepts keyboard focus

    The policy is Qt.TabFocus if the item accepts keyboard
    focus by tabbing, Qt.ClickFocus if the item accepts
    focus by clicking, Qt.StrongFocus if it accepts both, and
    Qt.NoFocus (the default) if it does not accept focus at
    all.

    You must enable keyboard focus for a item if it processes
    keyboard events. This is normally done from the item's
    constructor. For instance, the TextField constructor calls
    setFocusPolicy(Qt.StrongFocus).

    If the item has a focus proxy, then the focus policy will
    be propagated to it.

    \sa enabled
*/

/*!
    \property bool Item::visible
    \brief Whether the item is visible

    Setting visible to \e true sets the item to visible
    status if all its parent items up to the window are visible. If
    an ancestor is not visible, the item won't become visible until
    all its ancestors are shown. If its size or position has changed,
    Qt guarantees that a item gets move and resize events just
    before it is shown. If the item has not been resized yet, Qt
    will adjust the item's size to a useful default.

    Setting visible to \e false hides a item explicitly. An
    explicitly hidden item will never become visible, even if all
    its ancestors become visible, unless you show it.

    A item receives show and hide events when its visibility status
    changes. Between a hide and a show event, there is no need to
    waste CPU cycles preparing or displaying information to the user.
    A video application, for example, might simply stop generating new
    frames.

    A item that happens to be obscured by other windows on the
    screen is considered to be visible. The same applies to iconified
    windows and windows that exist on another virtual
    desktop (on platforms that support this concept). A item
    receives spontaneous show and hide events when its mapping status
    is changed by the window system, e.g. a spontaneous hide event
    when the user minimizes the window, and a spontaneous show event
    when the window is restored again.
    
    \sa opacity
*/

/*!
    \property real Item::opacity
    \brief The opacity of the item
    
    This property is also applied to the item's
    visual children.
    
    \sa visible
*/

/*!
    \property size Item::sizeHint
    \brief The recommended size for the item

    If the value of this property is an invalid size, no size is
    recommended.

    The default implementation of sizeHint() returns an invalid size
    if there is no layout for this item, and returns the layout's
    preferred size otherwise.

    \sa minimumSizeHint, minimumSize
*/

/*!
    \property size Item::minimumSizeHint
    \brief The recommended minimum size for the item

    If the value of this property is an invalid size, no minimum size
    is recommended.

    The default minimumSizeHint is an invalid
    size if there is no layout for this item, and returns the
    layout's minimum size otherwise.

    QLayout will never resize a item to a size smaller than the
    minimum size hint unless minimumSize() is set or the size policy is
    set to sizePolicy::Ignore. If minimumSize() is set, the minimum
    size hint will be ignored.

    \sa minimumSize
*/

/*!
    \property Item Item::parent

    Returns the parent of this item, or \c undefined if it does not have any
    parent item.
*/

/*!
    \property flags Item::inputMethodHints
    \brief What input method specific hints the item has.

    This is only relevant for input items. It is used by
    the input method to retrieve hints as to how the input method
    should operate. For example, if the Qt.ImhFormattedNumbersOnly flag
    is set, the input method may change its visual components to reflect
    that only numbers can be entered.

    The default value is Qt.ImhNone.
*/

/*!
    \property bool Item::windowModified
    \brief Whether the document shown in the window has unsaved changes

    A modified window is a window whose content has changed but has
    not been saved to disk.

    The window title must contain a "[*]" placeholder, which
    indicates where the '*' should appear. Normally, it should appear
    right after the file name (e.g., "document1.txt[*] - Text
    Editor"). If the window isn't modified, the placeholder is simply
    removed.
*/

/*!
    \property string Item::toolTip

    \brief The item's tooltip

    Note that by default tooltips are only shown for items that are
    children of the active window. You can change this behavior by
    setting the attribute Qt.WA_AlwaysShowToolTips on the \e window,
    not on the item with the tooltip.

    By default, this property contains an empty string.

    \sa statusTip, whatsThis
*/

/*!
    \property string Item::statusTip
    \brief The item's status tip

    By default, this property contains an empty string.

    \sa toolTip, whatsThis
*/

/*!
    \property string Item::whatsThis

    \brief The item's What's This help text.

    By default, this property contains an empty string.

    \sa toolTip, statusTip
*/

/*!
    \property Anchors Item::anchors
    \brief The item's anchors
*/

/*!
    \property AnchorLine Item::left
    \brief The item's left anchor line
*/

/*!
    \property AnchorLine Item::right
    \brief The item's right anchor line
*/

/*!
    \property AnchorLine Item::top
    \brief The item's top anchor line
*/

/*!
    \property AnchorLine Item::bottom
    \brief The item's bottom anchor line
*/

/*!
    \property list<Object> Item::actions
    \brief The item's actions
    
    Actions can be of type Action or ActionGroup.
*/

/*!
    \property list<Object> Item::data
    \brief The item's child objects
*/

/*!
    \property list<Item> Item::children
    \brief The item's visual children
*/

/*!
    \fn void Item::customContextMenuRequested(point pos)

    This signal is emitted when the item's contextMenuPolicy is
    Qt.CustomContextMenu, and the user has requested a context menu on
    the item. The position \a pos is the position of the context menu
    event that the item receives.

    \sa contextMenuPolicy, Menu
*/

};

#endif // ITEM_P_H
