/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef ACTIONGROUP_P_H
#define ACTIONGROUP_P_H

class ActionGroup
{

    Q_PROPERTY(bool exclusive READ isExclusive WRITE setExclusive)
    Q_PROPERTY(bool enabled READ isEnabled WRITE setEnabled)
    Q_PROPERTY(bool visible READ isVisible WRITE setVisible)

signals:
    void triggered(Action action);
    
/*!
    \class ActionGroup
    \brief The ActionGroup component groups actions together.

    \ingroup org-hildon-components
    \ingroup org-hildon-components-controls

    In some situations it is useful to group Action components together.
    One simple way of achieving this is to group the actions together in
    an action group.

    An ActionGroup emits a triggered() signal when one of its
    actions is chosen. Each action in an action group emits its
    triggered() signal as usual.

    An action group is \e exclusive by default; it ensures that only 
    one checkable action is active at any one time. 
    If you want to group checkable actions without making them
    exclusive, you can turn of exclusiveness by setting the exclusive 
    property to \e false.
    
    A common use case for an action group is to assign it to the 
    \e tools property of a Page or Window to add a filter group to 
    the main menu:
    
    \code
    Page {
        id: page
        
        tools: [
            ActionGroup {
                id: actionGroup
                
                Action {
                    checkable: true
                    checked: true
                    text: "Group One"
                }
                
                Action {
                    checkable: true
                    text: "Group Two"
                }
                
                Action {
                    checkable: true
                    text: "Group Three"
                }
            },
            
            Action {
                text: "Action One"
            },
            
            Action {
                text: "Action Two"
            }
        ]
    }
    \endcode

    \sa Action
*/

/*!
    \property bool ActionGroup::exclusive
    \brief Whether the action group does exclusive checking

    If exclusive is true, only one checkable action in the action group
    can ever be active at any time. If the user chooses another
    checkable action in the group, the one they chose becomes active and
    the one that was active becomes inactive.

    \sa Action::checkable
*/

/*!
    \property bool ActionGroup::enabled
    \brief Whether the action group is enabled

    Each action in the group will be enabled or disabled unless it
    has been explicitly disabled.

    \sa Action::enabled
*/

/*!
    \property bool ActionGroup::visible
    \brief Whether the action group is visible

    Each action in the action group will match the visible state of
    this group unless it has been explicitly hidden.

    \sa Action::visible
*/

/*!
    \fn void ActionGroup::triggered(Action action)

    This signal is emitted when the given \a action in the action
    group is activated by the user; for example, when the user clicks
    a menu option, toolbar button, or presses an action's shortcut key
    combination.

    Connect to this signal for command actions.

    \sa Action::trigger()
*/

};

#endif // ACTIONGROUP_P_H
