/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef ACTION_P_H
#define ACTION_P_H

class Action
{

    Q_PROPERTY(bool checkable READ isCheckable WRITE setCheckable)
    Q_PROPERTY(bool checked READ isChecked WRITE setChecked)
    Q_PROPERTY(string text READ text WRITE setText)
    Q_PROPERTY(string iconText READ iconText WRITE setIconText)
    Q_PROPERTY(string toolTip READ toolTip WRITE setToolTip)
    Q_PROPERTY(string statusTip READ statusTip WRITE setStatusTip)
    Q_PROPERTY(string whatsThis READ whatsThis WRITE setWhatsThis)
    Q_PROPERTY(font font READ font WRITE setFont)
    Q_PROPERTY(bool enabled READ isEnabled WRITE setEnabled)
    Q_PROPERTY(string icon READ iconSource WRITE setIconSource)
    Q_PROPERTY(string shortcut READ shortcutString WRITE setShortcutString)
    Q_PROPERTY(enumeration shortcutContext READ shortcutContext WRITE setShortcutContext)
    Q_PROPERTY(bool autoRepeat READ autoRepeat WRITE setAutoRepeat)
    Q_PROPERTY(bool visible READ isVisible WRITE setVisible)
    Q_PROPERTY(bool iconVisibleInMenu READ isIconVisibleInMenu WRITE setIconVisibleInMenu)
    Q_PROPERTY(enumeration priority READ priority WRITE setPriority)
    
public:
     void trigger();
     void toggle();

signals:
    void triggered(bool checked = false);
    void toggled(bool);
    
/*!
    \class Action
    \brief The Action component provides an abstract user interface
    action that can be inserted into visual items.
    
    \ingroup org-hildon-components
    \ingroup qml-control-components

    In applications many common commands can be invoked via menus,
    toolbar buttons, and keyboard shortcuts. Since the user expects
    each command to be performed in the same way, regardless of the
    user interface used, it is useful to represent each command as
    an \e action.
    
    Actions can be added to menus and toolbars, and will
    automatically keep them in sync. For example, in a word processor,
    if the user presses a Bold toolbar button, the Bold menu item
    will automatically be checked.
    
    Actions can be created as independent objects, but they may
    also be created during the construction of menus.
    
    An Action may contain an icon, menu text, a shortcut, status text,
    "What's This?" text, and a tooltip.
*/

/*!
    \property string Action::icon
    \brief The action's icon source
    
    Specifies the source to be used to load the icon.
    The source can be an icon name, Qt resource, or 
    absolute filepath. Only local paths can be used. 
    Specifying an empty path clears the icon.
*/

/*!
    \property string Action::shortcut
    \brief The action's primary shortcut key
*/

/*!
    \property enumeration Action::shortcutContext
    \brief The context for the action's shortcut
    
    Possible values are:
    
    \li \c Qt.WidgetShortcut - The shortcut is active when its parent item has focus.
    \li \c Qt.WidgetWithChildrenShortcut - The shortcut is active when its parent item, 
    or any of its children has focus. Children which are top-level items, 
    except pop-ups, are not affected by this shortcut context.
    \li \c Qt.WindowShortcut - The shortcut is active when its parent item is a 
    logical subitem of the active top-level window.
    \li \c Qt.ApplicationShortcut - The shortcut is active when one of the applications 
    windows are active.

    The default value is Qt.WindowShortcut.
*/

/*!
    \property font Action::font
    \brief The action's font

    The font property is used to render the text set on the
    Action. The font will can be considered a hint as it will not be
    consulted in all cases based upon application and style.

    By default, this property contains the application's default font.
*/

/*!
    \property string Action::text
    \brief The action's descriptive text

    If the action is added to a menu, the menu option will consist of
    the icon (if there is one), the text, and the shortcut (if there
    is one). If the text is not explicitly set, the action's description 
    icon text will be used as text. There is no default text.

    \sa iconText
*/

/*!
    \property string Action::iconText
    \brief The action's descriptive icon text

    If ToolBar::toolButtonStyle is set to a value that permits text to
    be displayed, the text defined held in this property appears as a
    label in the relevant tool button.

    If the icon text is not explicitly set, the action's normal text will be
    used for the icon text.

    By default, this property contains an empty string.

    \sa toolTip, statusTip
*/

/*!
    \property string Action::toolTip
    \brief The action's tooltip

    This text is used for the tooltip. If no tooltip is specified,
    the action's text is used.

    By default, this property contains the action's text.
*/

/*!
    \property string Action::statusTip
    \brief The action's status tip

    The status tip is displayed on all status bars provided by the
    action's top-level parent item.

    By default, this property contains an empty string.

    \sa toolTip statusText
*/

/*!
    \property string Action::whatsThis
    \brief The action's "What's This?" help text

    The "What's This?" text is used to provide a brief description of
    the action. The text may contain rich text. There is no default
    "What's This?" text.
*/

/*!
    \property bool Action::checkable
    \brief Whether the action is a checkable action

    A checkable action is one which has an on/off state. For example,
    in a word processor, a Bold toolbar button may be either on or
    off. An action which is not a toggle action is a command action;
    a command action is simply executed, e.g. file save.
    By default, this property is false.

    In some situations, the state of one toggle action should depend
    on the state of others. For example, "Left Align", "Center" and
    "Right Align" toggle actions are mutually exclusive. To achieve
    exclusive toggling, add the relevant toggle actions to an
    ActionGroup with the ActionGroup::exclusive property set to
    true.

    \sa checked
*/

/*!
    \property bool Action::checked
    \brief Whether the action is checked.

    Only checkable actions can be checked.  By default, this is false
    (the action is unchecked).

    \sa checkable
*/

/*!
    \property bool Action::enabled
    \brief Whether the action is enabled

    Disabled actions cannot be chosen by the user. They do not
    disappear from menus or toolbars, but they are displayed in a way
    which indicates that they are unavailable. For example, they might
    be displayed using only shades of gray.

    An action will be disabled when all items to which it is added
    are disabled or not visible. When an action is disabled, 
    it is not possible to trigger it through its shortcut.

    By default, this property is true (actions are enabled).

    \sa text
*/

/*!
    \property bool Action::autoRepeat
    \brief Whether the action can auto repeat

    If true, the action will auto repeat when the keyboard shortcut
    combination is held down, provided that keyboard auto repeat is
    enabled on the system.
    The default value is true.
*/

/*!
    \property bool Action::visible
    \brief Whether the action can be seen (in menus and toolbars, for example)

    If \e visible is true the action can be seen (e.g. in menus and
    toolbars) and chosen by the user; if \e visible is false the
    action cannot be seen or chosen by the user.

    Actions which are not visible are \e not grayed out; they do not
    appear at all.

    By default, this property is true (actions are visible).
*/

/*!
    \property bool Action::iconVisibleInMenu
    \brief Whether or not an action should show an icon in a menu

    In some applications, it may make sense to have actions with icons in the
    toolbar, but not in menus. If true, the icon (if valid) is shown in the menu, when it
    is false, it is not shown.

    The default is to follow whether the Qt.AA_DontShowIconsInMenus attribute
    is set for the application. Explicitly settings this property overrides
    the presence (or abscence) of the attribute.

    \sa icon
*/

/*!
    \property enumeration Action::priority

    \brief The actions's priority in the user interface.

    This property can be set to indicate how the action should be prioritized
    in the user interface.

    For instance, when toolbars have the Qt.ToolButtonTextBesideIcon
    mode set, then actions with LowPriority will not show the text
    labels.
    
    Possible values are:
    
    \li \c LowPriority - The action should not be prioritized in
    the user interface.

    \li \c NormalPriority - The default priority.

    \li \c HighPriority - The action should be prioritized in
    the user interface.
*/

/*!
    \fn void Action::trigger()

    Triggers the action. The triggered() signal is emitted.
*/

/*!
    \fn void Action::toggle()

    This is a convenience function for the \e checked property.
    Connect to it to change the checked state to its opposite state.
*/

/*!
    \fn void Action::triggered(bool checked)

    This signal is emitted when an action is activated by the user;
    for example, when the user clicks a menu option, toolbar button,
    or presses an action's shortcut key combination, or when trigger()
    was called. Notably, it is \e not emitted when checked is changed
    programmatically or toggle() is called.

    If the action is checkable, \a checked is true if the action is
    checked, or false if the action is unchecked.

    \sa toggled(), checked
*/

/*!
    \fn void Action::toggled(bool checked)

    This signal is emitted whenever a checkable action changes its
    checked status. This can be the result of a user interaction,
    or because checked was changed programmatically.

    \a checked is true if the action is checked, or false if the
    action is unchecked.

    \sa triggered(), checked
*/

};

#endif // ACTION_P_H
