import QtQuick 1.0
import "scripts/createobject.js" as ObjectCreator

ListModel {
    id: transfers

    property string _DOWNLOAD: "download"
    property string _UPLOAD: "upload"
    property string _IN_PROGRESS: "inProgress"
    property string _PAUSED: "paused"
    property string _QUEUED: "queued"
    property string _FAILED: "failed"
    property string _COMPLETED: "completed"
    property int currentTransfer
    property bool transferInProgress: (DownloadManager.isDownloading) || (YouTube.isUploading)
    property bool transferHasFailed: false
    property variant statusDict

    signal downloadAdded(string videoId)

    Component.onCompleted: {
        transfers.statusDict = { "paused": qsTr("Paused"), "queued": qsTr("Queued"), "inProgress": qsTr("In progress"), "converting": qsTr("Converting"), "failed": qsTr("Failed") }
    }

    function videoInDownloads(videoId) {
        var found = false;
        var i = 0;
        while ((!found) && (i < transfers.count)) {
            if (videoId == transfers.get(i).videoId) {
                found = true;
            }
            i++;
        }
        return found;
    }

    function appendTransfer(transferItem) {
        transfers.append(transferItem);
        if ((transferItem.status == transfers._QUEUED) && (!transfers.transferInProgress)) {
            if (transferItem.type == transfers._DOWNLOAD) {
                DownloadManager.startDownload(transferItem);
            }
            else if (transferItem.youtube) {
                YouTube.uploadVideo(transferItem);
            }
            else {
                Dailymotion.uploadVideo(transferItem);
            }
            transfers.currentTransfer = transfers.count - 1;
        }
        if (transferItem.type == transfers._DOWNLOAD) {
            transfers.downloadAdded(transferItem.videoId);
        }
    }

    function transferCompleted() {
        var transferItem = transfers.get(transfers.currentTransfer);
        if (transferItem.type == transfers._DOWNLOAD) {
            messages.displayMessage(qsTr("Download of") + " '" + transferItem.title + "' " + qsTr("completed"));
            Database.removeStoredDownload(transferItem.videoId);
        }
        else {
            messages.displayMessage(qsTr("Upload of") + " '" + transferItem.title + "' " + qsTr("completed"));
        }
        transfers.remove(transfers.currentTransfer);
        getNextTransfer();
    }

    function transferFailed(reason) {
        transfers.transferHasFailed = true;
        transfers.setProperty(transfers.currentTransfer, "status", transfers._FAILED);
        transfers.setProperty(transfers.currentTransfer, "reason", reason);
        getNextTransfer();
    }

    function resumeTransfer(index) {
        if (transfers.get(index).status == transfers._PAUSED) {
            transfers.setProperty(index, "status", transfers._QUEUED);
        }
    }

    function pauseTransfer(index) {
        if (index == transfers.currentTransfer) {
            if (DownloadManager.isDownloading) {
                DownloadManager.pauseDownload();
            }
            else if ((YouTube.isUploading) || (Dailymotion.isUploading)) {
                messages.displayMessage(qsTr("Uploads cannot be paused once in progress"));
            }
        }
        else {
            transfers.setProperty(index, "status", transfers._PAUSED);
        }
    }

    function cancelTransfer(index) {
        if ((index == transfers.currentTransfer) && (transfers.transferInProgress)) {
            if (DownloadManager.isDownloading) {
                DownloadManager.cancelDownload();
            }
            else if (YouTube.isUploading) {
                YouTube.abortVideoUpload();
            }
            else if (Dailymotion.isUploading) {
                Dailymotion.abortVideoUpload();
            }
        }
        else {
            deleteTransfer(index);
        }
    }

    function deleteTransfer(index) {
        var itemToDelete = transfers.get(index);
        var transferItem;
        var deleted = false;
        var i = 0;
        while ((i < transfers.count) && (!deleted)) {
            transferItem = transfers.get(i);
            if (transferItem == itemToDelete) {
                if (transferItem.type == transfers._DOWNLOAD) {
                    Database.removeStoredDownload(transferItem.videoId);
                    DownloadManager.deleteIncompleteDownload(transferItem.title);
                }
                if (i < transfers.currentTransfer) {
                    transfers.currentTransfer--;
                }
                transfers.remove(i);
            }
            i++;
        }
    }

    function getNextTransfer() {
        if (!transfers.transferInProgress) {
            if (Settings.transferPriority == "uploads") {
                if (!getNextUpload()) {
                    getNextDownload();
                }
            }
            else if (!getNextDownload()) {
                getNextUpload();
            }
        }
    }

    function getNextUpload() {
        var i = 0;
        var found = false;
        while ((i < transfers.count) && (!found)) {
            var transferItem = transfers.get(i);
            if ((transferItem.status == transfers._QUEUED) && (transferItem.type == transfers._UPLOAD)) {
                found = true;
                transfers.currentTransfer = i;
                if (transferItem.youtube) {
                    YouTube.uploadVideo(ObjectCreator.cloneObject(transferItem));
                }
                else {
                    Dailymotion.uploadVideo(ObjectCreator.cloneObject(transferItem));
                }
            }
            i++;
        }
        return found;
    }

    function getNextDownload() {
        var i = 0;
        var found = false;
        while ((i < transfers.count) && (!found)) {
            var transferItem = transfers.get(i);
            if ((transferItem.status == transfers._QUEUED) && (transferItem.type == transfers._DOWNLOAD)) {
                found = true;
                transfers.currentTransfer = i;
                DownloadManager.startDownload(ObjectCreator.cloneObject(transferItem));
            }
            i++;
        }
        return found;
    }
}
