#include "qmlapplicationviewer.h"



#include <qplatformdefs.h> // MEEGO_EDITION_HARMATTAN

#ifdef HARMATTAN_BOOSTER
#include <MDeclarativeCache>
#endif

#if defined(QMLJSDEBUGGER) && QT_VERSION < 0x040800

#include <qt_private/qdeclarativedebughelper_p.h>

#if !defined(NO_JSDEBUGGER)
#include <jsdebuggeragent.h>
#endif
#if !defined(NO_QMLOBSERVER)
#include <qdeclarativeviewobserver.h>
#endif

// Enable debugging before any QDeclarativeEngine is created
struct QmlJsDebuggingEnabler
{
    QmlJsDebuggingEnabler()
    {
        QDeclarativeDebugHelper::enableDebugging();
    }
};

// Execute code in constructor before first QDeclarativeEngine is instantiated
static QmlJsDebuggingEnabler enableDebuggingHelper;

#endif // QMLJSDEBUGGER


class QmlApplicationViewerPrivate
{
    QString mainQmlFile;
    friend class QmlApplicationViewer;
    static QString adjustPath(const QString &path);
};

QString QmlApplicationViewerPrivate::adjustPath(const QString &path)
{
#ifdef Q_OS_UNIX
#ifdef Q_OS_MAC
    if (!QDir::isAbsolutePath(path))
        return QString::fromLatin1("%1/../Resources/%2")
                .arg(QCoreApplication::applicationDirPath(), path);
#else
    const QString pathInInstallDir =
            QString::fromLatin1("%1/../%2").arg(QCoreApplication::applicationDirPath(), path);
    if (QFileInfo(pathInInstallDir).exists())
        return pathInInstallDir;
#endif
#endif
    return path;
}

QmlApplicationViewer::QmlApplicationViewer(QWidget *parent)
    : QDeclarativeView(parent)
    , d(new QmlApplicationViewerPrivate())
{
    connect(engine(), SIGNAL(quit()), SLOT(close()));
    setResizeMode(QDeclarativeView::SizeRootObjectToView);
    // Qt versions prior to 4.8.0 don't have QML/JS debugging services built in
#if defined(QMLJSDEBUGGER) && QT_VERSION < 0x040800
#if !defined(NO_JSDEBUGGER)
    new QmlJSDebugger::JSDebuggerAgent(engine());
#endif
#if !defined(NO_QMLOBSERVER)
    new QmlJSDebugger::QDeclarativeViewObserver(this, this);
#endif
#endif

#ifdef Q_WS_MAEMO_5
    _HILDON_WM_WINDOW_TYPE_STATUS_AREA = XInternAtom(QX11Info::display(), "_HILDON_WM_WINDOW_TYPE_STATUS_AREA", false);
#else
    _HILDON_WM_WINDOW_TYPE_STATUS_AREA = XInternAtom(QX11Info::display(), "_NET_WM_WINDOW_TYPE_DOCK", false);
#endif
    _NET_WM_WINDOW_TYPE = XInternAtom(QX11Info::display(), "_NET_WM_WINDOW_TYPE", false);

    findStatusWindows();

    onOrientationChanged();
    onKeyboardSlideChanged();

    isPressed = false;
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(onOrientationChanged()));

    imageProvider = new ImageProvider();
    engine()->addImageProvider("imageProvider", imageProvider);
    connect(imageProvider, SIGNAL(imageReady()), this, SLOT(onImageReady()));

    QDBusConnection::systemBus().connect("com.nokia.mce",
                                        "/com/nokia/mce/signal",
                                        "com.nokia.mce.signal",
                                        "sig_device_orientation_ind",
                                        this, SLOT(onDBusOrientationChanged(const QString &)));

    QDBusConnection::systemBus().connect("org.freedesktop.Hal",
                                        "/org/freedesktop/Hal/devices/platform_slide",
                                        "org.freedesktop.Hal.Device",
                                        "PropertyModified",
                                        this, SLOT(onKeyboardSlideChanged()));

}

QmlApplicationViewer::~QmlApplicationViewer()
{
    delete d;
    imageProvider->deleteLater();
}

QmlApplicationViewer *QmlApplicationViewer::create()
{
    return new QmlApplicationViewer();
}

void QmlApplicationViewer::setMainQmlFile(const QString &file)
{
    d->mainQmlFile = QmlApplicationViewerPrivate::adjustPath(file);
    setSource(QUrl::fromLocalFile(d->mainQmlFile));
}

void QmlApplicationViewer::addImportPath(const QString &path)
{
    engine()->addImportPath(QmlApplicationViewerPrivate::adjustPath(path));
}

void QmlApplicationViewer::setOrientation(ScreenOrientation orientation)
{
#if defined(Q_OS_SYMBIAN)
    // If the version of Qt on the device is < 4.7.2, that attribute won't work
    if (orientation != ScreenOrientationAuto) {
        const QStringList v = QString::fromAscii(qVersion()).split(QLatin1Char('.'));
        if (v.count() == 3 && (v.at(0).toInt() << 16 | v.at(1).toInt() << 8 | v.at(2).toInt()) < 0x040702) {
            qWarning("Screen orientation locking only supported with Qt 4.7.2 and above");
            return;
        }
    }
#endif // Q_OS_SYMBIAN

    Qt::WidgetAttribute attribute;
    switch (orientation) {
#if QT_VERSION < 0x040702
    // Qt < 4.7.2 does not yet have the Qt::WA_*Orientation attributes
    case ScreenOrientationLockPortrait:
        attribute = static_cast<Qt::WidgetAttribute>(128);
        break;
    case ScreenOrientationLockLandscape:
        attribute = static_cast<Qt::WidgetAttribute>(129);
        break;
    default:
    case ScreenOrientationAuto:
        attribute = static_cast<Qt::WidgetAttribute>(130);
        break;
#else // QT_VERSION < 0x040702
    case ScreenOrientationLockPortrait:
        attribute = Qt::WA_LockPortraitOrientation;
        break;
    case ScreenOrientationLockLandscape:
        attribute = Qt::WA_LockLandscapeOrientation;
        break;
    default:
    case ScreenOrientationAuto:
        attribute = Qt::WA_AutoOrientation;
        break;
#endif // QT_VERSION < 0x040702
    };
    setAttribute(attribute, true);
}

void QmlApplicationViewer::showExpanded()
{
#if defined(Q_OS_SYMBIAN) || defined(MEEGO_EDITION_HARMATTAN) || defined(Q_WS_SIMULATOR)
    showFullScreen();
#elif defined(Q_WS_MAEMO_5)
    showFullScreen();
#else
    show();
#endif
}

QApplication *createApplication(int &argc, char **argv)
{
#ifdef HARMATTAN_BOOSTER
    return MDeclarativeCache::qApplication(argc, argv);
#else
    return new QApplication(argc, argv);
#endif
}

QVariant QmlApplicationViewer::windowId()
{
    return QVariant::fromValue(this->winId());
}

void QmlApplicationViewer::findStatusWindows()
{
    Window root_return, parent_return, *children_return;
    uint n;

    if(XQueryTree(QX11Info::display(), QX11Info::appRootWindow(0), &root_return,
                  &parent_return, &children_return, &n) == 0)
        return;

   // qDebug() << "n: " << n;

    for(int i = n - 1; i >= 0; --i)
    {
        if(getWindowType(children_return[i]) == _HILDON_WM_WINDOW_TYPE_STATUS_AREA)
        {
            statusArea = children_return[i];
            break;
        }
    }

    XFree(children_return);
}
Atom QmlApplicationViewer::getWindowType(const Window &wId)
{
    Atom type;
    ulong n;
    ulong remain;
    uchar *data = NULL;
    int format;

    if(XGetWindowProperty(QX11Info::display(), wId, _NET_WM_WINDOW_TYPE,
                          0, 1, false, XA_ATOM, &type, &format, &n, &remain, &data) != Success)
        return Atom();

    if(data != NULL)
    {
        Atom wtype = ((Atom*)data)[0];
        XFree(data);
        return wtype;
    }
    else return Atom();
}

void QmlApplicationViewer::mousePressEvent(QMouseEvent *event)
{
    QDeclarativeView::mousePressEvent(event);
    isPressed = true;
}

void QmlApplicationViewer::mouseReleaseEvent(QMouseEvent *event)
{
    QDeclarativeView::mouseReleaseEvent(event);
    isPressed = false;
}

void QmlApplicationViewer::showStatusMenu()
{
    XEvent ev;
    memset(&ev, 0, sizeof(ev));
    ev.xclient.type = ClientMessage;
    ev.xclient.window = statusArea;
    ev.xclient.message_type = XInternAtom(QX11Info::display(), "HILDON_STATUS_MENU_SHOW", false);
    ev.xclient.format = 32;
    ev.xclient.data.l[0] = 1;
    XSendEvent(QX11Info::display(), statusArea, False, NoEventMask, &ev);
}

void QmlApplicationViewer::updateStatusArea()
{
    if(!isPressed && _isVisible)
        imageProvider->registerImage(statusArea);
}

void QmlApplicationViewer::onImageReady()
{
    emit updateStatusAreaSignal(statusArea);
}

void QmlApplicationViewer::minimize()
{
    QDBusConnection::sessionBus().send(
                QDBusMessage::createSignal("/", "com.nokia.hildon_desktop", "exit_app_view"));
}

void QmlApplicationViewer::onDBusOrientationChanged(const QString &orientation)//mce update property when system rotation is set to "portrait lock" and  "autorotation";
{
    if(!isKeyboardOpen)
    {
        if(isActiveWindow())
        {
            isPortrait = (orientation == "portrait") ? true : false;
           // qDebug() << "isPortrait: " << (orientation == "portrait");
            rootContext()->setContextProperty("isPortrait", (orientation == "portrait"));
            //TODO set xproperty for qtedger
        }
        pendingOrientation.clear();
    }
    else pendingOrientation = orientation;
}

void QmlApplicationViewer::onKeyboardSlideChanged()
{
    QFile file("/sys/devices/platform/gpio-switch/slide/state");
    if(!file.open(QIODevice::ReadOnly | QIODevice::Text))
        return;

    QTextStream st(&file);
    if(st.readLine() == "open")
    {
        isKeyboardOpen = true;
        bool _false = false;
        rootContext()->setContextProperty("isPortrait", _false);
        pendingOrientation = isPortrait ? "portrait" : "landscape";
    }
    else {
        isKeyboardOpen = false;
        if(!pendingOrientation.isEmpty())
            onDBusOrientationChanged(pendingOrientation);
    }

}


void QmlApplicationViewer::onOrientationChanged()
{
    isPortrait = QApplication::desktop()->height() > QApplication::desktop()->width();

    rootContext()->setContextProperty("isPortrait", isPortrait);
    rootContext()->setContextProperty("isRealPortrait", isPortrait);

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5AutoOrientation); //it can be usefull when phone is in portrait and keyboard is open
#endif
   // qDebug() << "rotation changed";
}

void QmlApplicationViewer::changeEvent(QEvent *event)
{
    if(event->type() == QEvent::ActivationChange) {
        if(isActiveWindow()) {
            _isVisible = true;
#ifdef Q_WS_MAEMO_5
        if(isPortrait)
            setAttribute(Qt::WA_Maemo5PortraitOrientation);
        else
            setAttribute(Qt::WA_Maemo5LandscapeOrientation);
#endif
    updateStatusArea();
        }
        else {
            _isVisible = false;
        }
    }
}
