import QtQuick 1.0
import org.hildon.components 1.0
import "CreateObject.js" as ObjectCreator

Page {
    id: root

    property alias tabs: tabGroup.children
    property alias currentTab: tabGroup.currentTab
    property alias tabGroup: tabGroup

    title: (tabGroup.currentTab === null) || (!tabGroup.currentTab.title) ? qsTr("QML Browser") : tabGroup.currentTab.title
    tools: MenuLayout {

        MenuItem {
            text: qsTr("View tabs")
            onClicked: pageStack.push(Qt.resolvedUrl("TabsPage.qml"))
        }

        MenuItem {
            text: qsTr("New tab")
            onClicked: tabGroup.addTab()
        }

        MenuItem {
            text: qsTr("Reload")
            onClicked: tabGroup.currentTab.reload.trigger()
            visible: tabGroup.currentTab !== null
        }

        MenuItem {
            text: qsTr("Close tab")
            onClicked: tabGroup.closeCurrentTab()
            visible: tabGroup.currentTab !== null
        }

        MenuItem {
            text: qsTr("Settings")
            onClicked: {
                loader.sourceComponent = settingsDialog;
                loader.item.open();
            }
        }

        MenuItem {
            text: qsTr("About")
            onClicked: {
                loader.sourceComponent = aboutDialog;
                loader.item.open();
            }
        }
    }

    TabGroup {
        id: tabGroup

        function addTab(url) {
            var tab = ObjectCreator.createObject(Qt.resolvedUrl("BrowserTab.qml"), tabGroup);
            currentTab = tab;
            urlInput.text = "";
            tab.urlChanged.connect(urlInput.setUrl(url));

            if (url) {
                tab.url = url;
            }
            else {
                urlInput.focus = true;
            }
        }

        function closeCurrentTab() {
            currentTab.destroy();
            currentTab = null;

            if (tabs.length > 1) {
                currentTab = tabs[tabs.length - 2];
            }
        }

        anchors {
            top: parent.top
            left: parent.left
            right: parent.right
            bottom: toolBar.visible ? toolBar.top : parent.bottom
        }
        onCurrentTabChanged: currentTab === null ? urlInput.text = "" : urlInput.text = currentTab.url;
    }

    ToolBar {
        id: toolBar

        visible: (!appWindow.fullScreen) || ((tabGroup.currentTab !== null) && (tabGroup.currentTab.progress < 1))
        platformStyle: ToolBarStyle {
            background: "image://theme/ToolbarSecondaryBackground"
        }

        ToolBarLayout {

            ToolButton {
                iconSource: "/usr/share/icons/hicolor/64x64/hildon/browser_history.png"
                enabled: tabGroup.currentTab !== null
                onClicked: tabGroup.currentTab.back.trigger()
            }

            ToolButton {
                platformIconId: "general_add"
                enabled: tabGroup.currentTab !== null
                onClicked: {
                    loader.sourceComponent = bookmarkDialog;
                    loader.item.name = tabGroup.currentTab.title === "" ? tabGroup.currentTab.url : tabGroup.currentTab.title;
                    loader.item.address = tabGroup.currentTab.url;
                    loader.item.open();
                }
            }

            UrlInputField {
                id: urlInput

                function setUrl(url) {
                    if (url) {
                        urlInput.text = url;
                        urlInput.cursorPosition = 0;
                    }
                }

                function urlFromTextInput(url) {
                    if (url === "") {
                        return url;
                    }
                    if (url[0] === "/") {
                        return "file://" + url;
                    }
                    if (url.indexOf(":") < 0) {
                        if ((url.indexOf(".") < 0) || (url.indexOf(" ") >= 0)) {
                            return "http://google.com/search?q=" + url;
                        }
                        else {
                            return "http://" + url;
                        }
                    }

                    return url;
                }

                width: screen.inPortrait ? stopButton.visible ? 130 : 220 : stopButton.visible ? 450 : 520
                onAccepted: {
                    if (tabGroup.currentTab) {
                        tabGroup.currentTab.url = urlFromTextInput(text);
                    }
                    else {
                        tabGroup.addTab(urlFromTextInput(text));
                    }
                }
            }

            ToolButton {
                id: stopButton

                platformIconId: "general_stop"
                visible: (tabGroup.currentTab !== null) && (tabGroup.currentTab.progress < 1)
                onClicked: tabGroup.currentTab.stop.trigger()
            }

            ToolButton {
                platformIconId: "general_mybookmarks_folder"
                onClicked: pageStack.push(Qt.resolvedUrl("BookmarksPage.qml"))
            }

            ToolButton {
                platformIconId: "general_fullsize"
                enabled: tabGroup.currentTab !== null
                onClicked: appWindow.fullScreen = true
            }
        }
    }

    ToolButton {
        width: 80
        height: 80
        anchors {
            right: parent.right
            bottom: parent.bottom
            margins: -10
        }
        platformIconId: "general_fullsize"
        visible: timer.running
        onClicked: appWindow.fullScreen = false
        platformStyle: ToolButtonStyle {
            background: "toolbutton_background.png"
            pressedBackground: "toolbutton_background.png"
        }

        Timer {
            id: timer

            running: (!toolBar.visible) && (tabGroup.currentTab !== null) && (!tabGroup.currentTab.moving)
            interval: 3000
        }
    }

    Component {
        id: settingsDialog

        SettingsDialog {}
    }

    Component {
        id: bookmarkDialog

        NewBookmarkDialog {}
    }

    Component {
        id: aboutDialog

        QueryDialog {
            modal: false
            icon: "image://theme/icon/general_web"
            titleText: qsTr("About")
            message: "<b><font size='4'>QML Browser 0.0.1</font></b><br><br>" + qsTr("A simple web browser written using Qt Components Hildon. Includes support for tabbed browsing and bookmarks.")
        }
    }

    Loader {
        id: loader
    }
}
