/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "enginemanager.h"

#include <qimsysdebug.h>
#include <qimsysapplicationmanager.h>
#include <qimsysengine.h>
#include <qimsyspluginmanager.h>

#include <QSettings>

class EngineManager::Private : private QObject
{
    Q_OBJECT
public:
    Private(EngineManager *parent);
    ~Private();

private slots:
    void init();
    void currentEngineChanged(const QString &currentEngine);
    void update();

private:
    EngineManager *q;
    QimsysApplicationManager manager;
};

EngineManager::Private::Private(EngineManager *parent)
    : QObject(parent)
    , q(parent)
{
    qimsysDebugIn() << parent;
    q->setCategoryType(Hidden);

    manager.init();
    connect(&manager, SIGNAL(currentEngineChanged(QString)), this, SLOT(currentEngineChanged(QString)));
    metaObject()->invokeMethod(this, "init", Qt::QueuedConnection);
    qimsysDebugOut();
}

EngineManager::Private::~Private()
{
}

void EngineManager::Private::init()
{
    qimsysDebugIn();
    QSettings settings;
    settings.beginGroup(q->metaObject()->className());
    QString defaultEngine = settings.value("Engine").toString();

    foreach(QimsysEngine *engine, QimsysPluginManager::objects<QimsysEngine>()) {
        connect(engine, SIGNAL(enabledChanged(bool)), this, SLOT(update()));
        connect(engine, SIGNAL(availableChanged(bool)), this, SLOT(update()));
        if (defaultEngine == engine->identifier()) {
            if (!engine->isEnabled() || !engine->isAvailable()) {
                defaultEngine.clear();
            }
        }
    }

    // find default
    if (defaultEngine.isNull()) {
        foreach(QimsysEngine *engine, QimsysPluginManager::objects<QimsysEngine>(this)) {
            if (engine->isEnabled() && engine->isAvailable()) {
                defaultEngine = engine->identifier();
                break;
            }
        }
    }

    manager.setCurrentEngine(defaultEngine);
    qimsysDebugOut();
}

void EngineManager::Private::currentEngineChanged(const QString &currentEngine)
{
    qimsysDebugIn() << currentEngine;
    QSettings settings;
    settings.beginGroup(q->metaObject()->className());
    settings.setValue("Engine", currentEngine);
    qimsysDebugOut();
}

void EngineManager::Private::update()
{
    qimsysDebugIn();
    QimsysEngine *engine = qobject_cast<QimsysEngine*>(sender());
    QString currentEngine = manager.currentEngine();
    if (currentEngine.isNull()) {
        if (engine->isAvailable()) {
            manager.setCurrentEngine(engine->identifier());
        }
    } else if (engine->identifier() == currentEngine) {
        bool found = false;
        foreach(QimsysEngine *e, QimsysPluginManager::objects<QimsysEngine>(this)) {
            if (e->isEnabled() && e->isAvailable()) {
                manager.setCurrentEngine(e->identifier());
                found = true;
                break;
            }
        }
        if (!found) {
            manager.setCurrentEngine(QString::null);
        }
    }
    qimsysDebugOut();
}

EngineManager::EngineManager(QObject *parent)
    : QimsysAbstractPluginObject(parent)
{
    qimsysDebugIn() << parent;
    d = new Private(this);
    qimsysDebugOut();
}

EngineManager::~EngineManager()
{
    qimsysDebugIn();
    delete d;
    qimsysDebugOut();
}

#include "enginemanager.moc"
