/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "socialime.h"
#include "socialimesettings.h"
#include "qimsysdebug.h"
#include "translator.h"

#include <QApplication>
#include <QIcon>
#include <QHostInfo>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QSslError>
#include <QUrl>
#include <QSettings>
#include <QTimer>

class SocialIME::Private : private QObject
{
    Q_OBJECT
public:
    Private(SocialIME *parent);
    ~Private();

    void setInputString(const QString &str);
    void convert();
    void resize(int index, int delta);
    void commit(const QimsysConversionItemList &c);

    QimsysSettingsWidget *settings(const QString &hint, QWidget *parent);

private slots:
    void readConversion();

    void error(QNetworkReply::NetworkError err);
    void sslErrors(QList<QSslError> errs);
    void saved();

    void predict();
    void readPrediction();

    void enabledChanged(bool enabled);
    void checkAvailable();

private:
    SocialIME *q;
    QNetworkAccessManager *networkManager;
    QList<int> initialSizes;
    QString user;
    QTimer predictTimer;
    QTimer checkAvailableTimer;

public:
    QString inputString;
    QimsysConversionItemList conversionList;
    QList< QimsysConversionItemList > candidateList;
};

SocialIME::Private::Private(SocialIME *parent)
    : QObject(parent)
    , q(parent)
{
    networkManager = new QNetworkAccessManager(this);

    q->setCategoryType(CanBeNone);
    TR(q, "categoryName", this, QT_TR_NOOP("Input/Conversion Engine"));

    q->setIcon(QIcon(":/icons/socialime.png"));
    TR(q, "name", this, QT_TR_NOOP("Social IME Engine"));
    TR(q, "author", this, QT_TR_NOOP("Tasuku Suzuki"));
    TR(q, "translator", this, QT_TR_NOOP("None"));
    TR(q, "description", this, QT_TR_NOOP("See <a href=\"http://www.social-ime.com/\">http://www.social-ime.com/</a> for detail."));

    QSettings settings;
    settings.beginGroup(q->metaObject()->className());
    user = settings.value("User").toString();

    predictTimer.setSingleShot(true);
    connect(&predictTimer, SIGNAL(timeout()), this, SLOT(predict()));

    connect(&checkAvailableTimer, SIGNAL(timeout()), this, SLOT(checkAvailable()));
    connect(q, SIGNAL(enabledChanged(bool)), this, SLOT(enabledChanged(bool)));
    enabledChanged(q->isEnabled());
}

SocialIME::Private::~Private()
{
}

void SocialIME::Private::enabledChanged(bool enabled)
{
    if (enabled) {
        checkAvailableTimer.start(1 * 60 * 1000);    // 1 mins.
        checkAvailable();
    } else {
        if (checkAvailableTimer.isActive())
            checkAvailableTimer.stop();
    }
}

void SocialIME::Private::checkAvailable()
{
    qimsysDebugIn();
    static QUrl url("http://www.social-ime.com/");
    q->setAvailable(!QHostInfo::fromName(url.host()).error());
    qimsysDebug() << q->isAvailable();
    qimsysDebugOut();
}

void SocialIME::Private::setInputString(const QString &str)
{
    if (inputString == str) return;
    inputString = str;
    if (predictTimer.isActive()) {
        predictTimer.stop();
    }
    if (!inputString.isEmpty()) {
        predictTimer.start(0);
    }
}

void SocialIME::Private::predict()
{
    QString url = QString("http://www.social-ime.com/api2/predict.php?string=%1").arg(inputString);
    url.append(QLatin1String("&charset=UTF8"));
    url.append(QString("&user=%1").arg(user));
    QNetworkRequest request;
    request.setUrl(QUrl(url));
    request.setRawHeader("User-Agent", "QIMSYS");

    QNetworkReply *reply = networkManager->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(readPrediction()));
    connect(reply, SIGNAL(error(QNetworkReply::NetworkError))
            , this, SLOT(error(QNetworkReply::NetworkError)));
    connect(reply, SIGNAL(sslErrors(QList<QSslError>))
            , this, SLOT(sslErrors(QList<QSslError>)));
}

void SocialIME::Private::readPrediction()
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    reply->deleteLater();
    QStringList result = QString::fromUtf8(reply->readAll()).split("\t");
    result.removeAll(QLatin1String("\n"));
    QimsysConversionItemList list;
    foreach(const QString &p, result) {
        QimsysConversionItem e;
        e.from = inputString;
        e.to = p;
        e.index = -1;
        list.append(e);
    }
    emit q->predicted(list);
}

SocialIME::SocialIME(QObject *parent)
    : QimsysEngine(parent)
{
    d = new Private(this);
}

SocialIME::~SocialIME()
{
    delete d;
}

QString SocialIME::identifier() const
{
    return QLatin1String("social ime");
}

void SocialIME::setRawString(const QString &rawString)
{
    Q_UNUSED(rawString);
}

void SocialIME::setInputString(const QString &inputString)
{
    d->setInputString(inputString);
}

void SocialIME::Private::convert()
{
    QString url = QString("http://www.social-ime.com/api/?string=%1").arg(inputString);
    url.append(QLatin1String("&charset=UTF8"));
    url.append(QString("&user=%1").arg(user));
    QNetworkRequest request;
    request.setUrl(QUrl(url));
    request.setRawHeader("User-Agent", "QIMSYS");

    QNetworkReply *reply = networkManager->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(readConversion()));
    connect(reply, SIGNAL(error(QNetworkReply::NetworkError))
            , this, SLOT(error(QNetworkReply::NetworkError)));
    connect(reply, SIGNAL(sslErrors(QList<QSslError>))
            , this, SLOT(sslErrors(QList<QSslError>)));
    initialSizes.clear();
    QApplication::setOverrideCursor(Qt::WaitCursor);
}

void SocialIME::Private::resize(int index, int delta)
{
    if (conversionList[index].from.length() + delta < 1) return;
    if (delta > 0 && index == conversionList.count() - 1) return;
    QString url = QString("http://www.social-ime.com/api/?string=%1").arg(inputString);
    while (initialSizes.count() - 1 > index) {
        initialSizes.takeLast();
    }
    for (int i = 0; i < index; i++) {
        url.append(QString("&resize[%1]=%2").arg(i).arg(conversionList[i].from.length() - initialSizes[i]));
    }
    url.append(QString("&resize[%1]=%2").arg(index).arg(conversionList[index].from.length() - initialSizes[index] + delta));
    url.append(QLatin1String("&charset=UTF8"));
    url.append(QString("&user=%1").arg(user));
    QNetworkRequest request;
    request.setUrl(QUrl(url));
    request.setRawHeader("User-Agent", "QIMSYS");

    QNetworkReply *reply = networkManager->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(readConversion()));
    connect(reply, SIGNAL(error(QNetworkReply::NetworkError))
            , this, SLOT(error(QNetworkReply::NetworkError)));
    connect(reply, SIGNAL(sslErrors(QList<QSslError>))
            , this, SLOT(sslErrors(QList<QSslError>)));
    QApplication::setOverrideCursor(Qt::WaitCursor);
}

void SocialIME::Private::commit(const QimsysConversionItemList &c)
{
    QString url = QString("http://www.social-ime.com/api/?string=%1").arg(inputString);
    if (c.length() != initialSizes.length()) return;
    for (int i = 0; i < initialSizes.length(); i++) {
        url.append(QString("&resize[%1]=%2").arg(i).arg(c[i].from.length() - initialSizes[i]));
    }
    initialSizes.clear();
    for (int i = 0; i < c.length(); i++) {
        url.append(QString("&commit[%1]=%2").arg(i).arg(c[i].index));
    }
    url.append(QString("&user=%1").arg(user));
    QNetworkRequest request;
    request.setUrl(QUrl(url));
    request.setRawHeader("User-Agent", "QIMSYS");

    networkManager->get(request);
}

void SocialIME::Private::readConversion()
{
    conversionList.clear();
    candidateList.clear();
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    reply->deleteLater();
    QStringList all = QString::fromUtf8(reply->readAll()).split("\n");
    QString from = inputString;
    foreach(const QString &line, all) {
        QStringList data = line.split("\t");
        Q_ASSERT(!data.isEmpty());
        if (data.first().isEmpty()) continue;
        QimsysConversionItem conv;
        conv.to = data[0];
        QimsysConversionItemList cand;
        for (int i = 0; i < data.length(); i++) {
            if (data[i].isEmpty()) continue;
            QimsysConversionItem c;
            c.to = data[i];
            c.index = i;
            cand.append(c);

            if (conv.from.isNull() && from.startsWith(data[i])) {
                conv.from = data[i];
                from = from.mid(conv.from.length());
            }

        }
        for (int i = 0; i < cand.count(); i++) {
            cand[i].from = conv.from;
        }
        if (conv.from.isEmpty()) {
            qimsysWarning() << line << data;
        }
        conversionList.append(conv);
        candidateList.append(cand);
    }
    for (int i = initialSizes.count(); i < conversionList.count(); i++) {
        initialSizes.append(conversionList[i].from.length());
    }
    emit q->conversionsChanged(conversionList);
    QApplication::restoreOverrideCursor();
}

void SocialIME::Private::error(QNetworkReply::NetworkError err)
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    reply->deleteLater();
    qimsysWarning() << err;
}

void SocialIME::Private::sslErrors(QList<QSslError> errs)
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    reply->deleteLater();
    qimsysWarning() << errs.count();
}

QimsysSettingsWidget *SocialIME::Private::settings(const QString &hint, QWidget *parent)
{
    Q_UNUSED(hint);
    SocialIMESettings *settings = new SocialIMESettings(q, parent);
    connect(settings, SIGNAL(saved()), this, SLOT(saved()));
    return settings;
}

void SocialIME::Private::saved()
{
    QSettings settings;
    settings.beginGroup(q->metaObject()->className());
    user = settings.value("User").toString();
}

void SocialIME::convert()
{
    d->convert();
}

void SocialIME::resize(int index, int delta)
{
    d->resize(index, delta);
}

QimsysConversionItemList SocialIME::candidates(int index)
{
    QimsysConversionItemList candidates;
    if (index < 0) return candidates;
    if (index > d->conversionList.count() - 1) return candidates;
    candidates = d->candidateList[index];
    return candidates;
}

void SocialIME::commit(const QimsysConversionItemList &conversions)
{
    d->commit(conversions);
}

QimsysSettingsWidget *SocialIME::settings(const QString &hint, QWidget *parent)
{
    return d->settings(hint, parent);
}

#include "socialime.moc"
