/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "qt4immodule.h"

#include "qimsysdebug.h"
#include "qimsysapplicationmanager.h"
#include "qimsyskeymanager.h"
#include "qimsyspreedit.h"

#include <QApplication>
#include <QSettings>
#include <QTextCharFormat>
#include <QProcess>
#include <QTimer>

class Qt4IMModule::Private : private QObject
{
    Q_OBJECT
public:
    Private(Qt4IMModule *parent);
    ~Private();

    void setFocusWidget(QWidget *w);
    void widgetDestroyed(QWidget *w);

public slots:
    void update();

private slots:
    void init();
    void sendPreeditString(const QimsysPreeditItemList &items);
    void sendCommitString(const QString &commitString, uint target);

private:
    Qt4IMModule *q;
    QWidget *focusWidget;

public:
    QimsysKeyManager keyManager;
    QimsysPreedit preedit;
    QimsysApplicationManager manager;
    QMap<QWidget*, WId> wids;

};

Qt4IMModule::Private::Private(Qt4IMModule *parent)
    : QObject(parent)
    , q(parent)
    , focusWidget(0)
{
    qimsysDebugIn() << parent;
    QApplication::setOverrideCursor(Qt::WaitCursor);
    init();
    QApplication::restoreOverrideCursor();
    qimsysDebugOut();
}

Qt4IMModule::Private::~Private()
{
    qimsysDebugIn();
    qimsysDebugOut();
}

void Qt4IMModule::Private::init()
{
    qimsysDebugIn();
    manager.init();
    keyManager.init();
    preedit.init();
    connect(&preedit, SIGNAL(itemsChanged(QimsysPreeditItemList)), this, SLOT(sendPreeditString(QimsysPreeditItemList)));
    connect(&preedit, SIGNAL(committed(QString, uint)), this, SLOT(sendCommitString(QString, uint)));
    qimsysDebugOut();
}

void Qt4IMModule::Private::setFocusWidget(QWidget *w)
{
    if (focusWidget == w) return;
    qimsysDebugIn() << w;
    if (w) {
        focusWidget = w;
        if (!wids.contains(w)) {
            wids[w] = w->winId();
        }
        manager.setFocus(w->winId());
        update();
    } else {
        if (manager.focus() == focusWidget->winId()) {
            manager.setFocus(0);
        }
        focusWidget = w;
    }
    qimsysDebugOut();
}

void Qt4IMModule::Private::widgetDestroyed(QWidget *w)
{
    qimsysDebugIn() << w;
    if (wids.contains(w)) {
        wids.remove(w);
    }
    qimsysDebugOut();
}

void Qt4IMModule::Private::sendPreeditString(const QimsysPreeditItemList &items)
{
    if (!focusWidget || manager.focus() != focusWidget->winId()) return;
    qimsysDebugIn() << items;
    QList<QInputMethodEvent::Attribute> attrs;

    QString preeditString = 0;
    foreach(const QimsysPreeditItem &item, items) {
        bool textFormatAdded = false;
        if (item.cursor > -1) {
            attrs.append(QInputMethodEvent::Attribute(QInputMethodEvent::Cursor, preeditString.length() + item.cursor, !item.selection, QVariant()));
            if (item.selection != 0) {
                QTextCharFormat format;
                format.setForeground(QPalette().brush(QPalette::HighlightedText));
                format.setBackground(QPalette().brush(QPalette::Highlight));
                int start = preeditString.length() + item.cursor;
                start = qMin(start, start + item.selection);
                int len = qAbs(item.selection);
                attrs.append(QInputMethodEvent::Attribute(QInputMethodEvent::TextFormat, start, len, format));
                textFormatAdded = true;
            }
        }
        if (!textFormatAdded)
        {
            QTextCharFormat format;
            format.setUnderlineStyle(item.underline);
            attrs.append(QInputMethodEvent::Attribute(QInputMethodEvent::TextFormat, preeditString.length(), item.text.length(), format));
        }
        preeditString.append(item.text);
    }

    QInputMethodEvent e(preeditString, attrs);
    q->sendEvent(e);
    update();
    qimsysDebugOut();
}

void Qt4IMModule::Private::sendCommitString(const QString &commitString, uint target)
{
    if (!focusWidget || manager.focus() != focusWidget->winId()) return;
    qimsysDebugIn() << commitString << target;

    {
        QList<QInputMethodEvent::Attribute> attrs;
        attrs.append(QInputMethodEvent::Attribute(QInputMethodEvent::Cursor, 0, 1, QVariant()));
        QInputMethodEvent e(QString::null, attrs);
        e.setCommitString(commitString);
        foreach(QWidget *w, wids.keys()) {
            if (wids[w] == target) {
                QApplication::sendEvent(w, &e);
            }
        }
//  q->sendEvent( e );
    }
    metaObject()->invokeMethod(this, "update", Qt::QueuedConnection);
    qimsysDebugOut();
}

void Qt4IMModule::Private::update()
{
    qimsysDebugIn();
    QWidget *widget = q->focusWidget();
    if (widget && q->isComposing()) {
        QRect r = widget->inputMethodQuery(Qt::ImMicroFocus).toRect();
        QPoint topleft = widget->mapToGlobal(QPoint(0, 0));
        r.translate(topleft);
        preedit.setRect(r);
        preedit.setFont(widget->inputMethodQuery(Qt::ImFont).value<QFont>());
        preedit.setCursorPosition(widget->inputMethodQuery(Qt::ImCursorPosition).toInt());
        preedit.setSurroundingText(widget->inputMethodQuery(Qt::ImSurroundingText).toString());
        preedit.setCurrentSelection(widget->inputMethodQuery(Qt::ImCurrentSelection).toString());
        preedit.setMaximumTextLength(widget->inputMethodQuery(Qt::ImMaximumTextLength).toInt());
    }
    qimsysDebugOut();
}

Qt4IMModule::Qt4IMModule(QObject *parent)
    : QInputContext(parent)
{
    qimsysDebugIn() << parent;
    d = new Private(this);
    qimsysDebugOut();
}

Qt4IMModule::~Qt4IMModule()
{
    qimsysDebugIn();
    delete d;
    qimsysDebugOut();
}

bool Qt4IMModule::filterEvent(const QEvent *event)
{
    bool ret = false;
    switch (event->type()) {
    case QEvent::KeyPress:
        {
            qimsysDebugIn() << event;
            const QKeyEvent *e = static_cast<const QKeyEvent*>(event);
            ret = d->keyManager.keyPress(e->text(), e->key(), e->modifiers(), e->isAutoRepeat());
            qimsysDebug() << e->text() << e->key() << e->modifiers() << e->isAutoRepeat() << ret;
            qimsysDebugOut() << ret;
            break;
        }
    case QEvent::KeyRelease:
        {
            qimsysDebugIn() << event;
            const QKeyEvent *e = static_cast<const QKeyEvent*>(event);
            ret = d->keyManager.keyRelease(e->text(), e->key(), e->modifiers(), e->isAutoRepeat());
            qimsysDebug() << e->text() << e->key() << e->modifiers() << e->isAutoRepeat() << ret;
            qimsysDebugOut() << ret;
            break;
        }
    default:
        break;
    }
    update();
    return ret;
}

QFont Qt4IMModule::font() const
{
    QFont ret;
// if( focusWidget() )
// {
//  ret = focusWidget()->inputMethodQuery( Qt::ImFont ).value<QFont>();
// }
    return ret;
}

QString Qt4IMModule::identifierName()
{
    return "qimsys";
}

bool Qt4IMModule::isComposing() const
{
    return d->manager.composing();
}

QString Qt4IMModule::language()
{
    return d->manager.inputLanguage();
}

void Qt4IMModule::update()
{
    d->update();
}

void Qt4IMModule::reset()
{
    qimsysDebugIn();
    d->manager.exec(QimsysApplicationManager::Reset);
    update();
    qimsysDebugOut();
}

void Qt4IMModule::setFocusWidget(QWidget *w)
{
    qimsysDebugIn() << w;
    d->setFocusWidget(w);
    QInputContext::setFocusWidget(w);
    update();
    qimsysDebugOut();
}

void Qt4IMModule::widgetDestroyed(QWidget *w)
{
    d->widgetDestroyed(w);
}

bool Qt4IMModule::available() const
{
    return !d->manager.hasError() && !d->keyManager.hasError() && !d->preedit.hasError();
}

#include "qt4immodule.moc"
