/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <QtTest>
#include <QCoreApplication>

#include "qimsysdebug.h"
#include "qimsyspluginmanager.h"
#include "qimsysengine.h"

extern QString qimsys_plugin_path;

class AnthyTest : public QObject
{
    Q_OBJECT
public:
    AnthyTest(QObject *parent = 0)
        : QObject(parent)
        , anthy(0)
    {
        bool found = false;
        QDir dir(QCoreApplication::applicationDirPath());
        while (!found && dir.cdUp()) {
            foreach(const QString &subdir, dir.entryList(QDir::Dirs)) {
                if (subdir == QLatin1String("bin")) {
                    dir.cd("lib");
                    dir.cd("qimsys");
                    dir.cd("plugins");
                    found = true;
                    break;
                }
            }
        }
        if (found) {
            QPluginLoader loader(dir.absoluteFilePath("libanthy.so"));
            QObject *object = loader.instance();
            if (object) {
                QimsysPlugin *plugin = qobject_cast<QimsysPlugin*>(object);
                if (plugin) {
                    anthy = qobject_cast<QimsysEngine*>(plugin->object(this));
                } else {
                    delete object;
                }
            }
        } else {
            qimsysWarning() << "plugin path not found.";
        }
    }

    ~AnthyTest() {
        delete anthy;
    }

public slots:
    void initTestCase() {
        QVERIFY(anthy);
    }

    void init() {
    }

    void cleanup() {
    }

private slots:
    void split_data() {
        QTest::addColumn<QString>("inputString");
        QTest::addColumn<QimsysConversionItemList>("conversions");

        QTest::newRow("abcdef") << QString::fromUtf8("あいうえお");
        QTest::newRow("My name is nakano.") << QString::fromUtf8("わたしのなまえはなかのです。");
        QTest::newRow("I am a cat.") << QString::fromUtf8("わがはいはねこである。");
        QTest::newRow("I have no name yet.") << QString::fromUtf8("なまえはまだない。");
    }

    void split() {
        QFETCH(QString, inputString);
        anthy->setInputString(inputString);
        connect(anthy, SIGNAL(conversionsChanged(QimsysConversionItemList)), this, SLOT(conversionsChanged(QimsysConversionItemList)));
        waiting = true;
        expected = qVariantFromValue(inputString);
        anthy->convert();
        wait();
        disconnect(anthy, SIGNAL(conversionsChanged(QimsysConversionItemList)), this, SLOT(conversionsChanged(QimsysConversionItemList)));
    }

protected slots:
    void conversionsChanged(const QimsysConversionItemList &conversions) {
        waiting = false;
        QString from;
        QString to;
        foreach (const QimsysConversionItem &i, conversions) {
            from.append(i.from);
            to.append(i.to);
        }
        QCOMPARE(expected.toString(), from);

        // we assume "from" does not equal to "to"
        QVERIFY(expected.toString() != to);
    }

private:
    void wait() {
        QTime timer;
        timer.start();
        while (waiting) {
            QTest::qWait(50);
            if (timer.elapsed() > 250) break;
        }
        QVERIFY(!waiting);
    }
private:
    QimsysEngine *anthy;
    bool waiting;
    QVariant expected;
};

QTEST_MAIN(AnthyTest)
#include "tst_anthy.moc"
