#include "sheet.h"

#include <QTransform>
#include <QLinearGradient>
#include <QColor>
#include <QString>
#include <QFont>
#include <QBrush>
#include <QDebug>

Sheet::Sheet(QWidget *parent, Theme * theme) :
    QWidget(parent)
    , _number(0)
    , _theme(theme)
{
    setAttribute(Qt::WA_OpaquePaintEvent, false);
    setAttribute(Qt::WA_NoSystemBackground, false);
    connect(&_animator, SIGNAL(frameChanged(int)), SLOT(update()));
    _animator.setFrameRange(0, 100);
    _animator.setDuration(600);
    _animator.setCurveShape(QTimeLine::EaseInOutCurve);
}


void Sheet::setTransition(TransitionsEnum tr)
{
    _transition = tr;
}

TransitionsEnum Sheet::transition() const
{
    return _transition;
}

void Sheet::setNumber(int n)
{
    if (_number != n)
    {
        _number = qBound(0, n, 99);
        preparePixmap();
        update();
    }
}

/*
 * force flag is used to redraw time
 * as fast as possible, w/o animation.
 * Useful for some cases, like switching
 * from another homescreen (time should
 * be updated as soon as possible to not
 * confuse user)
 */
void Sheet::flipTo(int n, bool force)
{
    if (_number != n)
    {
        _number = qBound(0, n, 99);
        _lastPixmap = _pixmap;
        preparePixmap();
        if (force)
        {
            preparePixmap();
            update();
        }
        else
        {
            _animator.stop();
            _animator.start();
        }
    }
}

void Sheet::drawFrame(QPainter *p, const QRect &rect)
{
    p->setPen(Qt::NoPen);
    p->setBrush(_theme->getSheetBrush());
    QRect r = rect;
    int cr = _theme->getCornerRadius();
    p->drawRoundedRect(r, cr, cr, Qt::RelativeSize);
    if (_theme->DrawEdge())
    {
        r.adjust(1, 3, -1, -3);
        p->setPen(_theme->EdgeColor());
        p->setBrush(Qt::NoBrush);
        p->drawRoundedRect(r, cr, cr, Qt::RelativeSize);
    }
    if (_theme->DrawMiddleLine())
    {
        p->setPen(_theme->MiddleLineColor());
        int y = rect.top() + rect.height() / 2 - 1;
        p->drawLine(rect.left(), y, rect.right(), y);
    }
}

QPixmap Sheet::drawDigits(int n, const QRect &rect, bool addZero)
{
    QString str = QString::number(n);
    if (str.length() == 1 && addZero)
        str.prepend("0");

    return _theme->drawText(0.55, str, rect, true)
            .scaledToWidth(width(), Qt::SmoothTransformation);
}

void Sheet::preparePixmap()
{
    _pixmap = QPixmap(size());
    _pixmap.fill(Qt::transparent);
    QPainter p;
    p.begin(&_pixmap);
    p.drawPixmap(0, 0, drawDigits(_number, rect(), true));
    p.end();
}

void Sheet::resizeEvent(QResizeEvent*)
{
    preparePixmap();
    update();
}

void Sheet::paintStatic()
{
    QPainter p(this);
    p.fillRect(rect(), Qt::transparent);

    drawFrame(&p, rect());
    p.drawPixmap(0, 0, _pixmap);
}

void Sheet::paintSlide()
{
    QPainter p(this);
    p.fillRect(rect(), Qt::transparent);

    QRect fr = rect();
    drawFrame(&p, fr);
    p.setClipRect(fr);

    int y = height() * _animator.currentFrame() / 100;
    p.drawPixmap(0, y, _lastPixmap);
    p.drawPixmap(0, y - height(), _pixmap);
}

void Sheet::paintFlip()
{
    QPainter p(this);

    p.setRenderHint(QPainter::SmoothPixmapTransform, true);
    p.setRenderHint(QPainter::Antialiasing, true);

    p.fillRect(rect(), Qt::transparent);

    int hw = width() / 2;
    int hh = height() / 2;

    // behind is the new pixmap
    QRect fr = rect();
    drawFrame(&p, fr);
    p.drawPixmap(0, 0, _pixmap);

    int index = _animator.currentFrame();

    if (index <= 50)
    {
        // the top part of the old pixmap is flipping
        int angle = -180 * index / 100;
        QTransform transform;
        transform.translate(hw, hh);
        transform.rotate(angle, Qt::XAxis);
        p.setTransform(transform);
        drawFrame(&p, fr.adjusted(-hw, -hh, -hw, -hh));
        p.drawPixmap(-hw, -hh, _lastPixmap);

        // the bottom part is still the old pixmap
        p.resetTransform();
        p.setClipRect(0, hh, width(), hh);
        drawFrame(&p, fr);
        p.drawPixmap(0, 0, _lastPixmap);
    }
    else
    {
        p.setClipRect(0, hh, width(), hh);

        // the bottom part is still the old pixmap
        drawFrame(&p, fr);
        p.drawPixmap(0, 0, _lastPixmap);

        // the bottom part of the new pixmap is flipping
        int angle = 180 - 180 * _animator.currentFrame() / 100;
        QTransform transform;
        transform.translate(hw, hh);
        transform.rotate(angle, Qt::XAxis);
        p.setTransform(transform);
        drawFrame(&p, fr.adjusted(-hw, -hh, -hw, -hh));
        p.drawPixmap(-hw, -hh, _pixmap);

    }

}

void Sheet::paintRotate()
{
    QPainter p(this);

    QRect fr = rect();
    drawFrame(&p, fr);
    p.setClipRect(fr);

    int angle1 = -180 * _animator.currentFrame() / 100;
    int angle2 = 180 - 180 * _animator.currentFrame() / 100;
    int angle = (_animator.currentFrame() <= 50) ? angle1 : angle2;
    QPixmap pix = (_animator.currentFrame() <= 50) ? _lastPixmap : _pixmap;

    QTransform transform;
    transform.translate(width() / 2, height() / 2);
    transform.rotate(angle, Qt::XAxis);

    p.setTransform(transform);
    p.setRenderHint(QPainter::SmoothPixmapTransform, true);
    p.drawPixmap(-width() / 2, -height() / 2, pix);
}

void Sheet::paintEvent(QPaintEvent *event)
{
    Q_UNUSED(event);
    if (_animator.state() == QTimeLine::Running)
    {
        if (_transition == Transition_None)
            paintStatic();
        if (_transition == Transition_Slide)
            paintSlide();
        if (_transition == Transition_Flip)
            paintFlip();
        if (_transition == Transition_Rotate)
            paintRotate();
    }
    else
    {
        paintStatic();
    }
}

void Sheet::reloadTheme()
{
    preparePixmap();
    update();
}
