#ifndef TRANSFERMODEL_H
#define TRANSFERMODEL_H

#include "enums.h"
#include <QAbstractItemModel>

class Transfer;
class QTimer;

class TransferModel : public QAbstractItemModel
{
    Q_OBJECT

    Q_PROPERTY(int count
               READ rowCount
               NOTIFY countChanged)
    Q_PROPERTY(QString searchQuery
               READ searchQuery
               WRITE setSearchQuery)
    Q_PROPERTY(Transfers::Status statusFilter
               READ statusFilter
               WRITE setStatusFilter)
    Q_PROPERTY(int totalDownloadSpeed
               READ totalDownloadSpeed
               NOTIFY totalDownloadSpeedChanged)
    Q_PROPERTY(int activeTransfers
               READ activeTransfers
               NOTIFY activeTransfersChanged)

public:
    ~TransferModel();

#if (QT_VERSION >= 0x050000) || (QT_VERSION < 0x040600)
    QHash<int, QByteArray> roleNames() const;
#endif
    Qt::DropActions supportedDropActions() const;
    Qt::ItemFlags flags(const QModelIndex &index) const;

    int rowCount(const QModelIndex &parent = QModelIndex()) const;
    int columnCount(const QModelIndex &parent = QModelIndex()) const;

    QModelIndex index(int row, int column, const QModelIndex &parent = QModelIndex()) const;
    QModelIndex parent(const QModelIndex &child) const;

    QVariant data(const QModelIndex &index, int role = Qt::DisplayRole) const;
    Q_INVOKABLE QVariant data(int row, int parentRow, const QByteArray &role) const;
    Q_INVOKABLE QVariant data(const QString &id, const QByteArray &role) const;

    QMap<int, QVariant> itemData(const QModelIndex &index) const;
    Q_INVOKABLE QVariantMap itemData(int row, int parentRow) const;
    Q_INVOKABLE QVariantMap itemData(const QString &id) const;
    Q_INVOKABLE QVariantList allItemData() const;

    bool setData(const QModelIndex &index, const QVariant &value, int role);
    Q_INVOKABLE bool setData(int row, int parentRow, const QVariant &value, const QByteArray &role);
    Q_INVOKABLE bool setData(const QString &id, const QVariant &value, const QByteArray &role);

    Transfer* get(const QModelIndex &index) const;
    Q_INVOKABLE Transfer* get(int row, int parentRow) const;
    Q_INVOKABLE Transfer* get(const QString &id) const;

    QModelIndexList match(const QModelIndex &start, int role, const QVariant &value, int hits = -1, Qt::MatchFlags flags = Qt::MatchExactly) const;

    QString searchQuery() const;
    Transfers::Status statusFilter() const;

    int totalDownloadSpeed() const;

    int activeTransfers() const;

    static TransferModel* instance();

public slots:
    void setSearchQuery(const QString &query);
    void setStatusFilter(Transfers::Status status);

    void addTransfer(const QUrl &url, const QString &service, const QString &fileName);

    bool start();
    bool pause();
    bool start(const QString &id);
    bool pause(const QString &id);
    bool cancel(const QString &id);

    void storeTransfers();
    void restoreStoredTransfers();

private:
    TransferModel();

    void filter();
    QString generateTransferId(QString seedFileName) const;
    void getNextTransfers();

    void addActiveTransfer(Transfer *transfer);
    void removeActiveTransfer(Transfer *transfer);

private slots:
    void startNextTransfers();
    void onTransferDataChanged(int role);
    void onTransferStatusChanged(Transfers::Status status);
    void onMaximumConcurrentTransfersChanged(int oldMaximum, int newMaximum);
    void onTransfersRestored(const QList<Transfer*> &transfers);

signals:
    void countChanged(int count);
    void totalDownloadSpeedChanged(int speed);
    void activeTransfersChanged(int active);

private:
    static TransferModel *m_instance;

    Transfer* m_rootItem;
    QTimer *m_queueTimer;
    QList<Transfer*> m_activeTransfers;
    QList<Transfer*> m_filteredTransfers;
    QString m_searchQuery;
    Transfers::Status m_statusFilter;
    QHash<int, QByteArray> m_roleNames;
};

#endif // TRANSFERMODEL_H
