#include "archivepasswordsdialog.h"
#include "../shared/database.h"
#include <QPushButton>
#include <QLabel>
#include <QLineEdit>
#include <QGridLayout>
#include <QStringListModel>
#include <QListView>
#include <QMenu>

ArchivePasswordsDialog::ArchivePasswordsDialog(QWidget *parent) :
    QDialog(parent),
    m_model(new QStringListModel(Database::instance()->getArchivePasswords(), this)),
    m_view(new QListView(this)),
    m_contextMenu(new QMenu(this)),
    m_removePasswordAction(m_contextMenu->addAction(tr("Remove"), this, SLOT(removeArchivePassword()))),
    m_passwordEdit(new QLineEdit(this)),
    m_passwordButton(new QPushButton(tr("Add"), this))
{
    this->setWindowTitle(tr("Archive passwords"));
    this->setAttribute(Qt::WA_DeleteOnClose, true);

    m_view->setModel(m_model);
    m_view->setEditTriggers(QListView::NoEditTriggers);
    m_view->setContextMenuPolicy(Qt::CustomContextMenu);
    m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    m_passwordEdit->setPlaceholderText(tr("Add password"));

    m_passwordButton->setFixedWidth(150);
    m_passwordButton->setEnabled(false);

    QGridLayout *grid = new QGridLayout(this);
    grid->addWidget(m_view, 0, 0);
    grid->addWidget(m_passwordEdit, 1, 0);
    grid->addWidget(m_passwordButton, 1, 1);

    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_passwordEdit, SIGNAL(textChanged(QString)), this, SLOT(onPasswordEditTextChanged(QString)));
    this->connect(m_passwordEdit, SIGNAL(returnPressed()), this, SLOT(addArchivePassword()));
    this->connect(m_passwordButton, SIGNAL(clicked()), this, SLOT(addArchivePassword()));
    this->connect(Database::instance(), SIGNAL(archivePasswordsChanged()), this, SLOT(onArchivePasswordsChanged()));
}

ArchivePasswordsDialog::~ArchivePasswordsDialog() {}

void ArchivePasswordsDialog::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(m_view->mapToGlobal(pos), m_removePasswordAction);
}

void ArchivePasswordsDialog::onPasswordEditTextChanged(const QString &text) {
    m_passwordButton->setEnabled(!text.isEmpty());
}

void ArchivePasswordsDialog::addArchivePassword() {
    QMetaObject::invokeMethod(Database::instance(), "addArchivePassword", Qt::QueuedConnection, Q_ARG(QString, m_passwordEdit->text()));
    m_passwordEdit->clear();
}

void ArchivePasswordsDialog::removeArchivePassword() {
    if (m_view->currentIndex().isValid()) {
        QMetaObject::invokeMethod(Database::instance(), "removeArchivePassword", Qt::QueuedConnection, Q_ARG(QString, m_view->currentIndex().data().toString()));
    }
}

void ArchivePasswordsDialog::onArchivePasswordsChanged() {
    m_model->setStringList(Database::instance()->getArchivePasswords());
}
