#ifndef TRANSFERMODEL_H
#define TRANSFERMODEL_H

#include "transferitem.h"
#include <QAbstractTableModel>
#include <QSharedPointer>

class TransferModel : public QAbstractTableModel
{
    Q_OBJECT

    Q_ENUMS(TransferModel::Roles)

    Q_PROPERTY(int count
               READ rowCount
               NOTIFY countChanged)
    Q_PROPERTY(bool rowsSelected
               READ rowsSelected
               NOTIFY rowsSelectedChanged)

public:
    enum Roles {

        FileNameRole = Qt::UserRole + 1,
        CategoryRole,
        DownloadPathRole,
        StatusRole,
        StatusTextRole,
        StatusInfoRole,
        ProgressRole,
        SizeRole,
        PriorityRole,
        PriorityTextRole,
        ConvertibleToAudioRole,
        SaveAsAudioRole,
        ServiceNameRole,
        ServiceIconRole,
        EnteredRole,
        SelectedRole
    };

public:
    explicit TransferModel(QObject *parent = 0);
    ~TransferModel();
    Qt::DropActions supportedDropActions() const;
    int rowCount(const QModelIndex &parent = QModelIndex()) const;
    int columnCount(const QModelIndex &parent = QModelIndex()) const;
    QVariant headerData(int section, Qt::Orientation orientation = Qt::Horizontal, int role = Qt::DisplayRole) const;
    Q_INVOKABLE QVariant data(const QModelIndex &index, int role) const;
    Q_INVOKABLE bool setData(const QModelIndex &index, const QVariant &value, int role);
    bool setData(int row, const QVariant &value, int role);
    QSharedPointer<TransferItem> getTransfer(int row) const;
    QSharedPointer<TransferItem> getTransfer(TransferItem *transfer) const;
    Q_INVOKABLE TransferItem* getTransferForQML(int row) const;
    Q_INVOKABLE inline QList<int> selectedRows() const { return m_selectedRows; }
    inline bool rowsSelected() const { return !this->selectedRows().isEmpty(); }

public slots:
    void addTransfer(QSharedPointer<TransferItem> transfer);
    void addTransfers(QList< QSharedPointer<TransferItem> > transfers);
    void removeTransfer(QSharedPointer<TransferItem> transfer);
    void moveTransfer(int sourceRow, int destinationRow);
    void toggleRowSelected(int row);
    void selectRow(int row);
    void deselectRow(int row);
    void selectAllRows();
    void clearSelectedRows();
    void setRowEntered(int row);
    void pauseSelectedTransfers();
    void resumeSelectedTransfers();
    void cancelSelectedTransfers();
    void moveSelectedTransfers(int destinationRow);

private:
    void clearTransfers();
    int indexOf(TransferItem *transfer) const;

private slots:
    void onTransferStatusChanged();
    void onTransferProgressChanged();
    void onTransferSizeChanged();
    void onTransferFileNameChanged();
    void onTransferCategoryChanged();
    void onTransferPriorityChanged();

signals:
    void alert(const QString &message);
    void countChanged(int count);
    void rowsSelectedChanged(bool selected);

private:
    QList< QSharedPointer<TransferItem> > m_list;
    QList<int> m_selectedRows;
    int m_enteredRow;
};

#endif // TRANSFERMODEL_H
