#include "transfermodel.h"
#include <QDeclarativeEngine>

TransferModel::TransferModel(QObject *parent) :
    QAbstractTableModel(parent),
    m_enteredRow(-1)
{
    QHash<int, QByteArray> roles;
    roles[FileNameRole] = "fileName";
    roles[CategoryRole] = "category";
    roles[DownloadPathRole] = "downloadPath";
    roles[StatusRole] = "status";
    roles[StatusTextRole] = "statusText";
    roles[StatusInfoRole] = "statusInfo";
    roles[ProgressRole] = "progress";
    roles[SizeRole] = "size";
    roles[PriorityRole] = "priority";
    roles[PriorityTextRole] = "priorityText";
    roles[ConvertibleToAudioRole] = "convertibleToAudio";
    roles[SaveAsAudioRole] = "saveAsAudio";
    roles[ServiceNameRole] = "serviceName";
    roles[ServiceIconRole] = "serviceIcon";
    roles[EnteredRole] = "entered";
    roles[SelectedRole] = "selected";

    this->setRoleNames(roles);
}

TransferModel::~TransferModel() {
    this->clearTransfers();
}

Qt::DropActions TransferModel::supportedDropActions() const {
    return Qt::MoveAction;
}

int TransferModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

int TransferModel::columnCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return 6;
}

QVariant TransferModel::headerData(int section, Qt::Orientation orientation, int role) const {
    if ((role != Qt::DisplayRole) || (orientation != Qt::Horizontal)) {
        return QVariant();
    }

    switch (section) {
    case 0:
        return tr("Details");
    case 1:
        return tr("Category");
    case 2:
        return tr("Priority");
    case 3:
        return tr("Size");
    case 4:
        return tr("Progress");
    case 5:
        return tr("Status");
    default:
        return QVariant();
    }
}

QVariant TransferModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case FileNameRole:
        return m_list.at(index.row()).data()->fileName();
    case CategoryRole:
        return m_list.at(index.row()).data()->category();
    case StatusRole:
        return m_list.at(index.row()).data()->status();
    case StatusTextRole:
        return m_list.at(index.row()).data()->statusText();
    case StatusInfoRole:
        return m_list.at(index.row()).data()->statusInfo();
    case ProgressRole:
        return m_list.at(index.row()).data()->progress();
    case SizeRole:
        return m_list.at(index.row()).data()->size();
    case PriorityRole:
        return m_list.at(index.row()).data()->priority();
    case PriorityTextRole:
        return m_list.at(index.row()).data()->priorityText();
    case ConvertibleToAudioRole:
        return m_list.at(index.row()).data()->convertibleToAudio();
    case SaveAsAudioRole:
        return m_list.at(index.row()).data()->saveAsAudio();
    case ServiceNameRole:
        return m_list.at(index.row()).data()->service();
    case ServiceIconRole:
        return m_list.at(index.row()).data()->iconName();
    case EnteredRole:
        return m_enteredRow == index.row();
    case SelectedRole:
        return QVariant(m_selectedRows.contains(index.row()));
    default:
        return QVariant();
    }
}

bool TransferModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    if (!index.isValid()) {
        return false;
    }

    switch (role) {
    case FileNameRole:
        m_list.at(index.row()).data()->setFileName(value.toString());
        break;
    case CategoryRole:
        m_list.at(index.row()).data()->setCategory(value.toString());
        break;
    case DownloadPathRole:
        m_list.at(index.row()).data()->setDownloadPath(value.toString());
        break;
    case StatusRole:
        m_list.at(index.row()).data()->setStatus(static_cast<Transfers::Status>(value.toInt()));
        break;
    case PriorityRole:
        m_list.at(index.row()).data()->setPriority(static_cast<Transfers::Priority>(value.toInt()));
        break;
    case SaveAsAudioRole:
        m_list.at(index.row()).data()->setSaveAsAudio(value.toBool());
        break;
    default:
        return false;
    }

    return true;
}

bool TransferModel::setData(int row, const QVariant &value, int role) {
    return setData(this->index(row, 0), value, role);
}

void TransferModel::addTransfer(QSharedPointer<TransferItem> transfer) {
    this->beginInsertRows(QModelIndex(), rowCount(), rowCount());
    m_list.append(transfer);
    this->endInsertRows();

    this->connect(transfer.data(), SIGNAL(progressChanged(int)), this, SLOT(onTransferProgressChanged()));
    this->connect(transfer.data(), SIGNAL(sizeChanged(qint64)), this, SLOT(onTransferSizeChanged()));
    this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::Status)), this, SLOT(onTransferStatusChanged()));
    this->connect(transfer.data(), SIGNAL(statusInfoChanged(QString)), this, SLOT(onTransferStatusChanged()));
    this->connect(transfer.data(), SIGNAL(fileNameChanged(QString)), this, SLOT(onTransferFileNameChanged()));
    this->connect(transfer.data(), SIGNAL(categoryChanged(QString)), this, SLOT(onTransferCategoryChanged()));
    this->connect(transfer.data(), SIGNAL(priorityChanged(Transfers::Priority)), this, SLOT(onTransferPriorityChanged()));

    emit countChanged(this->rowCount());
}

void TransferModel::addTransfers(QList< QSharedPointer<TransferItem> > transfers) {
    this->beginInsertRows(QModelIndex(), rowCount(), rowCount() + transfers.size() - 1);

    while (!transfers.isEmpty()) {
        QSharedPointer<TransferItem> transfer = transfers.takeFirst();
        m_list.append(transfer);

        this->connect(transfer.data(), SIGNAL(progressChanged(int)), this, SLOT(onTransferProgressChanged()));
        this->connect(transfer.data(), SIGNAL(sizeChanged(qint64)), this, SLOT(onTransferSizeChanged()));
        this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::Status)), this, SLOT(onTransferStatusChanged()));
        this->connect(transfer.data(), SIGNAL(statusInfoChanged(QString)), this, SLOT(onTransferStatusChanged()));
        this->connect(transfer.data(), SIGNAL(fileNameChanged(QString)), this, SLOT(onTransferFileNameChanged()));
        this->connect(transfer.data(), SIGNAL(categoryChanged(QString)), this, SLOT(onTransferCategoryChanged()));
        this->connect(transfer.data(), SIGNAL(priorityChanged(Transfers::Priority)), this, SLOT(onTransferPriorityChanged()));
    }

    this->endInsertRows();

    emit countChanged(this->rowCount());
}

void TransferModel::removeTransfer(QSharedPointer<TransferItem> transfer) {
    int row = m_list.indexOf(transfer);

    if (row >= 0) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.removeAt(row);
        this->endRemoveRows();

        emit countChanged(this->rowCount());
    }
}

void TransferModel::onTransferStatusChanged() {
    TransferItem *transfer = qobject_cast<TransferItem*>(sender());
    int row = this->indexOf(transfer);

    emit dataChanged(this->index(row, 5), this->index(row, 5));
}

void TransferModel::onTransferProgressChanged() {
    TransferItem *transfer = qobject_cast<TransferItem*>(sender());
    int row = this->indexOf(transfer);

    emit dataChanged(this->index(row, 4), this->index(row, 4));
}

void TransferModel::onTransferSizeChanged() {
    TransferItem *transfer = qobject_cast<TransferItem*>(sender());
    int row = this->indexOf(transfer);

    emit dataChanged(this->index(row, 3), this->index(row, 3));
}

void TransferModel::onTransferFileNameChanged() {
    TransferItem *transfer = qobject_cast<TransferItem*>(sender());
    int row = this->indexOf(transfer);

    emit dataChanged(this->index(row, 0), this->index(row, 0));
}

void TransferModel::onTransferCategoryChanged() {
    TransferItem *transfer = qobject_cast<TransferItem*>(sender());
    int row = this->indexOf(transfer);

    emit dataChanged(this->index(row, 1), this->index(row, 1));
}

void TransferModel::onTransferPriorityChanged() {
    TransferItem *transfer = qobject_cast<TransferItem*>(sender());
    int row = this->indexOf(transfer);

    emit dataChanged(this->index(row, 2), this->index(row, 2));
}

QSharedPointer<TransferItem> TransferModel::getTransfer(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return m_list.at(row);
    }

    return QSharedPointer<TransferItem>();
}

QSharedPointer<TransferItem> TransferModel::getTransfer(TransferItem *transfer) const {
    int i = this->indexOf(transfer);

    if (i >= 0) {
        return m_list.at(i);
    }

    return QSharedPointer<TransferItem>();
}

TransferItem* TransferModel::getTransferForQML(int row) const {
    if (TransferItem *transfer = this->getTransfer(row).data()) {
        QDeclarativeEngine::setObjectOwnership(transfer, QDeclarativeEngine::CppOwnership);
        return transfer;
    }

    return 0;
}

void TransferModel::selectRow(int row) {
    m_selectedRows.append(row);

    emit dataChanged(this->index(row, 0), this->index(row, 4));
    emit rowsSelectedChanged(this->rowsSelected());
}

void TransferModel::deselectRow(int row) {
    m_selectedRows.removeOne(row);

    emit dataChanged(this->index(row, 0), this->index(row, 4));
    emit rowsSelectedChanged(this->rowsSelected());
}

void TransferModel::toggleRowSelected(int row) {
    if (m_selectedRows.contains(row)) {
        this->deselectRow(row);
    }
    else {
        this->selectRow(row);
    }

    emit rowsSelectedChanged(this->rowsSelected());
}

void TransferModel::selectAllRows() {
    m_selectedRows.clear();

    for (int i = 0; i < m_list.size(); i++) {
        m_selectedRows.append(i);
    }

    emit dataChanged(this->index(0, 0), this->index(this->rowCount() - 1, 4));
    emit rowsSelectedChanged(true);
}

void TransferModel::clearSelectedRows() {
    m_selectedRows.clear();

    emit dataChanged(this->index(0, 0), this->index(this->rowCount() - 1, 4));
    emit rowsSelectedChanged(false);
}

void TransferModel::setRowEntered(int row) {
    m_enteredRow = row;

    emit dataChanged(this->index(row - 1, 0), this->index(row + 1, 4));
}

void TransferModel::pauseSelectedTransfers() {
    QList<int> rows = this->selectedRows();
    qSort(rows.begin(), rows.end(), qLess<int>());

    foreach (int i, rows) {
        this->setData(i, Transfers::Paused, StatusRole);
    }
}

void TransferModel::resumeSelectedTransfers() {
    QList<int> rows = this->selectedRows();
    qSort(rows.begin(), rows.end(), qLess<int>());

    foreach (int i, rows) {
        this->setData(i, Transfers::Queued, StatusRole);
    }
}

void TransferModel::cancelSelectedTransfers() {
    QList<int> rows = this->selectedRows();
    qSort(rows.begin(), rows.end(), qGreater<int>());

    foreach (int i, rows) {
        this->setData(i, Transfers::Cancelled, StatusRole);
    }
}

void TransferModel::moveSelectedTransfers(int destinationRow) {
    Q_ASSERT((destinationRow >= 0) || (destinationRow < this->rowCount()));

    QList<int> rows = this->selectedRows();
    qSort(rows.begin(), rows.end(), qLess<int>());

    int movedUp = 0;
    int movedDown = 0;

    foreach (int sourceRow, rows) {
        if (sourceRow < destinationRow) {
            this->moveTransfer(sourceRow - movedDown, destinationRow);
            movedDown++;
        }
        else if (sourceRow > destinationRow){
            this->moveTransfer(sourceRow, destinationRow + movedUp);
            movedUp++;
        }
    }
}

void TransferModel::moveTransfer(int sourceRow, int destinationRow) {
    Q_ASSERT((sourceRow >= 0) && (sourceRow < this->rowCount()) && (destinationRow >= 0) && (destinationRow < this->rowCount()));

    if (sourceRow == destinationRow) {
        return;
    }

    if (sourceRow < destinationRow) {
        destinationRow--;
        emit layoutAboutToBeChanged();
        m_list.move(sourceRow, destinationRow);
        emit layoutChanged();
        m_list.at(destinationRow).data()->setRow(destinationRow);

        for (int i = sourceRow; i < destinationRow; i++) {
            m_list.at(i).data()->setRow(i - 1);
        }
    }
    else {
        emit layoutAboutToBeChanged();
        m_list.move(sourceRow, destinationRow);
        emit layoutChanged();
        m_list.at(destinationRow).data()->setRow(destinationRow);

        for (int i = destinationRow + 1; i <= sourceRow; i++) {
            m_list.at(i).data()->setRow(i + 1);
        }
    }
}

int TransferModel::indexOf(TransferItem *transfer) const {
    for (int i = 0; i < this->rowCount(); i++) {
        if (m_list.at(i).data() == transfer) {
            return i;
        }
    }

    return -1;
}

void TransferModel::clearTransfers() {
    m_list.clear();

    emit countChanged(this->rowCount());
}
