#include "settings.h"
#include <qplatformdefs.h>
#include <QDir>
#include <QDesktopServices>
#include <QNetworkAccessManager>
#include <QNetworkProxy>
#include <QAuthenticator>

Settings::Settings(QObject *parent) :
    QSettings(QString("QDL"), QString("QDL"), parent) {
}

Settings::~Settings() {
    this->saveSettings();
}

void Settings::restoreSettings() {
    this->beginGroup("Transfers");
    this->setStartTransfersAutomatically(value("startTransfersAutomatically", true).toBool());

#if (defined Q_WS_MAEMO_5) || (defined MEEGO_EDITION_HARMATTAN)
    this->setDownloadPath(value("downloadPath", QString("/home/user/MyDocs/QDL/")).toString());
#elif (defined Q_OS_SYMBIAN)
    this->setDownloadPath(value("downloadPath", QString("E:/QDL/")).toString());
#else
    this->setDownloadPath(value("downloadPath", QDesktopServices::storageLocation(QDesktopServices::HomeLocation).append("/QDL/")).toString());
#endif

    this->setMaximumConcurrent(value("maximumConcurrent", 1).toInt());
    this->setExtractDownloadedArchives(value("extractDownloadedArchives", false).toBool());
    this->setDeleteExtractedArchives(value("deleteExtractedArchives", false).toBool());
    this->endGroup();

    this->beginGroup("Decaptcha");
    this->setDecaptchaService(value("decaptchaService", QString()).toString());
    this->endGroup();

    this->beginGroup("System");
    this->setLanguage(value("language", QString("en")).toString());
    this->setMonitorClipboard(value("monitorClipboard", false).toBool());
    this->setScreenOrientation(static_cast<ScreenOrientation::Orientation>(value("screenOrientation", 0).toInt()));
    this->setNetworkProxyType(static_cast<NetworkProxyType::ProxyType>(value("networkProxyType", NetworkProxyType::ProxyType(NetworkProxyType::HttpProxy)).toInt()));
    this->setNetworkProxyHostName(value("networkProxyHostName", QString()).toString());
    this->setNetworkProxyPort(value("networkProxyPort", 80).toUInt());
    this->setNetworkProxyUser(value("networkProxyUser", QString()).toString());
    this->setNetworkProxyPassword(QString(QByteArray::fromBase64(value("networkProxyPassword", QByteArray()).toByteArray())));
    this->setNetworkProxy();
    this->endGroup();
}

void Settings::saveSettings() {
    this->beginGroup("Transfers");
    this->setValue("startTransfersAutomatically", this->startTransfersAutomatically());
    this->setValue("downloadPath", this->downloadPath());
    this->setValue("maximumConcurrent", this->maximumConcurrent());
    this->setValue("extractDownloadedArchives", this->extractDownloadedArchives());
    this->setValue("deleteExtractedArchives", this->extractDownloadedArchives());
    this->endGroup();

    this->beginGroup("Decaptcha");
    this->setValue("decaptchaService", this->decaptchaService());
    this->endGroup();

    this->beginGroup("System");
    this->setValue("language", this->language());
    this->setValue("monitorClipboard", this->monitorClipboard());
    this->setValue("screenOrientation", ScreenOrientation::Orientation(this->screenOrientation()));
    this->setValue("networkProxyType", NetworkProxyType::ProxyType(this->networkProxyType()));
    this->setValue("networkProxyHostName", this->networkProxyHostName());
    this->setValue("networkProxyPort", this->networkProxyPort());
    this->setValue("networkProxyUser", this->networkProxyUser());
    this->setValue("networkProxyPassword", this->networkProxyPassword().toUtf8().toBase64());
    this->endGroup();
}

QVariant Settings::setting(const QString &key) const {
    return this->value(key);
}

void Settings::setSetting(const QString &key, const QVariant &value) {
    this->setValue(key, value);
}

void Settings::setStartTransfersAutomatically(bool automatic) {
    if (automatic != this->startTransfersAutomatically()) {
        m_automatic = automatic;
        emit startTransfersAutomaticallyChanged(automatic);
    }
}

void Settings::setDownloadPath(const QString &path) {
    if (path != this->downloadPath()) {
        m_path = path;

        if (!m_path.endsWith("/")) {
            m_path.append("/");
        }

        emit downloadPathChanged(this->downloadPath());
    }
}

void Settings::setLanguage(const QString &lang) {
    if (lang != this->language()) {
        m_language = lang;
        emit languageChanged(lang);
    }
}

void Settings::setMaximumConcurrent(int maximum) {
    int oldMaximum = this->maximumConcurrent();

    if ((maximum != oldMaximum) && (maximum > 0) && (maximum <= 5)) {
        m_maxConcurrent = maximum;
        emit maximumConcurrentChanged(oldMaximum, maximum);
    }
}

void Settings::setMonitorClipboard(bool monitor) {
    if (monitor != this->monitorClipboard()) {
        m_monitorClipboard = monitor;
        emit monitorClipboardChanged(monitor);
    }
}

void Settings::setDecaptchaService(const QString &service) {
    if (service != this->decaptchaService()) {
        m_decaptchaService = service;
        emit decaptchaServiceChanged(service);
    }
}

void Settings::setScreenOrientation(ScreenOrientation::Orientation orientation) {
    if (orientation != this->screenOrientation()) {
        m_orientation = orientation;
        emit screenOrientationChanged(orientation);
        emit screenOrientationStringChanged(this->screenOrientationString());
    }
}

QString Settings::screenOrientationString() const {
    switch (this->screenOrientation()) {
    case ScreenOrientation::Automatic:
        return tr("Automatic");
    case ScreenOrientation::LockLandscape:
        return tr("Landscape");
    case ScreenOrientation::LockPortrait:
        return tr("Portrait");
    default:
        return tr("Automatic");
    }
}

QString Settings::networkProxyTypeString() const {
    switch (this->networkProxyType()) {
    case NetworkProxyType::Socks5Proxy:
        return QString("SOCKS5");
    case NetworkProxyType::HttpProxy:
        return QString("HTTP");
    case NetworkProxyType::HttpCachingProxy:
        return QString("HTTP %1").arg(tr("caching"));
    default:
        return QString("HTTP");
    }
}

void Settings::setExtractDownloadedArchives(bool extract) {
    if (extract != this->extractDownloadedArchives()) {
        m_extractArchives = extract;
        emit extractDownloadedArchivesChanged(extract);
    }
}

void Settings::setDeleteExtractedArchives(bool deleteExtracted) {
    if (deleteExtracted != this->deleteExtractedArchives()) {
        m_deleteArchives = deleteExtracted;
        emit deleteExtractedArchivesChanged(deleteExtracted);
    }
}

void Settings::setNetworkProxyType(NetworkProxyType::ProxyType proxyType) {
    if (proxyType != this->networkProxyType()) {
        m_proxyType = proxyType;
        emit networkProxyTypeChanged(proxyType);
        emit networkProxyTypeStringChanged(this->networkProxyTypeString());
    }
}

void Settings::setNetworkProxyHostName(const QString &hostName) {
    if (hostName != this->networkProxyHostName()) {
        m_proxyHost = hostName;
        emit networkProxyHostNameChanged(hostName);
    }
}

void Settings::setNetworkProxyPort(quint16 port) {
    if (port != this->networkProxyPort()) {
        m_proxyPort = port;
        emit networkProxyPortChanged(port);
    }
}

void Settings::setNetworkProxyUser(const QString &user) {
    if (user != this->networkProxyUser()) {
        m_proxyUser = user;
        emit networkProxyUserChanged(user);
    }
}

void Settings::setNetworkProxyPassword(const QString &password) {
    if (password != this->networkProxyPassword()) {
        m_proxyPass = password;
        emit networkProxyPasswordChanged(password);
    }
}

void Settings::setNetworkProxy() {
    QNetworkProxy proxy;

    if (!this->networkProxyHostName().isEmpty()) {
        proxy.setHostName(this->networkProxyHostName());
        proxy.setType(QNetworkProxy::ProxyType(this->networkProxyType()));

        if (this->networkProxyPort() > 0) {
            proxy.setPort(this->networkProxyPort());
        }
        else {
            proxy.setPort(80);
        }

        if ((!this->networkProxyUser().isEmpty()) && (!this->networkProxyPassword().isEmpty())) {
            proxy.setUser(this->networkProxyUser());
            proxy.setPassword(this->networkProxyPassword());
        }
    }

    QNetworkProxy::setApplicationProxy(proxy);
}

void Settings::onNetworkProxyAuthenticationRequested(const QNetworkProxy &proxy, QAuthenticator *authenticator) {
    Q_UNUSED(proxy)

    authenticator->setUser(this->networkProxyUser());
    authenticator->setPassword(this->networkProxyPassword());
}
