#include "transferselectionwindow.h"
#include "transferselectiondelegate.h"
#include "transfercategorydialog.h"
#include "transferprioritydialog.h"
#include "transfermovedialog.h"
#include "listview.h"
#include "../shared/session.h"
#include <QMaemo5EditBar>
#include <QPushButton>
#include <QVBoxLayout>

TransferSelectionWindow::TransferSelectionWindow(Session *session, const QString &title, const QString &buttonText, const char* slot, QWidget *parent) :
    QMainWindow(parent),
    m_session(session)
{
    this->setWindowState(Qt::WindowFullScreen);
    this->setAttribute(Qt::WA_Maemo5StackedWindow, true);
    this->setAttribute(Qt::WA_DeleteOnClose, true);
    this->setCentralWidget(new QWidget(this));

    ListView *view = new ListView(this);
    view->setModel(m_session->transferManager());
    view->setItemDelegate(new TransferSelectionDelegate(view));

    QMaemo5EditBar *editBar = new QMaemo5EditBar(title);
    QPushButton *acceptButton = new QPushButton(buttonText, this);
    editBar->addButton(acceptButton);

    QVBoxLayout *vbox = new QVBoxLayout(centralWidget());
    vbox->setContentsMargins(0, 0, 0, 0);
    vbox->setSpacing(0);
    vbox->addWidget(editBar);
    vbox->addWidget(view);

    this->connect(view, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(acceptButton, SIGNAL(clicked()), this, SLOT(onButtonClicked()));
    this->connect(this, SIGNAL(accepted()), this, slot);
}

TransferSelectionWindow::~TransferSelectionWindow() {
    m_session->transferManager()->clearSelectedRows();
}

void TransferSelectionWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        m_session->transferManager()->toggleRowSelected(index.row());
    }
}

void TransferSelectionWindow::onButtonClicked() {
    if (!m_session->transferManager()->selectedRows().isEmpty()) {
        emit accepted();
    }
}

void TransferSelectionWindow::resumeTransfers() {
    foreach (int row, m_session->transferManager()->selectedRows()) {
        m_session->transferManager()->setData(m_session->transferManager()->index(row, 0), Transfers::Queued, TransferModel::StatusRole);
    }

    this->close();
}

void TransferSelectionWindow::pauseTransfers() {
    foreach (int row, m_session->transferManager()->selectedRows()) {
        m_session->transferManager()->setData(m_session->transferManager()->index(row, 0), Transfers::Paused, TransferModel::StatusRole);
    }

    this->close();
}

void TransferSelectionWindow::cancelTransfers() {
    QList<int> rows = m_session->transferManager()->selectedRows();
    qSort(rows.begin(), rows.end(), qGreater<int>());

    foreach (int row, rows) {
        m_session->transferManager()->setData(m_session->transferManager()->index(row, 0), Transfers::Cancelled, TransferModel::StatusRole);
    }

    this->close();
}

void TransferSelectionWindow::showTransferCategoryDialog() {
    TransferCategoryDialog *dialog = new TransferCategoryDialog(m_session, this);
    dialog->setValue(tr("Default"));
    dialog->open();
    this->connect(dialog, SIGNAL(valueChanged(QString)), this, SLOT(setTransferCategories(QString)));
}

void TransferSelectionWindow::setTransferCategories(const QString &category) {
    foreach (int row, m_session->transferManager()->selectedRows()) {
        m_session->transferManager()->setData(m_session->transferManager()->index(row, 0), category, TransferModel::CategoryRole);
    }

    this->close();
}

void TransferSelectionWindow::showTransferPriorityDialog() {
    TransferPriorityDialog *dialog = new TransferPriorityDialog(this);
    dialog->setValue(Transfers::NormalPriority);
    dialog->open();
    this->connect(dialog, SIGNAL(valueChanged(Transfers::Priority)), this, SLOT(setTransferPriorities(Transfers::Priority)));
}

void TransferSelectionWindow::setTransferPriorities(Transfers::Priority priority) {
    foreach (int row, m_session->transferManager()->selectedRows()) {
        m_session->transferManager()->setData(m_session->transferManager()->index(row, 0), priority, TransferModel::PriorityRole);
    }

    this->close();
}

void TransferSelectionWindow::showMoveTransfersDialog() {
    TransferMoveDialog *dialog = new TransferMoveDialog(m_session, this);
    dialog->open();
    this->connect(dialog, SIGNAL(valueChanged(int)), this, SLOT(moveTransfers(int)));
}

void TransferSelectionWindow::moveTransfers(int row) {
    m_session->transferManager()->moveSelectedTransfers(row);
    this->close();
}
