#include "networkproxydialog.h"
#include "networkproxytypeselector.h"
#include "../shared/session.h"
#include <QCheckBox>
#include <QLineEdit>
#include <QGridLayout>
#include <QScrollArea>
#include <QLabel>
#include <QDialogButtonBox>
#include <QIntValidator>

NetworkProxyDialog::NetworkProxyDialog(Session *session, QWidget *parent) :
    QDialog(parent),
    m_session(session),
    m_proxyCheckbox(new QCheckBox(tr("Use network proxy"), this)),
    m_proxyWidget(new QWidget(this)),
    m_proxyTypeSelector(new NetworkProxyTypeSelector(tr("Proxy type"), this)),
    m_hostEdit(new QLineEdit(this)),
    m_portEdit(new QLineEdit(this)),
    m_userEdit(new QLineEdit(this)),
    m_passEdit(new QLineEdit(this))
{
    this->setWindowTitle(tr("Network proxy"));
    this->setAttribute(Qt::WA_DeleteOnClose, true);

    m_hostEdit->setMinimumWidth(380);
    m_portEdit->setValidator(new QIntValidator(0, 100000, this));
    m_passEdit->setEchoMode(QLineEdit::Password);

    QGridLayout *proxyGrid = new QGridLayout(m_proxyWidget);
    proxyGrid->setContentsMargins(0, 0, 0, 0);
    proxyGrid->addWidget(m_proxyTypeSelector, 0, 0, 1, 2);
    proxyGrid->addWidget(new QLabel(tr("Host"), this), 1, 0, 1, 1);
    proxyGrid->addWidget(new QLabel(tr("Port"), this), 1, 1, 1, 1);
    proxyGrid->addWidget(m_hostEdit, 2, 0, 1, 1);
    proxyGrid->addWidget(m_portEdit, 2, 1, 1, 1);
    proxyGrid->addWidget(new QLabel(tr("Username"), this), 3, 0, 1, 2);
    proxyGrid->addWidget(m_userEdit, 4, 0, 1, 2);
    proxyGrid->addWidget(new QLabel(tr("Password"), this), 5, 0, 1, 2);
    proxyGrid->addWidget(m_passEdit, 6, 0, 1, 2);

    QWidget *scrollWidget = new QWidget(this);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);
    vbox->setContentsMargins(0, 0, 0, 0);
    vbox->addWidget(m_proxyCheckbox);
    vbox->addWidget(m_proxyWidget);
    QScrollArea *scrollArea = new QScrollArea(this);
    scrollArea->setWidgetResizable(true);
    scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    scrollArea->setWidget(scrollWidget);

    QDialogButtonBox *buttonBox = new QDialogButtonBox(QDialogButtonBox::Save, Qt::Vertical, this);
    QHBoxLayout *hbox = new QHBoxLayout(this);
    hbox->addWidget(scrollArea);
    hbox->addWidget(buttonBox);

    this->connect(buttonBox, SIGNAL(accepted()), this, SLOT(saveSettings()));
    this->connect(m_proxyCheckbox, SIGNAL(toggled(bool)), m_proxyWidget, SLOT(setEnabled(bool)));
    this->connect(m_proxyCheckbox, SIGNAL(toggled(bool)), this, SLOT(onProxyCheckboxToggled(bool)));

    this->loadSettings();
}

void NetworkProxyDialog::loadSettings() {
    m_proxyCheckbox->setChecked(!m_session->settings()->networkProxyHostName().isEmpty());
    m_proxyTypeSelector->setValue(m_session->settings()->networkProxyType());
    m_hostEdit->setText(m_session->settings()->networkProxyHostName());
    m_portEdit->setText(QString::number(m_session->settings()->networkProxyPort()));
    m_userEdit->setText(m_session->settings()->networkProxyUser());
    m_passEdit->setText(m_session->settings()->networkProxyPassword());
    m_proxyWidget->setEnabled(m_proxyCheckbox->isChecked());
}

void NetworkProxyDialog::saveSettings() {
    m_session->settings()->setNetworkProxyType(m_proxyTypeSelector->currentValue());
    m_session->settings()->setNetworkProxyHostName(m_hostEdit->text());
    m_session->settings()->setNetworkProxyPort(m_portEdit->text().toUInt());
    m_session->settings()->setNetworkProxyUser(m_userEdit->text());
    m_session->settings()->setNetworkProxyPassword(m_passEdit->text());
    m_session->settings()->setNetworkProxy();
    this->accept();
}

void NetworkProxyDialog::onProxyCheckboxToggled(bool checked) {
    if (!checked) {
        m_hostEdit->clear();
        m_portEdit->setText(QString::number(80));
        m_userEdit->clear();
        m_passEdit->clear();
    }
}
