#include "mainwindow.h"
#include "transfertitledelegate.h"
#include "transfercategorydelegate.h"
#include "transferprioritydelegate.h"
#include "transfersizedelegate.h"
#include "transferprogressdelegate.h"
#include "transferstatusdelegate.h"
#include "transfermovedialog.h"
#include "transferprioritydialog.h"
#include "transfercategorydialog.h"
#include "transferdetailsdialog.h"
#include "transferselectionwindow.h"
#include "addurlsdialog.h"
#include "settingsdialog.h"
#include "aboutdialog.h"
#include "../shared/session.h"
#include "checkurlsdialog.h"
#include "treeview.h"
#include <QMenu>
#include <QHeaderView>
#include <QLineEdit>
#include <QFileDialog>
#include <QDesktopServices>
#include <QFile>
#include <QMenuBar>
#include <QVBoxLayout>
#include <QToolBar>

MainWindow::MainWindow(Session *session, QWidget *parent) :
    QMainWindow(parent),
    m_session(session),
    m_urlEdit(new QLineEdit(this)),
    m_addButton(new QAction(QIcon::fromTheme("general_add"), "", this)),
    m_view(new TreeView(this)),
    m_contextMenu(new QMenu(this)),
    m_contextSaveAsAudioAction(m_contextMenu->addAction(tr("Save as audio"), this, SLOT(setTransferSaveAsAudio()))),
    m_contextPauseResumeAction(m_contextMenu->addAction(tr("Pause"), this, SLOT(pauseResumeTransfer()))),
    m_checkDialog(new CheckUrlsDialog(m_session, this))
{
    this->setWindowTitle(tr("QDL"));
    this->setCentralWidget(new QWidget(this));
    this->setAttribute(Qt::WA_Maemo5StackedWindow, true);

    this->menuBar()->addAction(tr("Add URLs"), this, SLOT(showAddUrlsDialog()));
    this->menuBar()->addAction(tr("Import URLs"), this, SLOT(showTextFileDialog()));
    this->menuBar()->addAction(tr("Resume downloads"), this, SLOT(resumeSelectedTransfers()));
    this->menuBar()->addAction(tr("Pause downloads"), this, SLOT(pauseSelectedTransfers()));
    this->menuBar()->addAction(tr("Move downloads"), this, SLOT(moveSelectedTransfers()));
    this->menuBar()->addAction(tr("Cancel downloads"), this, SLOT(cancelSelectedTransfers()));
    this->menuBar()->addAction(tr("Set categories"), this, SLOT(setCategoryOfSelectedTransfers()));
    this->menuBar()->addAction(tr("Set priorities"), this, SLOT(setPriorityOfSelectedTransfers()));
    this->menuBar()->addAction(tr("Settings"), this, SLOT(showSettingsDialog()));
    this->menuBar()->addAction(tr("About"), this, SLOT(showAboutDialog()));

    m_contextMenu->addAction(tr("Move"), this, SLOT(showMoveTransferDialog()));
    m_contextMenu->addAction(tr("Cancel"), this, SLOT(cancelTransfer()));
    m_contextMenu->addAction(tr("Set category"), this, SLOT(showTransferCategoryDialog()));
    m_contextMenu->addAction(tr("Set priority"), this, SLOT(showTransferPriorityDialog()));

    m_contextSaveAsAudioAction->setCheckable(true);

    QToolBar *toolBar = new QToolBar(this);
    toolBar->addWidget(m_urlEdit);
    toolBar->addAction(m_addButton);

    QVBoxLayout *vbox = new QVBoxLayout(this->centralWidget());
    vbox->setContentsMargins(0, 0, 0, 0);
    vbox->addWidget(m_view);
    vbox->addWidget(toolBar);

    m_urlEdit->setPlaceholderText(tr("Add URL"));
    m_addButton->setEnabled(false);

    m_view->setModel(m_session->transferManager());
    m_view->setRootIsDecorated(false);
    m_view->header()->setDefaultAlignment(Qt::AlignCenter);
    m_view->setColumnWidth(0, 260);
    m_view->setColumnWidth(1, 140);
    m_view->setColumnWidth(2, 100);
    m_view->setColumnWidth(3, 90);
    m_view->setColumnWidth(4, 110);
    m_view->setColumnWidth(5, 90);
    m_view->setSelectionBehavior(QTreeView::SelectRows);
    m_view->setContextMenuPolicy(Qt::CustomContextMenu);
    m_view->setEditTriggers(QTreeView::NoEditTriggers);
    m_view->setExpandsOnDoubleClick(false);
    m_view->setItemsExpandable(false);
    m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_view->setItemDelegateForColumn(0, new TransferTitleDelegate(m_view));
    m_view->setItemDelegateForColumn(1, new TransferCategoryDelegate(m_view));
    m_view->setItemDelegateForColumn(2, new TransferPriorityDelegate(m_view));
    m_view->setItemDelegateForColumn(3, new TransferSizeDelegate(m_view));
    m_view->setItemDelegateForColumn(4, new TransferProgressDelegate(m_view));
    m_view->setItemDelegateForColumn(5, new TransferStatusDelegate(m_view));
    m_view->setFocus(Qt::OtherFocusReason);

    this->connect(m_urlEdit, SIGNAL(returnPressed()), this, SLOT(addTransferFromUrlEdit()));
    this->connect(m_urlEdit, SIGNAL(textChanged(QString)), this, SLOT(onUrlTextChanged(QString)));
    this->connect(m_addButton, SIGNAL(triggered()), this, SLOT(addTransferFromUrlEdit()));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_view, SIGNAL(clicked(QModelIndex)), this, SLOT(showTransferDetailsDialog(QModelIndex)));
}

void MainWindow::onUrlTextChanged(const QString &text) {
    m_addButton->setEnabled(!text.isEmpty());
}

void MainWindow::showContextMenu(const QPoint &pos) {
    QModelIndex index = m_view->currentIndex();

    if (!index.isValid()) {
        return;
    }

    m_contextSaveAsAudioAction->setEnabled(index.data(TransferModel::ConvertibleToAudioRole).toBool());
    m_contextSaveAsAudioAction->setChecked((m_contextSaveAsAudioAction->isEnabled()) && (index.data(TransferModel::SaveAsAudioRole).toBool()));

    Transfers::Status status = static_cast<Transfers::Status>(index.data(TransferModel::StatusRole).toInt());

    switch (status) {
    case Transfers::Paused:
        m_contextPauseResumeAction->setText(tr("Resume"));
        break;
    case Transfers::Failed:
        m_contextPauseResumeAction->setText(tr("Resume"));
        break;
    default:
        m_contextPauseResumeAction->setText(tr("Pause"));
    }

    m_contextMenu->popup(pos, m_contextPauseResumeAction);
}

void MainWindow::pauseResumeTransfer() {
    QModelIndex index = m_view->currentIndex();

    if (!index.isValid()) {
        return;
    }

    Transfers::Status status = static_cast<Transfers::Status>(index.data(TransferModel::StatusRole).toInt());

    switch (status) {
    case Transfers::Paused:
        m_session->transferManager()->setData(index, Transfers::Queued, TransferModel::StatusRole);
        break;
    case Transfers::Failed:
        m_session->transferManager()->setData(index, Transfers::Queued, TransferModel::StatusRole);
        break;
    default:
        m_session->transferManager()->setData(index, Transfers::Paused, TransferModel::StatusRole);
    }
}

void MainWindow::showMoveTransferDialog() {
    TransferMoveDialog *dialog = new TransferMoveDialog(m_session, this);
    dialog->open();
    this->connect(dialog, SIGNAL(valueChanged(int)), this, SLOT(moveTransfer(int)));
}

void MainWindow::moveTransfer(int row) {
    m_session->transferManager()->moveTransfer(m_view->currentIndex().row(), row);
}

void MainWindow::cancelTransfer() {    
    m_session->transferManager()->setData(m_view->currentIndex(), Transfers::Cancelled, TransferModel::StatusRole);
}

void MainWindow::setTransferSaveAsAudio() {
    m_session->transferManager()->setData(m_view->currentIndex(), m_contextSaveAsAudioAction->isChecked(), TransferModel::SaveAsAudioRole);
}

void MainWindow::showTransferPriorityDialog() {
    TransferPriorityDialog *dialog = new TransferPriorityDialog(this);
    dialog->setValue(static_cast<Transfers::Priority>(m_view->currentIndex().data(TransferModel::PriorityRole).toInt()));
    dialog->open();
    this->connect(dialog, SIGNAL(valueChanged(Transfers::Priority)), this, SLOT(setTransferPriority(Transfers::Priority)));
}

void MainWindow::setTransferPriority(Transfers::Priority priority) {
    m_session->transferManager()->setData(m_view->currentIndex(), priority, TransferModel::PriorityRole);
}

void MainWindow::showTransferCategoryDialog() {
    TransferCategoryDialog *dialog = new TransferCategoryDialog(m_session, this);
    dialog->setValue(m_view->currentIndex().data(TransferModel::CategoryRole).toString());
    dialog->open();
    this->connect(dialog, SIGNAL(valueChanged(QString)), this, SLOT(setTransferCategory(QString)));
}

void MainWindow::setTransferCategory(const QString &category) {
    m_session->transferManager()->setData(m_view->currentIndex(), category, TransferModel::CategoryRole);
}

void MainWindow::showTransferDetailsDialog(const QModelIndex &index) {
    if (index.isValid()) {
        TransferDetailsDialog *dialog = new TransferDetailsDialog(m_session, m_session->transferManager()->getTransfer(index.row()), this);
        dialog->open();
    }
}

void MainWindow::moveSelectedTransfers() {
    TransferSelectionWindow *window = new TransferSelectionWindow(m_session, tr("Select downloads to move"), tr("Move"), SLOT(showMoveTransfersDialog()), this);
    window->show();
}

void MainWindow::resumeSelectedTransfers() {
    TransferSelectionWindow *window = new TransferSelectionWindow(m_session, tr("Select downloads to resume"), tr("Resume"), SLOT(resumeTransfers()), this);
    window->show();
}

void MainWindow::pauseSelectedTransfers() {
    TransferSelectionWindow *window = new TransferSelectionWindow(m_session, tr("Select downloads to pause"), tr("Pause"), SLOT(pauseTransfers()), this);
    window->show();
}

void MainWindow::cancelSelectedTransfers() {
    TransferSelectionWindow *window = new TransferSelectionWindow(m_session, tr("Select downloads to cancel"), tr("Cancel"), SLOT(cancelTransfers()), this);
    window->show();
}

void MainWindow::setCategoryOfSelectedTransfers() {
    TransferSelectionWindow *window = new TransferSelectionWindow(m_session, tr("Select downloads to change"), tr("Set category"), SLOT(showTransferCategoryDialog()), this);
    window->show();
}

void MainWindow::setPriorityOfSelectedTransfers() {
    TransferSelectionWindow *window = new TransferSelectionWindow(m_session, tr("Select downloads to change"), tr("Set priority"), SLOT(showTransferPriorityDialog()), this);
    window->show();
}

void MainWindow::addTransferFromUrlEdit() {
    if (!m_urlEdit->text().isEmpty()) {
        m_session->urlChecker()->addUrlToQueue(QUrl::fromUserInput(m_urlEdit->text()));
        m_urlEdit->clear();
    }
}

void MainWindow::showAddUrlsDialog() {
    AddUrlsDialog *dialog = new AddUrlsDialog(m_session, this);
    dialog->open();
}

void MainWindow::showTextFileDialog() {
    QString filePath = QFileDialog::getOpenFileName(this, tr("Choose text file"), QDesktopServices::storageLocation(QDesktopServices::HomeLocation), "*.txt");

    if (!filePath.isEmpty()) {
        m_session->urlChecker()->importUrlsFromTextFile(filePath);
    }
}

void MainWindow::showSettingsDialog() {
    SettingsDialog *dialog = new SettingsDialog(m_session, this);
    dialog->open();
}

void MainWindow::showAboutDialog() {
    AboutDialog *dialog = new AboutDialog(this);
    dialog->open();
}
