#include "downloadsettingstab.h"
#include "../shared/selectionmodels.h"
#include <QPushButton>
#include <QLabel>
#include <QLineEdit>
#include <QCheckBox>
#include <QComboBox>
#include <QSpinBox>
#include <QGridLayout>
#include <QFileDialog>

DownloadSettingsTab::DownloadSettingsTab(Session *session, QWidget *parent) :
    QWidget(parent),
    m_session(session),
    m_pathEdit(new QLineEdit(this)),
    m_statusCheckbox(new QCheckBox(tr("Start downloads automatically"), this)),
    m_concurrentSpinbox(new QSpinBox(this)),
    m_clipboardCheckbox(new QCheckBox(tr("Monitor clipboard for URLs"), this)),
    m_extractCheckbox(new QCheckBox(tr("Extract downloaded archives"), this)),
    m_deleteCheckbox(new QCheckBox(tr("Delete extracted archives"), this)),
    m_proxyCheckbox(new QCheckBox(tr("Use network proxy"), this)),
    m_proxyTypeCombobox(new QComboBox(this)),
    m_proxyHostEdit(new QLineEdit(this)),
    m_proxyPortEdit(new QLineEdit(this)),
    m_proxyUserEdit(new QLineEdit(this)),
    m_proxyPassEdit(new QLineEdit(this))
{
    QPushButton *browseButton = new QPushButton(tr("Browse"), this);

    QFrame *proxyFrame = new QFrame(this);
    proxyFrame->setFrameStyle(QFrame::Raised);
    proxyFrame->setEnabled(!m_session->settings()->networkProxyHostName().isEmpty());
    QGridLayout *proxyGrid = new QGridLayout(proxyFrame);
    proxyGrid->addWidget(new QLabel(tr("Proxy type") + ":", this), 0, 0);
    proxyGrid->addWidget(m_proxyTypeCombobox, 0, 1);
    proxyGrid->addWidget(new QLabel(tr("Host") + ":", this), 1, 0);
    proxyGrid->addWidget(m_proxyHostEdit, 1, 1);
    proxyGrid->addWidget(new QLabel(tr("Port") + ":", this), 1, 2);
    proxyGrid->addWidget(m_proxyPortEdit, 1, 3);
    proxyGrid->addWidget(new QLabel(tr("Username") + ":", this), 2, 0);
    proxyGrid->addWidget(m_proxyUserEdit, 2, 1);
    proxyGrid->addWidget(new QLabel(tr("Password") + ":", this), 2, 2);
    proxyGrid->addWidget(m_proxyPassEdit, 2, 3);

    QGridLayout *grid = new QGridLayout(this);
    grid->addWidget(new QLabel(tr("Default download path") + ':', this), 0, 0);
    grid->addWidget(m_pathEdit, 0, 1);
    grid->addWidget(browseButton, 0, 2);
    grid->addWidget(new QLabel(tr("Maximum concurrent downloads") + ':', this), 1, 0);
    grid->addWidget(m_concurrentSpinbox, 1, 1);
    grid->addWidget(m_statusCheckbox, 2, 0);
    grid->addWidget(m_clipboardCheckbox, 3, 0);
    grid->addWidget(m_extractCheckbox, 4, 0);
    grid->addWidget(m_deleteCheckbox, 5, 0);
    grid->addWidget(m_proxyCheckbox, 6, 0);
    grid->addWidget(proxyFrame, 7, 0, 1, 3);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Minimum, QSizePolicy::Expanding), 8, 0, 1, 3);

    m_concurrentSpinbox->setMinimum(1);
    m_concurrentSpinbox->setMaximum(5);

    bool transferQueueIsEmpty = m_session->transferManager()->rowCount() == 0;

    m_pathEdit->setEnabled(transferQueueIsEmpty);
    browseButton->setEnabled(transferQueueIsEmpty);

    m_proxyPassEdit->setEchoMode(QLineEdit::Password);

    this->connect(browseButton, SIGNAL(clicked()), this, SLOT(showFileDialog()));
    this->connect(m_extractCheckbox, SIGNAL(toggled(bool)), m_deleteCheckbox, SLOT(setEnabled(bool)));
    this->connect(m_proxyCheckbox, SIGNAL(toggled(bool)), proxyFrame, SLOT(setEnabled(bool)));
    this->connect(m_proxyCheckbox, SIGNAL(toggled(bool)), this, SLOT(onProxyCheckboxToggled(bool)));

    this->loadSettings();
}

void DownloadSettingsTab::loadSettings() {
    m_pathEdit->setText(m_session->settings()->downloadPath());
    m_concurrentSpinbox->setValue(m_session->settings()->maximumConcurrent());
    m_statusCheckbox->setChecked(m_session->settings()->startTransfersAutomatically());
    m_clipboardCheckbox->setChecked(m_session->settings()->monitorClipboard());
    m_extractCheckbox->setChecked(m_session->settings()->extractDownloadedArchives());
    m_deleteCheckbox->setChecked(m_session->settings()->deleteExtractedArchives());
    m_deleteCheckbox->setEnabled(m_extractCheckbox->isChecked());
    m_proxyCheckbox->setChecked(!m_session->settings()->networkProxyHostName().isEmpty());
    m_proxyTypeCombobox->setModel(new NetworkProxyTypeModel(this));
    m_proxyTypeCombobox->setCurrentIndex(m_proxyTypeCombobox->findData(m_session->settings()->networkProxyType(), Qt::UserRole + 1));
    m_proxyHostEdit->setText(m_session->settings()->networkProxyHostName());
    m_proxyPortEdit->setText(QString::number(m_session->settings()->networkProxyPort()));
    m_proxyPortEdit->setValidator(new QIntValidator(0, 100000, this));
    m_proxyUserEdit->setText(m_session->settings()->networkProxyUser());
    m_proxyPassEdit->setText(m_session->settings()->networkProxyPassword());
}

void DownloadSettingsTab::saveSettings() {
    m_session->settings()->setDownloadPath(m_pathEdit->text());
    m_session->settings()->setMaximumConcurrent(m_concurrentSpinbox->value());
    m_session->settings()->setStartTransfersAutomatically(m_statusCheckbox->isChecked());
    m_session->settings()->setMonitorClipboard(m_clipboardCheckbox->isChecked());
    m_session->settings()->setExtractDownloadedArchives(m_extractCheckbox->isChecked());
    m_session->settings()->setDeleteExtractedArchives(m_deleteCheckbox->isChecked());
    m_session->settings()->setNetworkProxyType(static_cast<NetworkProxyType::ProxyType>(m_proxyTypeCombobox->itemData(m_proxyTypeCombobox->currentIndex(), Qt::UserRole + 1).toInt()));
    m_session->settings()->setNetworkProxyHostName(m_proxyHostEdit->text());
    m_session->settings()->setNetworkProxyPort(m_proxyPortEdit->text().toUInt());
    m_session->settings()->setNetworkProxyUser(m_proxyUserEdit->text());
    m_session->settings()->setNetworkProxyPassword(m_proxyPassEdit->text());
    m_session->settings()->setNetworkProxy();
}

void DownloadSettingsTab::showFileDialog() {
    QString path = QFileDialog::getExistingDirectory(this, tr("Choose folder"), m_session->settings()->downloadPath());

    if (!path.isEmpty()) {
        m_pathEdit->setText(path);
    }
}

void DownloadSettingsTab::onProxyCheckboxToggled(bool checked) {
    if (!checked) {
        m_proxyHostEdit->clear();
        m_proxyUserEdit->clear();
        m_proxyPassEdit->clear();
    }
}
