#include "urlchecker.h"
#include "utils.h"
#include "serviceplugin.h"
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QFile>

UrlChecker::UrlChecker(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_manager(0)
{
}

void UrlChecker::setNetworkAccessManager(QNetworkAccessManager *manager) {
    m_nam = manager;
}

void UrlChecker::setPluginManager(PluginManager *manager) {
    m_manager = manager;

    foreach (ServicePlugin *plugin, pluginManager()->servicePlugins()) {
        this->connect(plugin, SIGNAL(urlChecked(bool,QUrl,QString,QString,bool)), this, SLOT(onUrlChecked(bool,QUrl,QString,QString,bool)));
    }
}

void UrlChecker::checkUrl(const QUrl &url) {
    if (ServicePlugin *plugin = pluginManager()->getServicePlugin(url)) {
        plugin->checkUrl(url);
    }
    else {
        this->testFileDownload(url);
    }
}

void UrlChecker::testFileDownload(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply = networkAccessManager()->head(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkFileDownload()));
}

void UrlChecker::checkFileDownload() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit urlChecked(false, QUrl(), QString(), QString(), true);
        return;
    }

    QUrl redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (!redirect.isEmpty()) {
        this->testFileDownload(redirect);
    }
    else {
        int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
        QUrl url = reply->request().url();

        if ((statusCode == 200) || (statusCode == 206)) {
            QString fileName = QString(reply->rawHeader("Content-Disposition")).section("filename=", -1).remove(QRegExp("\'|\""));

            emit urlChecked(true, url, QString(), fileName.isEmpty() ? reply->request().url().toString().section('/', -1) : fileName, true);
        }
        else {
            emit urlChecked(false, QUrl(), QString(), QString(), true);
        }
    }

    reply->deleteLater();
}

void UrlChecker::addUrlToQueue(const QUrl &url) {
    this->cancelUrlChecks(false);
    emit urlsQueued(QList<QUrl>() << url);
    m_urlQueue.enqueue(url);

    if (!m_urlQueue.isEmpty()) {
        this->checkUrl(m_urlQueue.dequeue());
    }
}

void UrlChecker::addUrlToQueue(const QString &url) {
    this->cancelUrlChecks(false);
    emit urlsQueued(QStringList() << url);
    m_urlQueue.enqueue(QUrl::fromUserInput(url));

    if (!m_urlQueue.isEmpty()) {
        this->checkUrl(m_urlQueue.dequeue());
    }
}

void UrlChecker::addUrlsToQueue(QList<QUrl> urls) {
    this->cancelUrlChecks(false);
    emit urlsQueued(urls);

    while (!urls.isEmpty()) {
        m_urlQueue.enqueue(urls.takeFirst());
    }

    if (!m_urlQueue.isEmpty()) {
        this->checkUrl(m_urlQueue.dequeue());
    }
}

void UrlChecker::addUrlsToQueue(QStringList urls) {
    this->cancelUrlChecks(false);
    emit urlsQueued(urls);

    while (!urls.isEmpty()) {
        m_urlQueue.enqueue(QUrl::fromUserInput(urls.takeFirst()));
    }

    if (!m_urlQueue.isEmpty()) {
        this->checkUrl(m_urlQueue.dequeue());
    }
}

void UrlChecker::parseUrlsFromText(const QString &text) {
    QStringList urlStrings = text.split(QRegExp("\\s"), QString::SkipEmptyParts);
    this->addUrlsToQueue(urlStrings);
}

void UrlChecker::importUrlsFromTextFile(const QString &filePath) {
    QFile textFile(filePath);

    if (textFile.open(QIODevice::ReadOnly | QIODevice::Text)) {
        QString urlString = textFile.readAll();
        textFile.close();
        this->parseUrlsFromText(urlString);
    }
}

void UrlChecker::onUrlChecked(bool ok, const QUrl &url, const QString &service, const QString &fileName, bool done) {
    emit urlChecked(ok, url, service, fileName, done);

    if ((done) && (!m_urlQueue.isEmpty()) && (!urlChecksCancelled())) {
        this->checkUrl(m_urlQueue.dequeue());
    }
}
