#include "session.h"
#include <qplatformdefs.h>
#include <QNetworkAccessManager>
#include <QApplication>
#include <QClipboard>
#include <QDateTime>
#include <QRegExp>
#include <QDir>
#include <QProgressDialog>
#include <QThread>
#ifdef MEEGO_EDITION_HARMATTAN
#include <QTimer>
#endif

Session::Session(QObject *parent) :
    QObject(parent),
    m_settings(new Settings(this)),
    m_database(new Database),
    m_transferManager(new TransferModel(this)),
    m_jar(new CookieJar(this)),
    m_nam(new QNetworkAccessManager(this)),
    m_checker(new UrlChecker(this)),
    m_manager(new PluginManager(this)),
    m_progressDialog(0),
    m_thread(new QThread),
    m_clipboardTimer(0)
{
#ifdef MEEGO_EDITION_HARMATTAN
    m_clipboardTimer = new QTimer(this);
    m_clipboardTimer->setInterval(3000);
    m_clipboardTimer->setSingleShot(true);
#endif
    m_thread->start(QThread::LowPriority);
    this->database()->moveToThread(m_thread);
    this->networkAccessManager()->setCookieJar(this->cookieJar());
    this->urlChecker()->setNetworkAccessManager(this->networkAccessManager());
    this->pluginManager()->setNetworkAccessManager(this->networkAccessManager());
    this->settings()->restoreSettings();
    this->pluginManager()->setDatabase(this->database());
    this->urlChecker()->setPluginManager(this->pluginManager());

    this->connect(m_thread, SIGNAL(finished()), m_thread, SLOT(deleteLater()));
    this->connect(m_thread, SIGNAL(finished()), this->database(), SLOT(deleteLater()));

    this->connect(this->urlChecker(), SIGNAL(urlChecked(bool,QUrl,QString,QString,bool)), this, SLOT(onUrlChecked(bool,QUrl,QString,QString)));
    this->connect(this->settings(), SIGNAL(maximumConcurrentChanged(int,int)), this, SLOT(onMaximumConcurrentChanged(int,int)));
    this->connect(this->settings(), SIGNAL(monitorClipboardChanged(bool)), this, SLOT(onMonitorClipboardChanged(bool)));
#ifndef MEEGO_EDITION_HARMATTAN
    this->connect(this->pluginManager(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));
    this->connect(this->pluginManager(), SIGNAL(busyProgressChanged(int)), this, SLOT(updateProgressDialog(int)));
    this->connect(this->pluginManager(), SIGNAL(pluginsReady()), this, SLOT(cancelProgressDialog()));
#endif
    this->connect(this->pluginManager(), SIGNAL(pluginsReady()), this, SLOT(restoreStoredDownloads()));

    this->pluginManager()->loadPlugins();
}

Session::~Session() {
    m_thread->quit();
}

void Session::restoreStoredDownloads() {
    this->disconnect(this->pluginManager(), SIGNAL(pluginsReady()), this, SLOT(restoreStoredDownloads()));
    QList< QSharedPointer<TransferItem> > transfers = this->database()->getStoredDownloads();

    for (int i = 0; i < transfers.size(); i++) {
        QSharedPointer<TransferItem> transfer = transfers.at(i);
        transfer.data()->setSession(this);
        transfer.data()->setNetworkAccessManager(this->networkAccessManager());
        transfer.data()->setServicePlugin(this->pluginManager()->createServicePlugin(transfer->service()));

        if ((transfer.data()->servicePlugin()) && (transfer.data()->servicePlugin()->recaptchaRequired())) {
            transfer.data()->setRecaptchaPlugin(this->pluginManager()->createRecaptchaPlugin(transfer.data()->servicePlugin()->recaptchaServiceName()));
        }

        transfer.data()->setStatus(this->settings()->startTransfersAutomatically() ? Transfers::Queued : Transfers::Paused);
        this->transferManager()->addTransfer(transfer);

        this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::Status)), this, SLOT(onTransferStatusChanged(Transfers::Status)));
        this->connect(transfer.data(), SIGNAL(categoryChanged(QString)), this, SLOT(onTransferCategoryChanged(QString)));
        this->connect(transfer.data(), SIGNAL(priorityChanged(Transfers::Priority)), this, SLOT(onTransferPriorityChanged(Transfers::Priority)));
        this->connect(transfer.data(), SIGNAL(saveAsAudioChanged(bool)), this, SLOT(onTransferSaveAsAudioChanged(bool)));
        this->connect(transfer.data(), SIGNAL(fileNameChanged(QString)), this, SLOT(onTransferFileNameChanged(QString)));
        this->connect(transfer.data(), SIGNAL(rowChanged(uint)), this, SLOT(onTransferRowChanged(uint)));
    }

    if ((!transfers.isEmpty()) && (this->settings()->startTransfersAutomatically())) {
        int numToStart = this->settings()->maximumConcurrent() - this->concurrentDownloads();

        for (int i = 0; i < numToStart; i++) {
            transfers.at(i).data()->startDownload();
        }
    }
}

void Session::onUrlChecked(bool ok, const QUrl &url, const QString &service, const QString &fileName) {
    if (ok) {
        this->addTransfer(url, service, fileName);
    }
}

QString Session::generateTransferId(QString seedFileName) const {
    return QString("%1_%2").arg(QString::number(QDateTime::currentMSecsSinceEpoch())).arg(seedFileName.section('.', 0, -2));
}

void Session::addTransfer(const QUrl &webUrl, const QString &service, const QString &fileName) {
    QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem(this->generateTransferId(fileName), webUrl, service, fileName, this->settings()->startTransfersAutomatically() ? Transfers::Queued : Transfers::Paused));
    transfer.data()->setDownloadPath(this->settings()->downloadPath() + "incomplete/");
    transfer.data()->setCategory(tr("Default"));
    transfer.data()->setSession(this);
    transfer.data()->setNetworkAccessManager(this->networkAccessManager());
    transfer.data()->setRow(this->transferManager()->rowCount());

    if (!service.isEmpty()) {
        transfer.data()->setServicePlugin(this->pluginManager()->createServicePlugin(service));

        if ((transfer.data()->servicePlugin()) && (transfer.data()->servicePlugin()->recaptchaRequired())) {
            transfer.data()->setRecaptchaPlugin(this->pluginManager()->createRecaptchaPlugin(transfer.data()->servicePlugin()->recaptchaServiceName()));
        }
    }

    this->transferManager()->addTransfer(transfer);

    this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::Status)), this, SLOT(onTransferStatusChanged(Transfers::Status)));
    this->connect(transfer.data(), SIGNAL(categoryChanged(QString)), this, SLOT(onTransferCategoryChanged(QString)));
    this->connect(transfer.data(), SIGNAL(priorityChanged(Transfers::Priority)), this, SLOT(onTransferPriorityChanged(Transfers::Priority)));
    this->connect(transfer.data(), SIGNAL(saveAsAudioChanged(bool)), this, SLOT(onTransferSaveAsAudioChanged(bool)));
    this->connect(transfer.data(), SIGNAL(fileNameChanged(QString)), this, SLOT(onTransferFileNameChanged(QString)));
    this->connect(transfer.data(), SIGNAL(rowChanged(uint)), this, SLOT(onTransferRowChanged(uint)));

    if ((transfer.data()->status() == Transfers::Queued) && (this->concurrentDownloads() < this->settings()->maximumConcurrent())) {
        transfer.data()->startDownload();
    }

    this->database()->storeDownload(transfer);
}

QSharedPointer<TransferItem> Session::getNextTransfer() const {
    int i = 0;
    int priority = Transfers::HighPriority;

    while (priority <= Transfers::LowPriority) {

        while (i < this->transferManager()->rowCount()) {
            QSharedPointer<TransferItem> transfer = transferManager()->getTransfer(i);

            if ((!transfer.isNull()) && (transfer.data()->priority() == priority) && (transfer.data()->status() == Transfers::Queued)) {
                return transfer;
            }

            i++;
        }

        priority++;
        i = 0;
    }

    return QSharedPointer<TransferItem>();
}

void Session::onTransferStatusChanged(Transfers::Status status) {
    if ((status <= Transfers::Queued) && (this->concurrentDownloads() < this->settings()->maximumConcurrent())) {
        if (QSharedPointer<TransferItem> transfer = this->getNextTransfer()) {
            transfer.data()->startDownload();
        }
    }

    if ((status == Transfers::Completed) || (status == Transfers::Cancelled)) {
        if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
            if (status == Transfers::Completed) {
                this->onTransferCompleted(this->transferManager()->getTransfer(transfer));
            }
            else  {
                this->onTransferCancelled(this->transferManager()->getTransfer(transfer));
            }
        }
    }
}

void Session::onTransferRowChanged(uint row) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        QMetaObject::invokeMethod(this->database(), "updateStoredDownload", Q_ARG(QString, transfer->id()), Q_ARG(QString, "rowNumber"), Q_ARG(QVariant, row));
    }
}

void Session::onTransferCategoryChanged(const QString &category) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        QMetaObject::invokeMethod(this->database(), "updateStoredDownload", Q_ARG(QString, transfer->id()), Q_ARG(QString, "category"), Q_ARG(QVariant, category));
    }
}

void Session::onTransferPriorityChanged(Transfers::Priority priority) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        QMetaObject::invokeMethod(this->database(), "updateStoredDownload", Q_ARG(QString, transfer->id()), Q_ARG(QString, "priority"), Q_ARG(QVariant, priority));
    }
}

void Session::onTransferSaveAsAudioChanged(bool saveAsAudio) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        QMetaObject::invokeMethod(this->database(), "updateStoredDownload", Q_ARG(QString, transfer->id()), Q_ARG(QString, "saveAsAudio"), Q_ARG(QVariant, saveAsAudio));
    }
}

void Session::onTransferFileNameChanged(const QString &fileName) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        QMetaObject::invokeMethod(this->database(), "updateStoredDownload", Q_ARG(QString, transfer->id()), Q_ARG(QString, "fileName"), Q_ARG(QVariant, fileName));
    }
}

void Session::onTransferCompleted(QSharedPointer<TransferItem> transfer) {
    QString categoryPath = this->database()->getCategoryPath(transfer.data()->category());
    QString oldFileName = transfer.data()->downloadPath() + transfer.data()->tempFileName();
    QString newFileName;
    QDir dir;

    if (!categoryPath.isEmpty()) {
        newFileName = categoryPath + transfer.data()->fileName();
        dir.mkpath(categoryPath);
    }
    else {
        newFileName = this->settings()->downloadPath() + transfer.data()->fileName();
        dir.mkpath(this->settings()->downloadPath());
    }

    int num = 1;
    bool fileSaved = QFile::rename(oldFileName, newFileName);

    while ((!fileSaved) && (num < 100)) {
        if (num == 1) {
            newFileName = newFileName.insert(newFileName.lastIndexOf("."), "(" + QByteArray::number(num) + ")");
        }
        else {
            newFileName = newFileName.replace(newFileName.lastIndexOf("(" + QByteArray::number(num - 1) + ")"), 3, "(" + QByteArray::number(num) + ")");
        }

        fileSaved = QFile::rename(oldFileName, newFileName);
        num++;
    }

    if (!fileSaved) {
        QFile::rename(oldFileName, settings()->downloadPath() + transfer.data()->fileName());
    }

    QMetaObject::invokeMethod(this->database(), "removeStoredDownload", Q_ARG(QString, transfer.data()->id()));
    this->transferManager()->removeTransfer(transfer);
}

void Session::onTransferCancelled(QSharedPointer<TransferItem> transfer) {
    QMetaObject::invokeMethod(this->database(), "removeStoredDownload", Q_ARG(QString, transfer.data()->id()));
    this->transferManager()->removeTransfer(transfer);
}

void Session::onMaximumConcurrentChanged(int oldMaximum, int newMaximum) {
    if ((newMaximum > oldMaximum) && (newMaximum > this->concurrentDownloads())) {
        if (QSharedPointer<TransferItem> transfer = this->getNextTransfer()) {
            transfer.data()->startDownload();
        }
    }
}

int Session::concurrentDownloads() const {
    int concurrent = 0;

    for (int i = 0; i < this->transferManager()->rowCount(); i++) {
        if (QSharedPointer<TransferItem> transfer = this->transferManager()->getTransfer(i)) {
            if (transfer.data()->status() > Transfers::Queued) {
                concurrent++;
            }
        }
    }

    return concurrent;
}

void Session::onMonitorClipboardChanged(bool monitor) {
    if (monitor) {
        this->connect(QApplication::clipboard(), SIGNAL(dataChanged()), this, SLOT(onClipboardTextChanged()));
    }
    else {
        this->disconnect(QApplication::clipboard(), SIGNAL(dataChanged()), this, SLOT(onClipboardTextChanged()));
    }
}

void Session::onClipboardTextChanged() {
    if (!QApplication::clipboard()) {
        return;
    }
#ifdef MEEGO_EDITION_HARMATTAN
    if (m_clipboardTimer->isActive()) {
        return;
    }
    else {
        m_clipboardTimer->start();
    }
#endif
    QString text = QApplication::clipboard()->text();

    if (!text.isEmpty()) {
        this->urlChecker()->parseUrlsFromText(text);
    }
}

void Session::showProgressDialog(const QString &message, int numberOfOperations) {
#ifndef MEEGO_EDITION_HARMATTAN
    if (!m_progressDialog) {
        m_progressDialog = new QProgressDialog;
    }

    m_progressDialog->setWindowTitle(tr("Please wait"));
    m_progressDialog->setCancelButtonText(QString());
    m_progressDialog->setMinimumDuration(0);
    m_progressDialog->setMinimum(0);
    m_progressDialog->setLabelText(message);
    m_progressDialog->setMaximum(numberOfOperations);
    m_progressDialog->setValue(0);
    m_progressDialog->show();
#else
    Q_UNUSED(message)
    Q_UNUSED(numberOfOperations)
#endif
}

void Session::updateProgressDialog(int progress) {
#ifndef MEEGO_EDITION_HARMATTAN
    if (m_progressDialog) {
        m_progressDialog->setValue(progress);
    }
#else
    Q_UNUSED(progress)
#endif
}

void Session::cancelProgressDialog() {
#ifndef MEEGO_EDITION_HARMATTAN
    if ((m_progressDialog) && (m_progressDialog->isVisible())) {
        m_progressDialog->cancel();
    }
#endif
}
