#include "transferdetailsdialog.h"
#include "transfercategoryselector.h"
#include "transferpriorityselector.h"
#include "../shared/utils.h"
#include "../shared/paths.h"
#include "../shared/session.h"
#include <QLabel>
#include <QGridLayout>
#include <QScrollArea>
#include <QLineEdit>
#include <QCheckBox>

TransferDetailsDialog::TransferDetailsDialog(Session *session, QSharedPointer<TransferItem> transfer, QWidget *parent) :
    QDialog(parent),
    m_session(session),
    m_transfer(transfer),
    m_statusLabel(new QLabel(this)),
    m_statusIcon(new QLabel(this)),
    m_infoLabel(new QLabel(this)),
    m_sizeLabel(new QLabel(this)),
    m_progressLabel(new QLabel(this)),
    m_speedLabel(new QLabel(this)),
    m_nameEdit(new QLineEdit(m_transfer.data()->fileName(), this)),
    m_nameButton(new QPushButton(tr("Change"), this)),
    m_pauseResumeButton(new QPushButton(this)),
    m_cancelButton(new QPushButton(tr("Cancel"), this))
{
    this->setWindowTitle(tr("Download status"));
    this->setAttribute(Qt::WA_DeleteOnClose, true);

    QLabel *titleHeader = new QLabel(tr("Details") + ":", this);
    QLabel *serviceIcon = new QLabel(this);
    QLabel *statusHeader = new QLabel(tr("Status") + ":", this);
    QLabel *sizeHeader = new QLabel(tr("Size") + ":", this);
    QLabel *progressHeader = new QLabel(tr("Progress") + ":", this);
    TransferCategorySelector *categorySelector = new TransferCategorySelector(m_session, tr("Category"), this);
    TransferPrioritySelector *prioritySelector = new TransferPrioritySelector(tr("Priority"), this);
    categorySelector->setValue(m_transfer.data()->category());
    prioritySelector->setValue(m_transfer.data()->priority());

    m_infoLabel->setWordWrap(true);
    titleHeader->setFixedWidth(100);
    statusHeader->setFixedWidth(100);
    sizeHeader->setFixedWidth(100);
    progressHeader->setFixedWidth(100);
    m_sizeLabel->setFixedWidth(100);
    m_sizeLabel->setAlignment(Qt::AlignHCenter);
    m_progressLabel->setFixedWidth(100);
    m_progressLabel->setAlignment(Qt::AlignHCenter);
    serviceIcon->setFixedSize(48, 48);
    m_statusIcon->setFixedSize(48, 48);
    serviceIcon->setScaledContents(true);
    m_statusIcon->setScaledContents(true);
    m_nameEdit->setCursorPosition(0);
    m_nameButton->setEnabled(false);
    m_nameButton->setFixedWidth(150);
    m_pauseResumeButton->setFixedWidth(150);
    m_cancelButton->setFixedWidth(150);

    if (m_transfer.data()->iconName().isEmpty()) {
        serviceIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/filemanager_unknown_file.png"));
    }
    else {
        serviceIcon->setPixmap(QPixmap(ICON_PATH_MAEMO_5 + m_transfer.data()->iconName()));
    }

    QScrollArea *scrollArea = new QScrollArea(this);
    QWidget *scrollWidget = new QWidget(scrollArea);

    QGridLayout *grid = new QGridLayout(scrollWidget);
    grid->addWidget(titleHeader, 0, 0);
    grid->addWidget(serviceIcon, 0, 1, 1, 1, Qt::AlignHCenter);
    grid->addWidget(m_nameEdit, 0, 2);
    grid->addWidget(m_nameButton, 0, 3);
    grid->addWidget(statusHeader, 1, 0);
    grid->addWidget(m_statusIcon, 1, 1, 1, 1, Qt::AlignHCenter);
    grid->addWidget(m_statusLabel, 1, 2);
    grid->addWidget(m_infoLabel, 2, 2);
    grid->addWidget(sizeHeader, 3, 0);
    grid->addWidget(m_sizeLabel, 3, 1);
    grid->addWidget(progressHeader, 4, 0);
    grid->addWidget(m_progressLabel, 4, 1);
    grid->addWidget(m_speedLabel, 4, 2);

    if (m_transfer.data()->convertibleToAudio()) {
        QCheckBox *audioCheckbox = new QCheckBox(tr("Save as audio"), this);
        audioCheckbox->setChecked(m_transfer.data()->saveAsAudio());

        grid->addWidget(audioCheckbox, 5, 0, 1, 3);
        grid->addWidget(categorySelector, 6, 0, 1, 3);
        grid->addWidget(m_pauseResumeButton, 6, 3);
        grid->addWidget(prioritySelector, 7, 0, 1, 3);
        grid->addWidget(m_cancelButton, 7, 3);

        this->connect(audioCheckbox, SIGNAL(clicked(bool)), m_transfer.data(), SLOT(setSaveAsAudio(bool)));
    }
    else {
        grid->addWidget(categorySelector, 5, 0, 1, 3);
        grid->addWidget(m_pauseResumeButton, 5, 3);
        grid->addWidget(prioritySelector, 6, 0, 1, 3);
        grid->addWidget(m_cancelButton, 6, 3);
    }

    scrollArea->setWidgetResizable(true);
    scrollArea->setWidget(scrollWidget);
    scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    QGridLayout *grid2 = new QGridLayout(this);
    grid2->addWidget(scrollArea);

    this->onTransferStatusChanged(m_transfer.data()->status());
    this->onTransferStatusInfoChanged(m_transfer.data()->statusInfo());
    this->onTransferSizeChanged(m_transfer.data()->size());
    this->onTransferProgressChanged(m_transfer.data()->progress());
    this->onTransferSpeedChanged(m_transfer.data()->speed());

    this->connect(m_transfer.data(), SIGNAL(statusChanged(Transfers::Status)), this, SLOT(onTransferStatusChanged(Transfers::Status)));
    this->connect(m_transfer.data(), SIGNAL(statusInfoChanged(QString)), this, SLOT(onTransferStatusInfoChanged(QString)));
    this->connect(m_transfer.data(), SIGNAL(sizeChanged(qint64)), this, SLOT(onTransferSizeChanged(qint64)));
    this->connect(m_transfer.data(), SIGNAL(progressChanged(int)), this, SLOT(onTransferProgressChanged(int)));
    this->connect(m_transfer.data(), SIGNAL(speedChanged(double)), this, SLOT(onTransferSpeedChanged(double)));
    this->connect(categorySelector, SIGNAL(valueChanged(QString)), m_transfer.data(), SLOT(setCategory(QString)));
    this->connect(prioritySelector, SIGNAL(valueChanged(Transfers::Priority)), m_transfer.data(), SLOT(setPriority(Transfers::Priority)));
    this->connect(m_nameEdit, SIGNAL(textEdited(QString)), this, SLOT(onFileNameTextChanged(QString)));
    this->connect(m_nameButton, SIGNAL(clicked()), this, SLOT(setTransferFileName()));
    this->connect(m_pauseResumeButton, SIGNAL(clicked()), this, SLOT(onPauseResumeClicked()));
    this->connect(m_cancelButton, SIGNAL(clicked()), this, SLOT(cancelTransfer()));
}

void TransferDetailsDialog::onTransferStatusChanged(Transfers::Status status) {
    m_statusLabel->setText(m_transfer.data()->statusText());

    switch (status) {
    case Transfers::Downloading:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/as_proxy_attachment_download.png"));
        m_pauseResumeButton->setText(tr("Pause"));
        m_infoLabel->hide();
        break;
    case Transfers::Connecting:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/general_packetdata.png"));
        m_pauseResumeButton->setText(tr("Pause"));
        m_infoLabel->hide();
        break;
    case Transfers::Paused:
        m_statusIcon->setPixmap(QPixmap("/etc/hildon/theme/mediaplayer/Pause.png"));
        m_pauseResumeButton->setText(tr("Resume"));
        m_infoLabel->hide();
        break;
    case Transfers::ShortWait:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/general_clock.png"));
        m_pauseResumeButton->setText(tr("Pause"));
        m_infoLabel->show();
        break;
    case Transfers::LongWait:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/general_clock.png"));
        m_pauseResumeButton->setText(tr("Pause"));
        m_infoLabel->show();
        break;
    case Transfers::Converting:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/general_audio_file.png"));
        m_pauseResumeButton->setText(tr("Pause"));
        m_infoLabel->show();
        break;
    case Transfers::Failed:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/general_stop.png"));
        m_pauseResumeButton->setText(tr("Resume"));
        m_infoLabel->show();
        break;
    case Transfers::Completed:
        this->accept();
        break;
    case Transfers::Cancelled:
        this->accept();
        break;
    default:
        m_statusIcon->setPixmap(QPixmap("/etc/hildon/theme/mediaplayer/Stop.png"));
        m_pauseResumeButton->setText(tr("Pause"));
        m_infoLabel->hide();
    }
}

void TransferDetailsDialog::onTransferStatusInfoChanged(const QString &info) {
    m_infoLabel->setText(QString("<font color='red'>%1</font>").arg(info));
}

void TransferDetailsDialog::onTransferSizeChanged(qint64 size) {
    m_sizeLabel->setText(size > 0 ? Utils::fileSizeFromBytes(size) : "-");
}

void TransferDetailsDialog::onTransferProgressChanged(int progress) {
    m_progressLabel->setText(QString("%1%").arg(progress));
}

void TransferDetailsDialog::onTransferSpeedChanged(double speed) {
    m_speedLabel->setText(QString("<font color='#4d4d4d'>%1/s</font>").arg(Utils::fileSizeFromBytes(speed)));
}

void TransferDetailsDialog::onFileNameTextChanged(const QString &text) {
    m_nameButton->setEnabled(!text.isEmpty());
}

void TransferDetailsDialog::setTransferFileName() {
    m_transfer.data()->setFileName(m_nameEdit->text());
    m_nameButton->setEnabled(false);
}

void TransferDetailsDialog::onPauseResumeClicked() {
    if (m_transfer.data()->status() == Transfers::Paused) {
        m_transfer.data()->setStatus(Transfers::Queued);
    }
    else {
        m_transfer.data()->setStatus(Transfers::Paused);
    }
}

void TransferDetailsDialog::cancelTransfer() {
    m_transfer.data()->setStatus(Transfers::Cancelled);
}
