#include "checkurlsdialog.h"
#include "../shared/session.h"
#include <QGridLayout>
#include <QTreeWidgetItem>
#include <QDialogButtonBox>
#include <QTreeWidget>
#include <QProgressBar>
#include <QLabel>
#include <QPushButton>

CheckUrlsDialog::CheckUrlsDialog(Session *session, QWidget *parent) :
    QDialog(parent),
    m_session(session),
    m_view(new QTreeWidget(this)),
    m_progressBar(new QProgressBar(this)),
    m_infoLabel(new QLabel(QString("<i>%1</i>").arg(tr("Checking URLs")), this)),
    m_okButton(new QPushButton(tr("Done"), this)),
    m_cancelButton(new QPushButton(tr("Cancel"), this))
{
    this->setWindowTitle(tr("Check URLs"));
    this->setAttribute(Qt::WA_DeleteOnClose, false);
    this->setFixedHeight(340);

    m_view->setHeaderLabels(QStringList() << tr("URL") << tr("OK?"));
    m_view->setColumnWidth(0, 520);
    m_view->setColumnWidth(1, 50);
    m_view->setRootIsDecorated(false);
    m_view->setSelectionMode(QTreeWidget::NoSelection);
    m_view->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding);

    m_progressBar->setValue(0);
    m_progressBar->setMaximum(0);

    QDialogButtonBox *buttonBox = new QDialogButtonBox(Qt::Vertical, this);
    buttonBox->addButton(m_cancelButton, QDialogButtonBox::ActionRole);
    buttonBox->addButton(m_okButton, QDialogButtonBox::ActionRole);

    QGridLayout *grid = new QGridLayout(this);
    grid->addWidget(m_view, 0, 0);
    grid->addWidget(m_progressBar, 1, 0, Qt::AlignBottom);
    grid->addWidget(m_infoLabel, 2, 0, Qt::AlignBottom);
    grid->addWidget(buttonBox, 0, 1, 3, 1, Qt::AlignBottom);

    this->connect(m_session->urlChecker(), SIGNAL(urlsQueued(QList<QUrl>)), this, SLOT(checkUrls(QList<QUrl>)));
    this->connect(m_session->urlChecker(), SIGNAL(urlsQueued(QStringList)), this, SLOT(checkUrls(QStringList)));
    this->connect(m_session->urlChecker(), SIGNAL(urlChecked(bool,QUrl,QString,QString,bool)), this, SLOT(onUrlChecked(bool,QUrl,QString,QString,bool)));
    this->connect(m_okButton, SIGNAL(clicked()), this, SLOT(accept()));
    this->connect(m_cancelButton, SIGNAL(clicked()), m_session->urlChecker(), SLOT(cancelUrlChecks()));
}

void CheckUrlsDialog::hideEvent(QHideEvent *event) {
    Q_UNUSED(event);

    this->resetDialog();
}

void CheckUrlsDialog::checkUrls(QList<QUrl> urls) {
    m_okButton->setEnabled(false);
    m_cancelButton->setEnabled(true);

    if (m_progressBar->value() == m_progressBar->maximum()) {
        m_progressBar->setValue(0);
        m_progressBar->setMaximum(urls.size());
    }
    else {
        m_progressBar->setMaximum(m_progressBar->maximum() + urls.size());
    }

    for (int i = 0; i < urls.size(); i++) {
        QTreeWidgetItem *item = new QTreeWidgetItem(m_view);
        item->setText(0, urls.at(i).toString());
    }

    m_view->setCurrentItem(m_view->itemAt(0, 0));

    this->show();
}

void CheckUrlsDialog::checkUrls(QStringList urls) {
    m_okButton->setEnabled(false);
    m_cancelButton->setEnabled(true);

    if (m_progressBar->value() == m_progressBar->maximum()) {
        m_progressBar->setValue(0);
        m_progressBar->setMaximum(urls.size());
    }
    else {
        m_progressBar->setMaximum(m_progressBar->maximum() + urls.size());
    }

    for (int i = 0; i < urls.size(); i++) {
        QTreeWidgetItem *item = new QTreeWidgetItem(m_view);
        item->setText(0, urls.at(i));
    }

    m_view->setCurrentItem(m_view->itemAt(0, 0));

    this->show();
}

void CheckUrlsDialog::onUrlChecked(bool ok, const QUrl &url, const QString &service, const QString &fileName, bool done) {
    Q_UNUSED(url);
    Q_UNUSED(service);
    Q_UNUSED(fileName);

    if ((!this->isVisible()) || (!done)) {
        return;
    }

    if (m_view->currentItem()) {
        m_view->currentItem()->setIcon(1, ok ? QIcon::fromTheme("widgets_tickmark_list") : QIcon::fromTheme("general_stop"));
        m_view->setCurrentItem(m_view->itemBelow(m_view->currentItem()));
    }

    m_progressBar->setValue(m_progressBar->value() + 1);

    if (m_progressBar->value() == m_progressBar->maximum()) {
        m_okButton->setEnabled(true);
        m_cancelButton->setEnabled(false);
        m_infoLabel->setText(QString("<i>%1</i>").arg(m_session->urlChecker()->urlChecksCancelled() ? tr("Cancelled") : tr("Completed")));
    }
}

void CheckUrlsDialog::resetDialog() {
    m_progressBar->setValue(0);
    m_progressBar->setMaximum(0);
    m_infoLabel->setText(QString("<i>%1</i>").arg(tr("Checking URLs")));
    m_view->clear();
}
