#include "vimeo.h"
#include "json.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QRegExp>
#include <QSettings>

using namespace QtJson;

Vimeo::Vimeo(QObject *parent) :
    ServicePlugin(parent)
{
    m_formatList << "hd" << "sd";
}

QRegExp Vimeo::urlPattern() const {
    return QRegExp("http(s|)://vimeo.com/\\d+", Qt::CaseInsensitive);
}

bool Vimeo::urlSupported(const QUrl &url) const {
    return this->urlPattern().indexIn(url.toString()) == 0;
}

void Vimeo::checkUrl(const QUrl &webUrl) {
    QNetworkRequest request(webUrl);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUrlIsValid()));
}

void Vimeo::checkUrlIsValid() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit urlChecked(false);
        return;
    }

    QString response(reply->readAll());
    QString params = response.section("config:", 1, 1).section("};", 0, 0);
    bool ok;
    QVariantMap paramMap = Json::parse(params, ok).toMap();

    if (!ok) {
        emit urlChecked(false);
    }
    else {
        QVariantMap videoMap = paramMap.value("video").toMap();
        QString fileName = videoMap.value("title").toString() + ".mp4";
        emit urlChecked(true, reply->request().url(), this->serviceName(), fileName);
    }

    reply->deleteLater();
}

void Vimeo::getDownloadUrl(const QUrl &webUrl) {
    emit statusChanged(Connecting);
    QNetworkRequest request(webUrl);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseVideoPage()));
}

void Vimeo::parseVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QString response(reply->readAll());
    QString params = response.section("config:", 1, 1).section("};", 0, 0);
    bool ok;
    QVariantMap paramMap = Json::parse(params, ok).toMap();

    if (!ok) {
        emit error(UnknownError);
    }
    else {
        QString format = QSettings("QDL", "QDL").value("Vimeo/videoFormat", "sd").toString();
        QVariantMap requestMap = paramMap.value("request").toMap();
        QVariantMap videoMap = paramMap.value("video").toMap();
        QVariantMap formatMap = videoMap.value("files").toMap();
        QString codec("h264");
        QString quality;

        if (!formatMap.isEmpty()) {
            codec = formatMap.keys().first();

            QStringList qualities = formatMap.value(codec).toStringList();

            int i = m_formatList.indexOf(format);

            while ((quality.isEmpty()) && (i < qualities.size())) {
                quality = qualities.at(i);
                i++;
            }
        }

        QString timeStamp = requestMap.value("timestamp").toString();
        QString signature = requestMap.value("signature").toString();
        QString id = videoMap.value("id").toString();
        QUrl url(QString("http://player.vimeo.com/play_redirect?quality=%1&codecs=%2&clip_id=%3&time=%4&sig=%5&type=html5_desktop_local").arg(quality).arg(codec).arg(id).arg(timeStamp).arg(signature));

        this->getRedirect(url);
    }

    reply->deleteLater();
}

void Vimeo::getRedirect(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkRedirect()));
}

void Vimeo::checkRedirect() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QUrl redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (redirect.isEmpty()) {
        emit error(UrlError);
    }
    else {
        emit downloadRequestReady(QNetworkRequest(redirect));
    }

    reply->deleteLater();
}

Q_EXPORT_PLUGIN2(vimeo, Vimeo)
