#include "rapidgator.h"
#include "json.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>
#include <QRegExp>

using namespace QtJson;

RapidGator::RapidGator(QObject *parent) :
    ServicePlugin(parent),
    m_waitTimer(new QTimer(this)),
    m_waitTime(0)

{
    this->connect(m_waitTimer, SIGNAL(timeout()), this, SLOT(updateWaitTime()));
}

QRegExp RapidGator::urlPattern() const {
    return QRegExp("http(s|)://(www.|)rapidgator.net/file/\\w+", Qt::CaseInsensitive);
}

bool RapidGator::urlSupported(const QUrl &url) const {
    return this->urlPattern().indexIn(url.toString()) == 0;
}

void RapidGator::login(const QString &username, const QString &password) {
    QUrl url("https://rapidgator.net/auth/login");
    QString data = QString("LoginForm[email]=%1&LoginForm[password]=%2&LoginForm[rememberMe]=1").arg(username).arg(password);
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, data.toUtf8());
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkLogin()));
}

void RapidGator::checkLogin() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    emit loggedIn((statusCode == 302) || (statusCode == 200));

    reply->deleteLater();
}

void RapidGator::checkUrl(const QUrl &webUrl) {
    QNetworkRequest request(webUrl);
    request.setRawHeader("Accept-Language", "en-GB,en-US;q=0.8,en;q=0.6");
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUrlIsValid()));
}

void RapidGator::checkUrlIsValid() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit urlChecked(false);
        return;
    }

    QString redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();
    QRegExp re("http://pr\\d+.rapidgator.net//\\?r=download/index&session_id=[^'\"]+");

    if ((!redirect.isEmpty()) && (re.indexIn(redirect) == -1)) {
        this->checkUrl(QUrl(redirect));
    }
    else {
        QString response(reply->readAll());

        if (response.contains("File not found")) {
            emit urlChecked(false);
        }
        else {
            QString fileName = response.section("<title>Download file ", 1, 1).section('<', 0, 0).trimmed();

            if (fileName.isEmpty()) {
                emit urlChecked(false);
            }
            else {
                emit urlChecked(true, reply->request().url(), this->serviceName(), fileName);
            }
        }
    }

    reply->deleteLater();
}

void RapidGator::getDownloadUrl(const QUrl &webUrl) {
    m_url = webUrl;
    emit statusChanged(Connecting);
    QNetworkRequest request(webUrl);
    request.setRawHeader("Accept-Language", "en-GB,en-US;q=0.8,en;q=0.6");
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(onWebPageDownloaded()));
}

void RapidGator::onWebPageDownloaded() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QRegExp re("http://pr\\d+.rapidgator.net//\\?r=download/index&session_id=[^'\"]+");
    QString redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();

    if (re.indexIn(redirect) == 0) {
        QNetworkRequest request;
        request.setUrl(QUrl(re.cap()));
        emit downloadRequestReady(request);
    }
    else {
        QString response(reply->readAll());

        if (re.indexIn(response) >= 0) {
            QNetworkRequest request;
            request.setUrl(QUrl(re.cap()));
            emit downloadRequestReady(request);
        }
        else if (!redirect.isEmpty()) {
            this->getDownloadUrl(QUrl(redirect));
        }
        else if (response.contains("File not found")) {
            emit error(NotFound);
        }
        else if (response.contains("Please try again later")) {
            this->startWait(600000);
            this->connect(this, SIGNAL(waitFinished()), this, SLOT(onWaitFinished()));
        }
        else {
            m_fileId = response.section("var fid = ", 1, 1).section(';', 0, 0);
            m_secs = response.section("var secs = ", 1, 1).section(';', 0, 0).toInt();
            int mins = response.section("not less than ", 1, 1).section(" min", 0, 0).toInt();

            if ((m_fileId.isEmpty()) || (m_secs <= 0)) {
                emit error(UnknownError);
            }
            else if (mins > 0) {
                this->startWait(mins * 60000);
                this->connect(this, SIGNAL(waitFinished()), this, SLOT(onWaitFinished()));
            }
            else {
                this->getSessionId();
            }
        }
    }

    reply->deleteLater();
}

void RapidGator::getSessionId() {
    QUrl url("http://rapidgator.net/download/AjaxStartTimer");
    url.addQueryItem("fid", m_fileId);
    QNetworkRequest request(url);
    request.setRawHeader("X-Requested-With", "XMLHttpRequest");
    request.setRawHeader("Accept", "application/json, text/javascript, */*; q=0.01");
    request.setRawHeader("Referer", m_url.toString().toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkSessionId()));
}

void RapidGator::checkSessionId() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QString response(reply->readAll());
    QVariantMap map = Json::parse(response).toMap();

    if (map.value("state").toString() == "started") {
        m_sessionId = map.value("sid").toString();
        this->startWait(m_secs * 1000);
        this->connect(this, SIGNAL(waitFinished()), this, SLOT(getDownloadLink()));
    }
    else {
        emit error(UnknownError);
    }

    reply->deleteLater();
}

void RapidGator::getDownloadLink() {
    QUrl url("http://rapidgator.net/download/AjaxGetDownloadLink");
    url.addQueryItem("sid", m_sessionId);
    QNetworkRequest request(url);
    request.setRawHeader("X-Requested-With", "XMLHttpRequest");
    request.setRawHeader("Accept", "application/json, text/javascript, */*; q=0.01");
    request.setRawHeader("Referer", m_url.toString().toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDownloadLink()));
    this->disconnect(this, SIGNAL(waitFinished()), this, SLOT(getDownloadLink()));
}

void RapidGator::checkDownloadLink() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QString response(reply->readAll());
    QVariantMap map = Json::parse(response).toMap();

    if (map.value("state").toString() == "done") {
        emit statusChanged(CaptchaRequired);
    }
    else {
        emit error(UrlError);
    }

    reply->deleteLater();
}

void RapidGator::submitCaptchaResponse(const QString &challenge, const QString &response) {
    QString data = QString("DownloadCaptchaForm[captcha]=&adcopy_challenge=%1&adcopy_response=%2").arg(challenge).arg(response);
    QUrl url("http://rapidgator.net/download/captcha");
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, data.toUtf8());
    this->connect(reply, SIGNAL(finished()), this, SLOT(onCaptchaSubmitted()));
}

void RapidGator::onCaptchaSubmitted() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QRegExp re("http://pr\\d+.rapidgator.net//\\?r=download/index&session_id=[^'\"]+");
    QString response(reply->readAll());

    if (re.indexIn(response) >= 0) {
        QNetworkRequest request;
        request.setUrl(QUrl(re.cap()));
        emit downloadRequestReady(request);
    }
    else if (response.contains("verification code is incorrect")) {
        emit error(CaptchaError);
    }
    else if (response.contains("You have reached your daily downloads limit")) {
        this->startWait(600000);
        this->connect(this, SIGNAL(waitFinished()), this, SLOT(onWaitFinished()));
    }
    else {
        emit error(UnknownError);
    }

    reply->deleteLater();
}

void RapidGator::startWait(int msecs) {
    if (msecs > (m_secs * 1000)) {
        emit statusChanged(LongWait);
    }
    else {
        emit statusChanged(ShortWait);
    }

    emit waiting(msecs);
    m_waitTime = msecs;
    m_waitTimer->start(1000);
}

void RapidGator::updateWaitTime() {
    m_waitTime -= m_waitTimer->interval();
    emit waiting(m_waitTime);

    if (m_waitTime <= 0) {
        m_waitTimer->stop();
        emit waitFinished();
    }
}

void RapidGator::onWaitFinished() {
    emit statusChanged(Ready);
    this->disconnect(this, SIGNAL(waitFinished()), this, SLOT(onWaitFinished()));
}

Q_EXPORT_PLUGIN2(rapidgator, RapidGator)
