#include "bcprocess.h"

#define BC "/usr/bin/bc -l"
#define BC_WAIT 1000

BcProcess::BcProcess(QObject *parent) :
    QProcess(parent)
{
    outputBuffer = "";
    quiet = false;
    connect(this, SIGNAL(readyReadStandardOutput()), this, SLOT(readOutput()));
    connect(this, SIGNAL(readyReadStandardError()), this, SLOT(readError()));
}

void BcProcess::start()
{
    QProcess::start(BC);
}

void BcProcess::quit()
{
    write("quit\n");
    waitForFinished(BC_WAIT);
}


QHash<BuiltInVariables, QString> BcProcess::getBuiltInVariables()
{
    QHash<BuiltInVariables, QString> ret;

    quiet = true;

    saveState();
    write("obase=A\n");

    ret[last]=getVariable("last");
    ret[scale]=getVariable("scale");
    ret[ibase]=getVariable("ibase");
    ret[obase]=getVariable("qbctmpobase");

    restoreState();

    quiet = false;

    return ret;
}

void BcProcess::readOutput()
{
    if (!quiet)
    {
        outputBuffer += readAllStandardOutput();
        outputBuffer.remove("\n");
        if (!outputBuffer.isEmpty() && outputBuffer.right(1) != "\\")
        {
            outputBuffer.remove("\\");
            emit outputReady(outputBuffer);
            outputBuffer = "";
        }
    }
}

void BcProcess::readError()
{
    if (!quiet)
    {
        QString error = this->readAllStandardError();
        error.remove("\n");
        if (!error.isEmpty()) emit errorReady(error);
    }
}

QString BcProcess::getVariable(QString variable)
{
    QString ret = "";
    bool oldQuiet = quiet;
    quiet = true;
    write(variable.append("\n").toLatin1());
    waitForReadyRead(1000);
    while (canReadLine())
        ret += readLine();
    ret.remove("\n");
    ret.remove("\\");
    quiet = oldQuiet;
    return ret;
}

void BcProcess::setVariable(QString variable, QString value)
{
    bool oldQuiet = quiet;
    quiet = true;

    saveState(variable != "ibase", variable != "obase");
    write("ibase=A\n");
    write(QString(variable + "=" + value + "\n").toLatin1());
    restoreState(variable != "ibase", variable != "obase");

    quiet = oldQuiet;
}

void BcProcess::saveState(bool ibase, bool obase)
{
    if (obase)
        write("qbctmpobase=obase\n");
    if (ibase)
        write("qbctmpibase=ibase\n");
    write("qbctmplast=last\n");
}

void BcProcess::restoreState(bool ibase, bool obase)
{
    write("last=qbctmplast\n");
    if (ibase)
        write("ibase=qbctmpibase\n");
    if (obase)
        write("obase=qbctmpobase\n");
}
