## This file is part of mlpy.
## Extend.
    
## This code is written by Davide Albanese, <albanese@fbk.eu>.
## (C) 2009 Fondazione Bruno Kessler - Via Santa Croce 77, 38100 Trento, ITALY.

## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.

## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.

## You should have received a copy of the GNU General Public License
## along with this program.  If not, see <http://www.gnu.org/licenses/>.


__all__ = ['extend']

import numpy as np
import math
import misc



def extend(x, method='reflection', length='powerof2'):
    """Extend the 1D numpy array x beyond its original length.

    Input

    * *x* - [1D numpy array] data
    * *method* - [string] indicates which extension method to use
      ('reflection', 'periodic', 'zeros')
    * *length* - [string] indicates how to determinate the length
      of the extended data ('powerof2', 'double')
    
    Output
    
    * *xext* - [1D numpy array] extended version of x

    Example

    >>> import numpy as np
    >>> import mlpy
    >>> a = np.array([1,2,3,4,5])
    >>> mlpy.extend(a, method='periodic', length='powerof2')
    array([1, 2, 3, 4, 5, 1, 2, 3])    
    """

    if length == 'powerof2':
        lt = misc.next_power(x.shape[0], 2)
        lp = lt - x.shape[0]

    elif length == 'double':
        lp = x.shape[0]
       
    else:
        ValueError("length %s is not available" % length)


    if method == 'reflection':
        xret = np.append(x, x[::-1][:lp])

    elif method == 'periodic':
        xret = np.append(x, x[:lp])

    elif method == 'zeros':
        xret = np.append(x, np.zeros(lp, dtype=x.dtype))

    else:
        ValueError("method %s is not available" % method)

    return xret
        
        
    
    
