##############################################################################
#
# class smconv_XML_Importer
# 
# Author: Felix Engel <Felix.Engel@fcenet.de>
# 
# Import the xml file created by the smconv.pl script to Mnemosyne.
# smconv.pl is available at http://smconvpl.sourceforge.net and reads
# SuperMemo for Palm databases and exports them to XML.
#
# In order to import the generated XML into mnemosyne, care must be taken
# to ensure the correct charset encoding of the input file. In my case,
# the palm databases are "windows-1252". The xml file generated by smconv.pl
# was set to "us-ascii". This makes the XML parser fail. For me, changing
# the xml header to <?xml version="1.0" encoding="windows-1252"?>  worked
# well. However, your mileage may vary.
#
# Restrictions:
#
#  - SM for Palm has six fields for each card. Templates can be used to
#    format these fields and to control whether they are part of  the
#    question or of the answer. However this class assumes that the first
#    field is the question and the second field is the answer.
#  - No error handling. If the XML is not well formed or if fields are
#    missing, the behaviour is unpredictable.
#
##############################################################################

from xml.sax import saxutils, make_parser
from xml.sax.handler import feature_namespaces, ContentHandler

class smconv_XML_Importer(ContentHandler):
    
    def __init__(self, default_cat=None, reset_learning_data=False):
        
        self.reading, self.text = {}, {}
        
        self.reading["cat"] = False
        self.reading["Q"]   = False
        self.reading["A"]   = False

        self.default_cat = default_cat
        self.reset_learning_data = reset_learning_data

        self.imported_cards = []
        self.lapses = 0
        self.recalls = 0
        self.difficulty = 40 
        self.difficulty_prev = 40
        self.datecommit = ""
        self.interval = 0
        self.interval_prev = 0
        self.commit = 0

    def to_bool(self, string):
        if string == '0':
            return False
        else:
            return True
   
    def reset_elements(self):
        
        self.lapses = 0
        self.recalls = 0
        self.difficulty = 40 
        self.difficulty_prev = 40
        self.datecommit = ""
        self.interval = 0
        self.interval_pref = 0

    def startElement(self, name, attrs):
	
        global import_time_of_start
        
	# I cannot guarantee, when the header will be read, so
	# I use the epoch for now. Times will be normalized,
	# once the whole database has been read.
	
        if name == "card":
            self.card = Card();
            if attrs.get("commit"):
                self.commit = attrs.get("commit")
            if attrs.get("category"):
                self.category = attrs.get("category")

        elif name == "card_other":
            if attrs.get("lapses"):
                self.lapses = int(attrs.get("lapses"))
            if attrs.get("recalls"):
                self.recalls = int(attrs.get("recalls"))
            if attrs.get("difficulty"):
                self.difficulty = int(attrs.get("difficulty"))
            if attrs.get("difficulty_prev"):
                self.difficulty_prev = int(attrs.get("difficulty_prev"))
            if attrs.get("datecommit"):
                self.datecommit = attrs.get("datecommit")
            if attrs.get("datenexttest"):
                self.datenexttest = attrs.get("datenexttest")
            if attrs.get("interval"):
                self.interval = int(attrs.get("interval"))
            if attrs.get("interval_prev"):
                self.interval_prev = int(attrs.get("interval_prev"))

        elif name == "card_field":
            if attrs.get("idx") == "1":
                self.reading["Q"] = True
                self.text["Q"] = ""
            if  attrs.get("idx") == "2":
                self.reading["A"] = True
                self.text["A"] = ""
	elif name == "header":
            if attrs.get("datecommit"):
                start_time = attrs.get("datecommit")
                try:
                    struct_t = time.strptime(attrs.get("datecommit"),\
                                             "%Y-%m-%d")
                    t_sec = time.mktime(struct_t)
                    import_time_of_start = StartTime(t_sec); 
                except:
                    import_time_of_start = StartTime(0); 
			
	else: # Default action: do nothing.
            return

    def characters(self, ch):
        for name in self.reading.keys():
            if self.reading[name] == True:
                self.text[name] += ch

    def guess_grade(self):
        
	# Very easy cards are scarce in SM and must be easiest grade.
        
	if self.difficulty < 10:
		return 5

	# Assign passing grades, based upon whether the difficulty has
	# changed.
        
	if self.difficulty > self.difficulty_prev:
		return 2

	if self.difficulty == self.difficulty_prev:
		return 3

	if self.difficulty < self.difficulty_prev:
		return 4

	# If the interval becomes shorter, it must have been a failure.
        
	if self.interval < self.interval_prev:
            return 1

    def endElement(self, name):
        
	if name == "card":
            
            # Try to derive an easines factor EF from [1.3 .. 3.2] from
            # difficulty d from [1% .. 100%]. 
            # The math below is set to translate
            # difficulty=100% --> easiness = 1.3
            # difficulty=40% --> easiness = 2.5
            # difficulty=1% --> easiness = 3.2
            
            import math
            dp = self.difficulty * 0.01

            # Small values should be easy, large ones hard.
            
            if dp > 0.4:
                self.card.easiness = 1.28 - 1.32 * math.log(dp)
            else:
                self.card.easiness = 4.2 - (1.139 * math.exp(dp) )

            # Grades are 0-5. In SM for Palm there are commited and uncommited 
            # cards. Uncommited cards go to grade 0.
            # Otherwise try to extrapolate something from difficulty in SM
            # I have implemented guess_grade such, that the distribution of
            # grades looks reasonable for my test database of 4000 entries.
            # By "reasonable" I mean than most of the entries should be 
            # at grade 4. I've been learning that database for 4 years, so the
            # cards should have converged by now.
                
            if self.commit == False:
                self.card.grade = 0
            else:
                self.card.grade = self.guess_grade()

            self.card.lapses = self.lapses
		
            # Handle dates, assume starttime to be the epoch.
            # Need to determine last_rep and next_rep.

            try:
                struct_t = time.strptime(self.datenexttest,"%Y-%m-%d")
            except:
                print _("Failed to parse time - using zero.")

            t_sec = int(0)

            try:
                t_sec = time.mktime(struct_t)
            except:
                print _("mktime failed - using zero.")

            self.card.next_rep = int(t_sec / 86400)

            # last_rep is interval in days before next_rep.

            self.card.last_rep = self.card.next_rep - self.interval

            # Try to fill acquisiton reps and retention reps.
            # Since SM statistics are only available for commited
            # cards, I take acq_reps = 0 and ret_reps = lapses + recalls.
            
            self.card.ret_reps = self.lapses + self.recalls

            self.card.cat = get_category_by_name(self.category)
            self.imported_cards.append(self.card)
		
	elif name == "card_field":
            if self.reading["Q"]:
                self.reading["Q"] = False
                self.card.q = self.text["Q"]
                self.text["Q"] = ""
            if self.reading["A"]:
                self.reading["A"] = False
                self.card.a = self.text["A"]
                self.text["A"] = ""

	elif name == "smconv_pl":
            
            # During the import, there was no guarantee that the start time
            # has already been read. Now, at the smconv_pl closing tag, the
            # import_time_of_start variable has been set. Update all imported
            # cards accordingly.

            now = import_time_of_start.time
            diff = int(now / 86400)
            for i in self.imported_cards:
                i.next_rep = i.next_rep - diff
                i.last_rep = i.last_rep - diff



# TODO: remove duplication over different XML formats

##############################################################################
#
# import_XML
#
#   Note that we do not register separate file formats for Mnemosyne and
#   Memaid XML. We're able to figure out the difference on our own and do not
#   need to put this burden on the user.
#
##############################################################################

def import_XML(filename, default_cat, reset_learning_data=False):
    global cards

    # Determine if we import a Mnemosyne or a Memaid file.

    handler = None

    f = None
    try:
        f = file(filename)
    except:
        try:
            f = file(unicode(filename).encode("latin"))
        except:
            raise LoadError()
    
    l = f.readline()
    l += f.readline();    
    if "mnemosyne" in l:
        handler = XML_Importer(default_cat, reset_learning_data)
    elif "smconv_pl" in l:
    	handler = smconv_XML_Importer(default_cat, reset_learning_data)
    else:
        handler = memaid_XML_Importer(default_cat, reset_learning_data)
        
    f.close()

    # Parse XML file.
    
    parser = make_parser()
    parser.setFeature(feature_namespaces, 0)
    parser.setContentHandler(handler)

    try:
        # Use cStringIo to avoid a crash in sax when filename has unicode
        # characters.
        s = file(filename).read()
        f = cStringIO.StringIO(s)
        parser.parse(f)
    except Exception, e:
        raise XMLError(stack_trace=True)

    # Calculate offset with current start date.
    
    cur_start_date =        time_of_start.time
    imp_start_date = import_time_of_start.time
    
    offset = long(round((cur_start_date - imp_start_date) / 60. / 60. / 24.))
        
    # Adjust timings.

    if reset_learning_data == False:
        if cur_start_date <= imp_start_date :
            for card in handler.imported_cards:
                card.last_rep += abs(offset)
                card.next_rep += abs(offset)
        else:
            time_of_start = StartTime(imp_start_date)
            for card in cards:
                card.last_rep += abs(offset)
                card.next_rep += abs(offset)

    return handler.imported_cards



register_file_format(_("Supermemo for Palm through smconv.pl"),
                     filter=_("XML files (*.xml *.XML)"),
                     import_function=import_XML,
                     export_function=None)
