/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_phonon_python.h"

#include "phonon_platformplugin_wrapper.h"

// Extra includes
#include <QList>
#include <QPair>
#include <abstractmediastream.h>
#include <objectdescription.h>
#include <pysideconversions.h>
#include <qbytearray.h>
#include <qicon.h>
#include <qobject.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qurl.h>
#include <qvariant.h>

using namespace Shiboken;

// Native ---------------------------------------------------------

PlatformPluginWrapper::PlatformPluginWrapper() : Phonon::PlatformPlugin() {
    // ... middle
}

QString PlatformPluginWrapper::applicationName() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "applicationName"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.applicationName()' not implemented.");
        return QString();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QString();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QString >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "PlatformPlugin.applicationName", SbkType<QString >()->tp_name, py_result->ob_type->tp_name);
        return QString();
    }
    QString cpp_result(Shiboken::Converter<QString >::toCpp(py_result));
    return cpp_result;
}

QObject * PlatformPluginWrapper::createBackend()
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "createBackend"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.createBackend()' not implemented.");
        return 0;
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return 0;
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QObject * >::isConvertible(py_result);
    typeIsValid = typeIsValid || (py_result == Py_None);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "PlatformPlugin.createBackend", SbkType<QObject * >()->tp_name, py_result->ob_type->tp_name);
        return 0;
    }
    QObject * cpp_result(Shiboken::Converter<QObject * >::toCpp(py_result));
    return cpp_result;
}

QObject * PlatformPluginWrapper::createBackend(const QString & library, const QString & version)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "createBackend"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.createBackend()' not implemented.");
        return 0;
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(SS)",
        Shiboken::Converter<QString & >::toPython(library),
        Shiboken::Converter<QString & >::toPython(version)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return 0;
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QObject * >::isConvertible(py_result);
    typeIsValid = typeIsValid || (py_result == Py_None);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "PlatformPlugin.createBackend", SbkType<QObject * >()->tp_name, py_result->ob_type->tp_name);
        return 0;
    }
    QObject * cpp_result(Shiboken::Converter<QObject * >::toCpp(py_result));
    return cpp_result;
}

Phonon::AbstractMediaStream * PlatformPluginWrapper::createMediaStream(const QUrl & url, QObject * parent)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "createMediaStream"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.createMediaStream()' not implemented.");
        return 0;
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(SS)",
        Shiboken::Converter<QUrl & >::toPython(url),
        Shiboken::Converter<QObject * >::toPython(parent)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return 0;
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<Phonon::AbstractMediaStream * >::isConvertible(py_result);
    typeIsValid = typeIsValid || (py_result == Py_None);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "PlatformPlugin.createMediaStream", SbkType<Phonon::AbstractMediaStream * >()->tp_name, py_result->ob_type->tp_name);
        return 0;
    }
    Phonon::AbstractMediaStream * cpp_result(Shiboken::Converter<Phonon::AbstractMediaStream * >::toCpp(py_result));
    return cpp_result;
}

QList<QPair<QByteArray, QString > > PlatformPluginWrapper::deviceAccessListFor(const Phonon::AudioOutputDevice & arg__1) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "deviceAccessListFor"));
    if (py_override.isNull()) {
        return this->Phonon::PlatformPlugin::deviceAccessListFor(arg__1);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<Phonon::AudioOutputDevice & >::toPython(arg__1)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QList<QPair<QByteArray, QString > >();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QList<QPair<QByteArray, QString > > >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "PlatformPlugin.deviceAccessListFor", "list", py_result->ob_type->tp_name);
        return QList<QPair<QByteArray, QString > >();
    }
    QList<QPair<QByteArray, QString > > cpp_result(Shiboken::Converter<QList<QPair<QByteArray, QString > > >::toCpp(py_result));
    return cpp_result;
}

QIcon PlatformPluginWrapper::icon(const QString & name) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "icon"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.icon()' not implemented.");
        return QIcon();
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QString & >::toPython(name)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QIcon();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QIcon >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "PlatformPlugin.icon", SbkType<QIcon >()->tp_name, py_result->ob_type->tp_name);
        return QIcon();
    }
    QIcon cpp_result(Shiboken::Converter<QIcon >::toCpp(py_result));
    return cpp_result;
}

bool PlatformPluginWrapper::isMimeTypeAvailable(const QString & mimeType) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "isMimeTypeAvailable"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.isMimeTypeAvailable()' not implemented.");
        return bool(0);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QString & >::toPython(mimeType)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "PlatformPlugin.isMimeTypeAvailable", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

qreal PlatformPluginWrapper::loadVolume(const QString & outputName) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "loadVolume"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.loadVolume()' not implemented.");
        return qreal(0);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QString & >::toPython(outputName)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return qreal(0);
    }
    // Check return type
    bool typeIsValid = PyFloat_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "PlatformPlugin.loadVolume", SbkType<qreal >()->tp_name, py_result->ob_type->tp_name);
        return qreal(0);
    }
    qreal cpp_result(Shiboken::Converter<double >::toCpp(py_result));
    return cpp_result;
}

void PlatformPluginWrapper::notification(const char * notificationName, const QString & text, const QStringList & actions, QObject * receiver, const char * actionSlot) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "notification"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.notification()' not implemented.");
        return ;
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(zSSSz)",
        notificationName,
        Shiboken::Converter<QString & >::toPython(text),
        Shiboken::Converter<QStringList & >::toPython(actions),
        Shiboken::Converter<QObject * >::toPython(receiver),
        actionSlot
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

QList<int > PlatformPluginWrapper::objectDescriptionIndexes(Phonon::ObjectDescriptionType type) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "objectDescriptionIndexes"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.objectDescriptionIndexes()' not implemented.");
        return QList<int >();
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<Phonon::ObjectDescriptionType >::toPython(type)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QList<int >();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QList<int > >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "PlatformPlugin.objectDescriptionIndexes", "list", py_result->ob_type->tp_name);
        return QList<int >();
    }
    QList<int > cpp_result(Shiboken::Converter<QList<int > >::toCpp(py_result));
    return cpp_result;
}

QHash<QByteArray, QVariant > PlatformPluginWrapper::objectDescriptionProperties(Phonon::ObjectDescriptionType type, int index) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "objectDescriptionProperties"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.objectDescriptionProperties()' not implemented.");
        return QHash<QByteArray, QVariant >();
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(Si)",
        Shiboken::Converter<Phonon::ObjectDescriptionType >::toPython(type),
        index
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QHash<QByteArray, QVariant >();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QHash<QByteArray, QVariant > >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "PlatformPlugin.objectDescriptionProperties", "hash", py_result->ob_type->tp_name);
        return QHash<QByteArray, QVariant >();
    }
    QHash<QByteArray, QVariant > cpp_result(Shiboken::Converter<QHash<QByteArray, QVariant > >::toCpp(py_result));
    return cpp_result;
}

void PlatformPluginWrapper::saveVolume(const QString & outputName, qreal volume)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "saveVolume"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.saveVolume()' not implemented.");
        return ;
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(Sd)",
        Shiboken::Converter<QString & >::toPython(outputName),
        Shiboken::Converter<double >::toPython(volume)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

PlatformPluginWrapper::~PlatformPluginWrapper()
{
    BindingManager::instance().invalidateWrapper(this);
}

// Target ---------------------------------------------------------

static int
SbkPhonon_PlatformPlugin_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    PlatformPluginWrapper* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);
    SbkBaseWrapperType* type = reinterpret_cast<SbkBaseWrapperType*>(self->ob_type);
    SbkBaseWrapperType* myType = reinterpret_cast<SbkBaseWrapperType*>(SbkPySide_phononTypes[SBK_PHONON_PLATFORMPLUGIN_IDX]);
    if (type == myType) {
        PyErr_SetString(PyExc_NotImplementedError,
            "'Phonon::PlatformPlugin' represents a C++ abstract class and cannot be instantiated");
        return -1;
    }

    // PlatformPlugin()
    cptr = new PlatformPluginWrapper();
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !Shiboken::setCppPointer(sbkSelf, Shiboken::SbkType<Phonon::PlatformPlugin >(), cptr)) {
        delete cptr;
        return -1;
    }
    sbkSelf->validCppObject = 1;
    sbkSelf->containsCppWrapper = 1;
    BindingManager::instance().registerWrapper(sbkSelf, cptr);

    return 1;
}

static PyObject* SbkPhonon_PlatformPluginFunc_applicationName(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // applicationName()const
    if (SbkBaseWrapper_containsCppWrapper(self)) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.applicationName()' not implemented.");
        return 0;
    }
    Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
    QString cpp_result = cppSelf->applicationName();
    py_result = Shiboken::Converter<QString >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkPhonon_PlatformPluginFunc_createBackend(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    // invalid argument lengths
    if (numArgs == 1)
        goto SbkPhonon_PlatformPluginFunc_createBackend_TypeError;

    if (!PyArg_UnpackTuple(args, "createBackend", 0, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 0) {
        // createBackend()
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.createBackend()' not implemented.");
            return 0;
        }
        Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
        QObject * cpp_result = cppSelf->createBackend();
        py_result = Shiboken::Converter<QObject * >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(self, py_result);
    } else if (numArgs == 2 && Shiboken::Converter<QString & >::isConvertible(pyargs[0]) && Shiboken::Converter<QString & >::isConvertible(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QString > cpp_arg1_auto_ptr;
        QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
        if (!SbkQString_Check(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
        // createBackend(QString,QString)
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.createBackend()' not implemented.");
            return 0;
        }
        Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
        QObject * cpp_result = cppSelf->createBackend(*cpp_arg0, *cpp_arg1);
        py_result = Shiboken::Converter<QObject * >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(self, py_result);
    } else goto SbkPhonon_PlatformPluginFunc_createBackend_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkPhonon_PlatformPluginFunc_createBackend_TypeError:
        const char* overloads[] = {"", "PySide.QtCore.QString, PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.phonon.PlatformPlugin.createBackend", overloads);
        return 0;
}

static PyObject* SbkPhonon_PlatformPluginFunc_createMediaStream(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "createMediaStream", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QUrl & >::isConvertible(pyargs[0]) && Shiboken::Converter<QObject * >::isConvertible(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QUrl > cpp_arg0_auto_ptr;
        QUrl* cpp_arg0 = Shiboken::Converter<QUrl* >::toCpp(pyargs[0]);
        if (!SbkQUrl_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QUrl >(cpp_arg0);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        QObject* cpp_arg1 = Shiboken::Converter<QObject* >::toCpp(pyargs[1]);
        // createMediaStream(QUrl,QObject*)
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.createMediaStream()' not implemented.");
            return 0;
        }
        Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
        Phonon::AbstractMediaStream * cpp_result = cppSelf->createMediaStream(*cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<Phonon::AbstractMediaStream * >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(self, py_result);
    } else goto SbkPhonon_PlatformPluginFunc_createMediaStream_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkPhonon_PlatformPluginFunc_createMediaStream_TypeError:
        const char* overloads[] = {"PySide.QtCore.QUrl, PySide.QtCore.QObject", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.phonon.PlatformPlugin.createMediaStream", overloads);
        return 0;
}

static PyObject* SbkPhonon_PlatformPluginFunc_deviceAccessListFor(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<Phonon::AudioOutputDevice & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        Phonon::AudioOutputDevice* cpp_arg0 = Shiboken::Converter<Phonon::AudioOutputDevice* >::toCpp(arg);
        // deviceAccessListFor(Phonon::AudioOutputDevice)const
        Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
        QList<QPair<QByteArray, QString > > cpp_result = cppSelf->Phonon::PlatformPlugin::deviceAccessListFor(*cpp_arg0);
        py_result = Shiboken::Converter<QList<QPair<QByteArray, QString > > >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkPhonon_PlatformPluginFunc_deviceAccessListFor_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkPhonon_PlatformPluginFunc_deviceAccessListFor_TypeError:
        const char* overloads[] = {"PySide.phonon.Phonon::AudioOutputDevice", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.phonon.PlatformPlugin.deviceAccessListFor", overloads);
        return 0;
}

static PyObject* SbkPhonon_PlatformPluginFunc_icon(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QString & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!SbkQString_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // icon(QString)const
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.icon()' not implemented.");
            return 0;
        }
        Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
        QIcon cpp_result = cppSelf->icon(*cpp_arg0);
        py_result = Shiboken::Converter<QIcon >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkPhonon_PlatformPluginFunc_icon_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkPhonon_PlatformPluginFunc_icon_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.phonon.PlatformPlugin.icon", overloads);
        return 0;
}

static PyObject* SbkPhonon_PlatformPluginFunc_isMimeTypeAvailable(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QString & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!SbkQString_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // isMimeTypeAvailable(QString)const
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.isMimeTypeAvailable()' not implemented.");
            return 0;
        }
        Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->isMimeTypeAvailable(*cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkPhonon_PlatformPluginFunc_isMimeTypeAvailable_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkPhonon_PlatformPluginFunc_isMimeTypeAvailable_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.phonon.PlatformPlugin.isMimeTypeAvailable", overloads);
        return 0;
}

static PyObject* SbkPhonon_PlatformPluginFunc_loadVolume(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QString & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!SbkQString_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // loadVolume(QString)const
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.loadVolume()' not implemented.");
            return 0;
        }
        Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
        qreal cpp_result = cppSelf->loadVolume(*cpp_arg0);
        py_result = Shiboken::Converter<double >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkPhonon_PlatformPluginFunc_loadVolume_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkPhonon_PlatformPluginFunc_loadVolume_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.phonon.PlatformPlugin.loadVolume", overloads);
        return 0;
}

static PyObject* SbkPhonon_PlatformPluginFunc_notification(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0, 0};

    if (!PyArg_UnpackTuple(args, "notification", 2, 5, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3]), &(pyargs[4])))
        return 0;

    if (Shiboken::Converter<const char * >::isConvertible(pyargs[0]) && Shiboken::Converter<QString & >::isConvertible(pyargs[1])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QString > cpp_arg1_auto_ptr;
        QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
        if (!SbkQString_Check(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
        if (numArgs == 2) {
            // notification(const char*,QString,QStringList,QObject*,const char*)const
            if (SbkBaseWrapper_containsCppWrapper(self)) {
                PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.notification()' not implemented.");
                return 0;
            }
            Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
            cppSelf->notification(cpp_arg0, *cpp_arg1);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QStringList & >::isConvertible(pyargs[2])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[2]))
                return 0;
            std::auto_ptr<QStringList > cpp_arg2_auto_ptr;
            QStringList* cpp_arg2 = Shiboken::Converter<QStringList* >::toCpp(pyargs[2]);
            if (!SbkQStringList_Check(pyargs[2]))
                cpp_arg2_auto_ptr = std::auto_ptr<QStringList >(cpp_arg2);
            if (numArgs == 3) {
                // notification(const char*,QString,QStringList,QObject*,const char*)const
                if (SbkBaseWrapper_containsCppWrapper(self)) {
                    PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.notification()' not implemented.");
                    return 0;
                }
                Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
                cppSelf->notification(cpp_arg0, *cpp_arg1, *cpp_arg2);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<QObject * >::isConvertible(pyargs[3])) {
                if (Shiboken::cppObjectIsInvalid(pyargs[3]))
                    return 0;
                QObject* cpp_arg3 = Shiboken::Converter<QObject* >::toCpp(pyargs[3]);
                if (numArgs == 4) {
                    // notification(const char*,QString,QStringList,QObject*,const char*)const
                    if (SbkBaseWrapper_containsCppWrapper(self)) {
                        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.notification()' not implemented.");
                        return 0;
                    }
                    Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
                    cppSelf->notification(cpp_arg0, *cpp_arg1, *cpp_arg2, cpp_arg3);
                    //CppGenerator::writeParentChildManagement
                } else if (Shiboken::Converter<const char * >::isConvertible(pyargs[4])) {
                    const char * cpp_arg4 = Shiboken::Converter<const char * >::toCpp(pyargs[4]);
                    // notification(const char*,QString,QStringList,QObject*,const char*)const
                    if (SbkBaseWrapper_containsCppWrapper(self)) {
                        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.notification()' not implemented.");
                        return 0;
                    }
                    Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
                    cppSelf->notification(cpp_arg0, *cpp_arg1, *cpp_arg2, cpp_arg3, cpp_arg4);
                    //CppGenerator::writeParentChildManagement
                } else goto SbkPhonon_PlatformPluginFunc_notification_TypeError;
            } else goto SbkPhonon_PlatformPluginFunc_notification_TypeError;
        } else goto SbkPhonon_PlatformPluginFunc_notification_TypeError;
    } else goto SbkPhonon_PlatformPluginFunc_notification_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkPhonon_PlatformPluginFunc_notification_TypeError:
        const char* overloads[] = {"str, PySide.QtCore.QString, PySide.QtCore.QStringList = QStringList(), PySide.QtCore.QObject = None, str = None", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.phonon.PlatformPlugin.notification", overloads);
        return 0;
}

static PyObject* SbkPhonon_PlatformPluginFunc_objectDescriptionIndexes(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<Phonon::ObjectDescriptionType >::isConvertible(arg)) {
        Phonon::ObjectDescriptionType cpp_arg0 = Shiboken::Converter<Phonon::ObjectDescriptionType >::toCpp(arg);
        // objectDescriptionIndexes(Phonon::ObjectDescriptionType)const
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.objectDescriptionIndexes()' not implemented.");
            return 0;
        }
        Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
        QList<int > cpp_result = cppSelf->objectDescriptionIndexes(cpp_arg0);
        py_result = Shiboken::Converter<QList<int > >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkPhonon_PlatformPluginFunc_objectDescriptionIndexes_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkPhonon_PlatformPluginFunc_objectDescriptionIndexes_TypeError:
        const char* overloads[] = {"PySide.phonon.Phonon.ObjectDescriptionType", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.phonon.PlatformPlugin.objectDescriptionIndexes", overloads);
        return 0;
}

static PyObject* SbkPhonon_PlatformPluginFunc_objectDescriptionProperties(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "objectDescriptionProperties", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<Phonon::ObjectDescriptionType >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        Phonon::ObjectDescriptionType cpp_arg0 = Shiboken::Converter<Phonon::ObjectDescriptionType >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        // objectDescriptionProperties(Phonon::ObjectDescriptionType,int)const
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.objectDescriptionProperties()' not implemented.");
            return 0;
        }
        Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
        QHash<QByteArray, QVariant > cpp_result = cppSelf->objectDescriptionProperties(cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<QHash<QByteArray, QVariant > >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkPhonon_PlatformPluginFunc_objectDescriptionProperties_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkPhonon_PlatformPluginFunc_objectDescriptionProperties_TypeError:
        const char* overloads[] = {"PySide.phonon.Phonon.ObjectDescriptionType, int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.phonon.PlatformPlugin.objectDescriptionProperties", overloads);
        return 0;
}

static PyObject* SbkPhonon_PlatformPluginFunc_saveVolume(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "saveVolume", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QString & >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // saveVolume(QString,qreal)
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'PlatformPlugin.saveVolume()' not implemented.");
            return 0;
        }
        Phonon::PlatformPlugin* cppSelf = Shiboken::Converter<Phonon::PlatformPlugin* >::toCpp((PyObject*)self);
        cppSelf->saveVolume(*cpp_arg0, cpp_arg1);
        //CppGenerator::writeParentChildManagement
    } else goto SbkPhonon_PlatformPluginFunc_saveVolume_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkPhonon_PlatformPluginFunc_saveVolume_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.phonon.PlatformPlugin.saveVolume", overloads);
        return 0;
}

static PyMethodDef SbkPhonon_PlatformPlugin_methods[] = {
    {"applicationName", (PyCFunction)SbkPhonon_PlatformPluginFunc_applicationName, METH_NOARGS},
    {"createBackend", (PyCFunction)SbkPhonon_PlatformPluginFunc_createBackend, METH_VARARGS},
    {"createMediaStream", (PyCFunction)SbkPhonon_PlatformPluginFunc_createMediaStream, METH_VARARGS},
    {"deviceAccessListFor", (PyCFunction)SbkPhonon_PlatformPluginFunc_deviceAccessListFor, METH_O},
    {"icon", (PyCFunction)SbkPhonon_PlatformPluginFunc_icon, METH_O},
    {"isMimeTypeAvailable", (PyCFunction)SbkPhonon_PlatformPluginFunc_isMimeTypeAvailable, METH_O},
    {"loadVolume", (PyCFunction)SbkPhonon_PlatformPluginFunc_loadVolume, METH_O},
    {"notification", (PyCFunction)SbkPhonon_PlatformPluginFunc_notification, METH_VARARGS},
    {"objectDescriptionIndexes", (PyCFunction)SbkPhonon_PlatformPluginFunc_objectDescriptionIndexes, METH_O},
    {"objectDescriptionProperties", (PyCFunction)SbkPhonon_PlatformPluginFunc_objectDescriptionProperties, METH_VARARGS},
    {"saveVolume", (PyCFunction)SbkPhonon_PlatformPluginFunc_saveVolume, METH_VARARGS},
    {0} // Sentinel
};

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkPhonon_PlatformPlugin_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.phonon.PlatformPlugin",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::deallocWrapper,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkPhonon_PlatformPlugin_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             reinterpret_cast<PyTypeObject*>(&Shiboken::SbkBaseWrapper_Type),
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkPhonon_PlatformPlugin_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0,
    /*cpp_dtor*/            &Shiboken::callCppDestructor<Phonon::PlatformPlugin >,
    /*is_multicpp*/         0,
    /*is_user_type*/        0
};

static SbkBaseWrapperType* SbkPhonon_PlatformPlugin_typeDiscovery(void* cptr, SbkBaseWrapperType* instanceType)
{
    if (instanceType->mi_specialcast)
        cptr = instanceType->mi_specialcast(cptr, &SbkPhonon_PlatformPlugin_Type);
    TypeResolver* typeResolver = TypeResolver::get(typeid(*reinterpret_cast<Phonon::PlatformPlugin*>(cptr)).name());
    if (typeResolver)
        return reinterpret_cast<SbkBaseWrapperType*>(typeResolver->pythonType());
    return 0;
}


PyAPI_FUNC(void) init_Phonon_PlatformPlugin(PyObject* module)
{
    SbkPySide_phononTypes[SBK_PHONON_PLATFORMPLUGIN_IDX] = reinterpret_cast<PyTypeObject*>(&SbkPhonon_PlatformPlugin_Type);

    // Fill type discovery information
    SbkPhonon_PlatformPlugin_Type.type_discovery = new Shiboken::TypeDiscovery;
    SbkPhonon_PlatformPlugin_Type.type_discovery->addTypeDiscoveryFunction(&SbkPhonon_PlatformPlugin_typeDiscovery);

    if (PyType_Ready((PyTypeObject*)&SbkPhonon_PlatformPlugin_Type) < 0)
        return;

    PyDict_SetItemString(module,"PlatformPlugin", (PyObject*)&SbkPhonon_PlatformPlugin_Type);
    Shiboken::TypeResolver::createObjectTypeResolver<Phonon::PlatformPlugin >("Phonon::PlatformPlugin*");
    Shiboken::TypeResolver::createObjectTypeResolver<Phonon::PlatformPlugin >(typeid(Phonon::PlatformPlugin).name());
}


} // extern "C"

